import classnames from 'classnames';
import { __ } from '@wordpress/i18n';
import { useState } from '@wordpress/element';
import ServerSideRender from '@wordpress/server-side-render';
import { PanelBody, RangeControl, SelectControl, Disabled } from '@wordpress/components';
import { BotigaDimensions, BotigaTypography, BotigaColorPicker, BotigaTabsNavigation, BotigaTabsContent, BotigaBorder } from '../_inc/global-components';
import { InspectorControls, useBlockProps } from '@wordpress/block-editor';
import './editor.scss';
import { generateBlockCustomCss } from '../_inc/helpers';

/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {WPElement} Element to render.
 */
export default function Edit( props ) {
	const { name, attributes, setAttributes } = props;

	const { 
		customCss
	} = attributes;

	// Block ID
	const blockId = `botiga-tb-block-${ props.clientId.substr( 0, 8 ) }`;
	setAttributes( { blockId: blockId } );

	// Block Props
	const blockProps = useBlockProps( {
		className: classnames( blockId, 'botiga-tb-block', 'botiga-tb-sp-next-prev' )
	} );

	// Tab Navigation
	const [ tab, setTab ] = useState( 'style' );

	// Defaults
	const defaults = {
		align: 'left',
		buttonsSize: 73,
	}

	if( typeof customCss.align !== 'undefined' ) {
        defaults.align = customCss.align.value;
    }

	if( typeof customCss.buttonsSize !== 'undefined' ) {
        defaults.buttonsSize = customCss.buttonsSize.value;
    }

	// Custom CSS
	generateBlockCustomCss( blockId, customCss );

	// Separator Style
	const separatorStyle = {
		fontSize: 14,
		margin: '15px 0 15px',
		borderBottom: '1px solid #e5e5e5',
		padding: '13px 0px'
	}

	return (
		<div { ...blockProps }>
		
			<InspectorControls>
				<BotigaTabsNavigation
					exclude="settings"
					activeTab={ tab }
					setActiveTab={ setTab } />

				<BotigaTabsContent 
					tabid="style"
					activeTab={ tab }>
					
					<PanelBody title={ __( 'Buttons', 'botiga-pro' ) } initialOpen={ true }>
						<SelectControl
							label={ __( 'Alignment', 'botiga-pro' ) }
							value={ defaults.align }
							options={ [
								{ label: 'Left', value: 'flex-start' },
								{ label: 'Center', value: 'center' },
								{ label: 'Right', value: 'flex-end' },
							] }
							onChange={ ( value ) => { 
								const css = {
									align: {
										selector: `{{WRAPPER}} .botiga-product-navigation`,
										props: `justify-content: ${ value };`,
										value: value
									}
								};
								setAttributes( { customCss: { ...customCss, ...css } } ) 
							} }
						/>
						<RangeControl
							label={ __( 'Size', 'botiga-pro' ) }
							idPrefix={ `buttons_typography` }
							labelPosition="edge"
							value={ defaults.buttonsSize }
							onChange={ ( value ) => { 
								const css = {
									buttonsSize: {
										selector: '{{WRAPPER}} .botiga-product-navigation a',
										props: `font-size: ${ value }px !important;`,
										value: value
									}
								};
								setAttributes( { customCss: { ...customCss, ...css } } ) 
							} }
							min={ 1 }
							max={ 150 }
						/>
						<BotigaColorPicker 
							label={ __( 'Background Color', 'botiga-pro' ) }
							idPrefix={ 'buttons_background_color' }
							withHover={ true }
							attributes={ attributes }
							setAttributes={ setAttributes }
							cssProp={ `background-color` }
							customCssSelector={ `{{WRAPPER}} .botiga-product-navigation a` }
						/>
						<BotigaColorPicker 
							label={ __( 'Icon Color', 'botiga-pro' ) }
							idPrefix={ 'buttons_icon_color' }
							withHover={ true }
							attributes={ attributes }
							setAttributes={ setAttributes }
							cssProp={ `border-color` }
							customCssSelector={ `{{WRAPPER}} .botiga-product-navigation a i:before` }
						/>
						<BotigaColorPicker 
							label={ __( 'Border Color', 'botiga-pro' ) }
							idPrefix={ 'buttons_link_color' }
							withHover={ true }
							attributes={ attributes }
							setAttributes={ setAttributes }
							cssProp={ `border-color` }
							customCssSelector={ `{{WRAPPER}} .botiga-product-navigation a` }
						/>
						<BotigaDimensions
							label={ __( 'Border Radius', 'botiga-pro' ) }
							idPrefix={ 'buttons_border_radius' }
							type={ 'border-radius' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .botiga-product-navigation a` } 
						/>
					</PanelBody>
					<PanelBody title={ __( 'Popover Content', 'botiga-pro' ) } initialOpen={ false }>
						<BotigaColorPicker 
							label={ __( 'Border Color', 'botiga-pro' ) }
							idPrefix={ 'buttons_popover_border_color' }
							withHover={ false }
							attributes={ attributes }
							setAttributes={ setAttributes }
							cssProp={ `border-color` }
							customCssSelector={ `{{WRAPPER}} .botiga-product-navigation a span` }
						/>
						<BotigaColorPicker 
							label={ __( 'Arrow Color', 'botiga-pro' ) }
							idPrefix={ 'buttons_popover_arrow_border_color' }
							withHover={ false }
							attributes={ attributes }
							setAttributes={ setAttributes }
							cssProp={ `border-bottom-color` }
							customCssSelector={ `{{WRAPPER}} .botiga-product-navigation a span:after` }
						/>
						<BotigaColorPicker 
							label={ __( 'Background Color', 'botiga-pro' ) }
							idPrefix={ 'buttons_popover_background_color' }
							withHover={ false }
							attributes={ attributes }
							setAttributes={ setAttributes }
							cssProp={ `background-color` }
							customCssSelector={ `{{WRAPPER}} .botiga-product-navigation a span` }
						/>

						<h6 style={ separatorStyle }>{ __( 'Product Title', 'botiga-pro' ) }</h6>
						<BotigaTypography 
							label={ __( 'Typography', 'botiga-pro' ) }
							idPrefix={ 'buttons_popover_title_typography' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .botiga-product-navigation h2` }
						/>
						<BotigaColorPicker 
							label={ __( 'Text Color', 'botiga-pro' ) }
							idPrefix={ 'buttons_popover_title_color' }
							withHover={ false }
							attributes={ attributes }
							setAttributes={ setAttributes }
							cssProp={ `color` }
							customCssSelector={ `{{WRAPPER}} .botiga-product-navigation h2` }
						/>
					</PanelBody>

				</BotigaTabsContent>
				<BotigaTabsContent 
					tabid="advanced"
					activeTab={ tab }>
					
					<PanelBody title={ __( 'Layout', 'botiga-pro' ) } initialOpen={ true }>
						<BotigaDimensions
							label={ __( 'Margin', 'botiga-pro' ) }
							idPrefix={ '' }
							type={ 'margin' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}}` } />

						<BotigaDimensions
							label={ __( 'Padding', 'botiga-pro' ) }
							idPrefix={ '' }
							type={ 'padding' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}}` } />
					</PanelBody>

				</BotigaTabsContent>
			</InspectorControls>

			<Disabled>
				<ServerSideRender
					block={ name }
					httpMethod="POST"
					attributes={ attributes }
				/>
			</Disabled>

		</div>
	);
}
