import classnames from 'classnames';
import { __ } from '@wordpress/i18n';
import { useState } from '@wordpress/element';
import ServerSideRender from '@wordpress/server-side-render';
import { PanelBody, Disabled } from '@wordpress/components';
import { BotigaDimensions, BotigaTypography, BotigaColorPicker, BotigaTabsNavigation, BotigaTabsContent, BotigaBorder } from '../_inc/global-components';
import { InspectorControls, useBlockProps } from '@wordpress/block-editor';
import { generateBlockCustomCss } from '../_inc/helpers';

/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {WPElement} Element to render.
 */
export default function Edit( props ) {
	const { name, attributes, setAttributes } = props;

	const { 
		customCss
	} = attributes;

	// Block ID
	const blockId = `botiga-tb-block-${ props.clientId.substr( 0, 8 ) }`;
	setAttributes( { blockId: blockId } );

	// Block Props
	const blockProps = useBlockProps( {
		className: classnames( blockId, 'botiga-tb-block', 'botiga-tb-sp-image' )
	} );

	// Tab Navigation
	const [ tab, setTab ] = useState( 'style' );

	// Custom CSS
	generateBlockCustomCss( blockId, customCss );

	return (
		<div { ...blockProps }>
		
			<InspectorControls>
				<BotigaTabsNavigation
					exclude="settings"
					activeTab={ tab }
					setActiveTab={ setTab } />

				<BotigaTabsContent 
					tabid="style"
					activeTab={ tab }>
					
					<PanelBody title={ __( 'Product Image', 'botiga-pro' ) } initialOpen={ true }>
						<BotigaBorder 
							label={ __( 'Border', 'botiga-pro' ) }
							idPrefix={ 'product_image_border' }
							defaults={{
								borderType: 'none',
								borderWidth: {
									top: 1,
									left: 1,
									right: 1,
									bottom: 1
								},
								borderColor: '#CCC'
							}}
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .flex-viewport` }
						/>
					</PanelBody>
					<PanelBody title={ __( 'Product Image Thumbnails', 'botiga-pro' ) } initialOpen={ false }>
						<BotigaBorder 
							label={ __( 'Border', 'botiga-pro' ) }
							idPrefix={ 'product_image_thumbs_border' }
							defaults={{
								borderType: 'none',
								borderWidth: {
									top: 1,
									left: 1,
									right: 1,
									bottom: 1
								},
								borderColor: '#CCC'
							}}
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .flex-control-thumbs li img, {{WRAPPER}} .flex-control-thumbs > div img` }
						/>
						<BotigaBorder 
							label={ __( 'Border (hover/active)', 'botiga-pro' ) }
							idPrefix={ 'product_image_thumbs_border_hover' }
							defaults={{
								borderType: 'none',
								borderWidth: {
									top: 1,
									left: 1,
									right: 1,
									bottom: 1
								},
								borderColor: '#CCC'
							}}
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .flex-control-thumbs li img.flex-active, {{WRAPPER}} .flex-control-thumbs li img:hover, .flex-control-thumbs > div:nth-child(1) img` }
						/>
					</PanelBody>
					<PanelBody title={ __( 'Sale Badge', 'botiga-pro' ) } initialOpen={ false }>
						<BotigaTypography 
							label={ __( 'Typography', 'botiga-pro' ) }
							idPrefix={ 'sale_badge_typography' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .onsale` }
						/>
						<BotigaColorPicker 
							label={ __( 'Text Color', 'botiga-pro' ) }
							idPrefix={ 'sale_badge_color' }
							attributes={ attributes }
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .onsale` }
						/>
						<BotigaColorPicker 
							label={ __( 'Background Color', 'botiga-pro' ) }
							idPrefix={ 'sale_badge_background_color' }
							attributes={ attributes }
							setAttributes={ setAttributes }
							cssProp={ `background-color` }
							customCssSelector={ `{{WRAPPER}} .onsale` }
						/>
					</PanelBody>
				</BotigaTabsContent>
				<BotigaTabsContent 
					tabid="advanced"
					activeTab={ tab }>
					
					<PanelBody title={ __( 'Layout', 'botiga-pro' ) } initialOpen={ true }>
						<BotigaDimensions
							label={ __( 'Margin', 'botiga-pro' ) }
							idPrefix={ '' }
							type={ 'margin' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}}` } />

						<BotigaDimensions
							label={ __( 'Padding', 'botiga-pro' ) }
							idPrefix={ '' }
							type={ 'padding' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}}` } />
					</PanelBody>

				</BotigaTabsContent>
			</InspectorControls>

			<Disabled>
				<ServerSideRender
					block={ name }
					httpMethod="POST"
					attributes={ attributes }
				/>
			</Disabled>

		</div>
	);
}
