<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor icon list widget.
 *
 * Elementor widget that displays a bullet list with any chosen icons and texts.
 *
 * @since 1.0.0
 */
class Botiga_Elementor_Shop_Products extends Widget_Base {

	/**
	 * Constructor
	 */
	public function __construct( $data=[], $args=null ){
		parent::__construct( $data, $args );
	}

	/**
	 * Load styles dependencies
	 */
	public function get_style_depends(){
		return [ 'botiga-builder-widgets-styles' ];	
	}

	public function get_help_url() {
		return 'https://docs.athemes.com/article/pro-templates-builder-overview/';
	}
		
	/**
	 * Get widget name.
	 *
	 * Retrieve icon list widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'botiga-elementor-shop-products';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve icon list widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Product Query', 'botiga-pro' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve icon list widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-products';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the icon list widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'botiga-woocommerce' ];
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'archive', 'products', 'product', 'shop', 'botiga' ];
	}	

	/**
	 * Register icon list widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {

		// Content
		$this->start_controls_section(
			'section_content_content',
			[
				'label' => __( 'Content', 'botiga-pro' ),
				'tab' 	=> Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'queryType',
			[
				'label' 	=> __( 'Query Type', 'botiga-pro' ),
				'type' 		=> Controls_Manager::SELECT,
				'default' 	=> 'all-products',
				'options' 	=> [
					'all-products' 	    => __( 'All Products', 'botiga-pro' ),
					'related-products' => __( 'Related Products', 'botiga-pro' ),
					'upsell-products' => __( 'Upsell Products', 'botiga-pro' ),
				],
			]
		);

		$this->add_control(
			'queryOrderBy',
			[
				'label' 		=> __( 'Order by', 'botiga-pro' ),
				'type' 			=> Controls_Manager::SELECT,
				'default' 		=> 'date',
				'options' 		=> [
					'date' 			=> __( 'Date', 'botiga-pro' ),
					'price' 		=> __( 'Price', 'botiga-pro' ),
					'popularity' 	=> __( 'Popularity', 'botiga-pro' ),
					'rating' 		=> __( 'Rating', 'botiga-pro' ),
					'rand' 			=> __( 'Random', 'botiga-pro' ),
					'menu_order' 	=> __( 'Menu order', 'botiga-pro' ),
				],
				'description' 	=> __( 'Select how to sort retrieved products.', 'botiga-pro' ),
			]
		);

		$this->add_control(
			'queryOrder',
			[
				'label' 		=> __( 'Order', 'botiga-pro' ),
				'type' 			=> Controls_Manager::SELECT,
				'default' 		=> 'desc',
				'options' 		=> [
					'asc' 	=> __( 'ASC', 'botiga-pro' ),
					'desc' 	=> __( 'DESC', 'botiga-pro' ),
				],
				'description' 	=> __( 'Designates the ascending or descending order.', 'botiga-pro' ),
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_content_layout',
			[
				'label' => __( 'Layout', 'botiga-pro' ),
				'tab' 	=> Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'columns',
			[
				'label' 		=> __( 'Columns', 'botiga-pro' ),
				'type' 			=> Controls_Manager::SELECT,
				'default' 		=> '3',
				'options' 		=> [
					'1' => __( '1', 'botiga-pro' ),
					'2' => __( '2', 'botiga-pro' ),
					'3' => __( '3', 'botiga-pro' ),
					'4' => __( '4', 'botiga-pro' ),
					'5' => __( '5', 'botiga-pro' ),
					'6' => __( '6', 'botiga-pro' ),
				],
			]
		);

		$this->add_control(
			'columnsGap',
			[
				'label' => esc_html__( 'Columns Gap', 'botiga-pro' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px' ],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 50,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 15,
				],
				'selectors' => [
					'{{WRAPPER}} ul.products' => 'gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'rows',
			[
				'label' 		=> __( 'Rows', 'botiga-pro' ),
				'type' 			=> Controls_Manager::SELECT,
				'default' 		=> '3',
				'options' 		=> [
					'1' => __( '1', 'botiga-pro' ),
					'2' => __( '2', 'botiga-pro' ),
					'3' => __( '3', 'botiga-pro' ),
					'4' => __( '4', 'botiga-pro' ),
					'5' => __( '5', 'botiga-pro' ),
					'6' => __( '6', 'botiga-pro' ),
				],
				'description' 	=> __( 'Number of rows to show.', 'botiga-pro' ),
			]
		);

		$this->add_control(
			'displayResultCount',
			[
				'label' => esc_html__( 'Result Count', 'botiga-pro' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'botiga-pro' ),
				'label_off' => esc_html__( 'Hide', 'botiga-pro' ),
				'return_value' => 'yes',
				'default' => 'no',
			]
		);

		$this->add_control(
			'displayOrdering',
			[
				'label' => esc_html__( 'Ordering', 'botiga-pro' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'botiga-pro' ),
				'label_off' => esc_html__( 'Hide', 'botiga-pro' ),
				'return_value' => 'yes',
				'default' => 'no',
			]
		);

		$this->add_control(
			'displayGridListToggle',
			[
				'label' => esc_html__( 'Grid/List Toggle', 'botiga-pro' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'botiga-pro' ),
				'label_off' => esc_html__( 'Hide', 'botiga-pro' ),
				'return_value' => 'yes',
				'default' => 'no',
			]
		);

		$this->add_control(
			'displayPagination',
			[
				'label' => esc_html__( 'Pagination', 'botiga-pro' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'botiga-pro' ),
				'label_off' => esc_html__( 'Hide', 'botiga-pro' ),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_layout',
			[
				'label' => __( 'Layout', 'botiga-pro' ),
				'tab' 	=> Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'result_count_heading',
			[
				'label' => esc_html__( 'Result Count', 'botiga-pro' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'show_result_count' => 'yes'
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' 		=> 'result_count_typography',
				'selector' 	=> '{{WRAPPER}} .woocommerce-sorting-wrapper .woocommerce-result-count',
				'scheme' 	=> Core\Schemes\Typography::TYPOGRAPHY_3,
				'condition' => [
					'show_result_count' => 'yes'
				],
			]
		);

		$this->add_control(
			'result_count_color',
			[
				'label' => __( 'Text Color', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .woocommerce-sorting-wrapper .woocommerce-result-count' => 'color: {{VALUE}};',
				],
				'condition' => [
					'show_result_count' => 'yes'
				],
			]
		);

		$this->add_control(
			'ordering_heading',
			[
				'label' => esc_html__( 'Ordering', 'botiga-pro' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'show_ordering' => 'yes'
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' 		=> 'ordering_typography',
				'selector' 	=> '{{WRAPPER}} .woocommerce-sorting-wrapper .woocommerce-ordering select',
				'scheme' 	=> Core\Schemes\Typography::TYPOGRAPHY_3,
				'condition' => [
					'show_ordering' => 'yes'
				],
			]
		);

		$this->add_control(
			'ordering_background_color',
			[
				'label' => __( 'Background Color', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .woocommerce-sorting-wrapper .woocommerce-ordering select' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'show_ordering' => 'yes'
				],
			]
		);

		$this->add_control(
			'ordering_color',
			[
				'label' => __( 'Text Color', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .woocommerce-sorting-wrapper .woocommerce-ordering select' => 'color: {{VALUE}};',
				],
				'condition' => [
					'show_ordering' => 'yes'
				],
			]
		);

		$this->add_control(
			'grid_list_toggle_heading',
			[
				'label' => esc_html__( 'Grid/List Toggle', 'botiga-pro' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'show_grid_list_toggle' => 'yes'
				],
			]
		);

		$this->add_control(
			'grid_list_toggle_color',
			[
				'label' => __( 'Icon Color', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .woocommerce-sorting-wrapper .botiga-grid-list-view svg path' => 'fill: {{VALUE}};',
				],
				'condition' => [
					'show_grid_list_toggle' => 'yes'
				],
			]
		);

		$this->add_control(
			'pagination_heading',
			[
				'label' => esc_html__( 'Pagination', 'botiga-pro' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'displayPagination' => 'yes'
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' 		=> 'pagination_typography',
				'selector' 	=> '{{WRAPPER}} .woocommerce-pagination .page-numbers',
				'scheme' 	=> Core\Schemes\Typography::TYPOGRAPHY_3,
				'condition' => [
					'displayPagination' => 'yes'
				],
			]
		);

		$this->add_control(
			'pagination_background_color',
			[
				'label' => __( 'Background Color', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .woocommerce-pagination .page-numbers:not(ul).current' => 'background-color: {{VALUE}};',
					'{{WRAPPER}} .woocommerce-pagination .page-numbers:not(ul):hover'   => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'displayPagination' => 'yes'
				],
			]
		);

		$this->add_control(
			'pagination_color',
			[
				'label' => __( 'Text Color', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .woocommerce-pagination .page-numbers:not(ul)' => 'color: {{VALUE}};',
				],
				'condition' => [
					'displayPagination' => 'yes'
				],
			]
		);

		$this->add_control(
			'pagination_color_hover',
			[
				'label' => __( 'Text Color (hover)', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .woocommerce-pagination .page-numbers:not(ul).current' => 'color: {{VALUE}};',
					'{{WRAPPER}} .woocommerce-pagination .page-numbers:not(ul):hover'   => 'color: {{VALUE}};',
				],
				'condition' => [
					'displayPagination' => 'yes'
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_content_product_card',
			[
				'label' => __( 'Product Card', 'botiga-pro' ),
				'tab' 	=> Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'displayImage',
			[
				'label' => esc_html__( 'Product Image', 'botiga-pro' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'botiga-pro' ),
				'label_off' => esc_html__( 'Hide', 'botiga-pro' ),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);

		$this->add_control(
			'displaySaleBadge',
			[
				'label' => esc_html__( 'Sale Badge', 'botiga-pro' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'botiga-pro' ),
				'label_off' => esc_html__( 'Hide', 'botiga-pro' ),
				'return_value' => 'yes',
				'default' => 'yes',
				'condition' => [
					'displayImage' => 'yes'
				],
			]
		);

		$this->add_control(
			'displayTitle',
			[
				'label' => esc_html__( 'Title', 'botiga-pro' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'botiga-pro' ),
				'label_off' => esc_html__( 'Hide', 'botiga-pro' ),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);

		$this->add_control(
			'displayPrice',
			[
				'label' => esc_html__( 'Price', 'botiga-pro' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'botiga-pro' ),
				'label_off' => esc_html__( 'Hide', 'botiga-pro' ),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);

		$this->add_control(
			'displayCategories',
			[
				'label' => esc_html__( 'Categories', 'botiga-pro' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'botiga-pro' ),
				'label_off' => esc_html__( 'Hide', 'botiga-pro' ),
				'return_value' => 'yes',
				'default' => 'no',
			]
		);

		$this->add_control(
			'displayRating',
			[
				'label' => esc_html__( 'Rating', 'botiga-pro' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'botiga-pro' ),
				'label_off' => esc_html__( 'Hide', 'botiga-pro' ),
				'return_value' => 'yes',
				'default' => 'no',
			]
		);

		$this->add_control(
			'displayShortDescription',
			[
				'label' => esc_html__( 'Short Description', 'botiga-pro' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'botiga-pro' ),
				'label_off' => esc_html__( 'Hide', 'botiga-pro' ),
				'return_value' => 'yes',
				'default' => 'no',
			]
		);

		$this->add_control(
			'displayAddToCart',
			[
				'label' => esc_html__( 'Add To Cart', 'botiga-pro' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'botiga-pro' ),
				'label_off' => esc_html__( 'Hide', 'botiga-pro' ),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);

		$this->add_control(
			'displayWishlist',
			[
				'label' => esc_html__( 'Wishlist', 'botiga-pro' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'botiga-pro' ),
				'label_off' => esc_html__( 'Hide', 'botiga-pro' ),
				'return_value' => 'yes',
				'default' => 'no',
				'condition' => [
					'displayImage' => 'yes'
				],
			]
		);

		$this->add_control(
			'displayQuickView',
			[
				'label' => esc_html__( 'Quick View', 'botiga-pro' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'botiga-pro' ),
				'label_off' => esc_html__( 'Hide', 'botiga-pro' ),
				'return_value' => 'yes',
				'default' => 'no',
				'condition' => [
					'displayImage' => 'yes'
				],
			]
		);

		$this->add_control(
			'displayStock',
			[
				'label' => esc_html__( 'Stock', 'botiga-pro' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'botiga-pro' ),
				'label_off' => esc_html__( 'Hide', 'botiga-pro' ),
				'return_value' => 'yes',
				'default' => 'no',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_product_card',
			[
				'label' => __( 'Product Card', 'botiga-pro' ),
				'tab' 	=> Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'card_heading',
			[
				'label' => esc_html__( 'Card', 'botiga-pro' ),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'card_bg_color',
			[
				'label' => __( 'Background Color', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} ul.products li.product' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name' => 'card_border',
				'selector' => '{{WRAPPER}} ul.products li.product',
			]
		);

		$this->add_control(
			'card_border_radius',
			[
				'label' => esc_html__( 'Border Radius', 'botiga-pro' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px' ],
				'selectors' => [
					'{{WRAPPER}} ul.products li.product' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					'{{WRAPPER}} ul.products li.product .loop-image-wrap img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'card_padding',
			[
				'label' => esc_html__( 'Padding', 'botiga-pro' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors' => [
					'{{WRAPPER}} ul.products li.product' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'alignLastBlock',
			[
				'label' => esc_html__( 'Align all products height', 'botiga-pro' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'botiga-pro' ),
				'label_off' => esc_html__( 'Hide', 'botiga-pro' ),
				'return_value' => 'yes',
				'default' => 'no',
			]
		);

		$this->add_control(
			'sale_badge_heading',
			[
				'label' => esc_html__( 'Sale Badge', 'botiga-pro' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'displayImage' => 'yes'
				],
			]
		);

		$this->add_control(
			'sale_badge_alignment',
			[
				'label' 		=> __( 'Alignment', 'botiga-pro'),
				'type' => Controls_Manager::CHOOSE,
				'default' => 'left',
				'options' => [
					'left' => [
						'title' => __( 'Left', 'botiga-pro' ),
						'icon' => 'eicon-text-align-left',
					],
					'right' => [
						'title' => __( 'Right', 'botiga-pro' ),
						'icon' => 'eicon-text-align-right',
					],
				],
				'condition' => [
					'displaySaleBadge' => 'yes'
				],			
			]
		);

		$this->add_control(
			'sale_badge_bg_color',
			[
				'label' => __( 'Background Color', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} ul.products li.product .onsale' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'displaySaleBadge' => 'yes'
				],
			]
		);

		$this->add_control(
			'sale_badge_color',
			[
				'label' => __( 'Text Color', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} ul.products li.product .onsale' => 'color: {{VALUE}};',
				],
				'condition' => [
					'displaySaleBadge' => 'yes'
				],
			]
		);

		$this->add_control(
			'title_heading',
			[
				'label' => esc_html__( 'Title', 'botiga-pro' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'displayTitle' => 'yes'
				],
			]
		);

		$this->add_control(
			'title_alignment',
			[
				'label' 		=> __( 'Alignment', 'botiga-pro'),
				'type' => Controls_Manager::CHOOSE,
				'default' => 'left',
				'options' => [
					'left' => [
						'title' => __( 'Left', 'botiga-pro' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => __( 'Center', 'botiga-pro' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => __( 'Right', 'botiga-pro' ),
						'icon' => 'eicon-text-align-right',
					],
				],
				'selectors' => [
					'{{WRAPPER}} ul.products li.product .woocommerce-loop-product__title' => 'text-align: {{VALUE}};',
				],
				'condition' => [
					'displayTitle' => 'yes'
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' 		=> 'title_typography',
				'selector' 	=> '{{WRAPPER}} ul.products li.product .woocommerce-loop-product__title',
				'scheme' 	=> Core\Schemes\Typography::TYPOGRAPHY_3,
				'condition' => [
					'displayTitle' => 'yes'
				],
			]
		);

		$this->add_control(
			'title_color',
			[
				'label' => __( 'Text Color', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} ul.products li.product .woocommerce-loop-product__title' => 'color: {{VALUE}};',
				],
				'condition' => [
					'displayTitle' => 'yes'
				],
			]
		);

		$this->add_control(
			'categories_heading',
			[
				'label' => esc_html__( 'Categories', 'botiga-pro' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'displayCategories' => 'yes'
				],
			]
		);

		$this->add_control(
			'categories_alignment',
			[
				'label' 		=> __( 'Alignment', 'botiga-pro'),
				'type' => Controls_Manager::CHOOSE,
				'default' => 'left',
				'options' => [
					'left' => [
						'title' => __( 'Left', 'botiga-pro' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => __( 'Center', 'botiga-pro' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => __( 'Right', 'botiga-pro' ),
						'icon' => 'eicon-text-align-right',
					],
				],
				'selectors' => [
					'{{WRAPPER}} ul.products li.product .product-category' => 'text-align: {{VALUE}};',
				],
				'condition' => [
					'displayCategories' => 'yes'
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' 		=> 'categories_typography',
				'selector' 	=> '{{WRAPPER}} ul.products li.product .product-category, ul.products li.product .product-category a',
				'scheme' 	=> Core\Schemes\Typography::TYPOGRAPHY_3,
				'condition' => [
					'displayCategories' => 'yes'
				],
			]
		);

		$this->add_control(
			'categories_color',
			[
				'label' => __( 'Text Color', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} ul.products li.product .product-category, ul.products li.product .product-category a' => 'color: {{VALUE}};',
				],
				'condition' => [
					'displayCategories' => 'yes'
				],
			]
		);

		$this->add_control(
			'categories_color_hover',
			[
				'label' => __( 'Text Color (hover)', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} ul.products li.product .product-category a:hover' => 'color: {{VALUE}};',
				],
				'condition' => [
					'displayCategories' => 'yes'
				],
			]
		);

		$this->add_control(
			'price_heading',
			[
				'label' => esc_html__( 'Price', 'botiga-pro' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'displayPrice' => 'yes'
				],
			]
		);

		$this->add_control(
			'price_alignment',
			[
				'label' 		=> __( 'Alignment', 'botiga-pro'),
				'type' => Controls_Manager::CHOOSE,
				'default' => 'left',
				'options' => [
					'left' => [
						'title' => __( 'Left', 'botiga-pro' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => __( 'Center', 'botiga-pro' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => __( 'Right', 'botiga-pro' ),
						'icon' => 'eicon-text-align-right',
					],
				],
				'selectors' => [
					'{{WRAPPER}} ul.products li.product .price' => 'text-align: {{VALUE}};',
				],
				'condition' => [
					'displayPrice' => 'yes'
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' 		=> 'price_typography',
				'selector' 	=> '{{WRAPPER}} ul.products li.product .price',
				'scheme' 	=> Core\Schemes\Typography::TYPOGRAPHY_3,
				'condition' => [
					'displayPrice' => 'yes'
				],
			]
		);

		$this->add_control(
			'price_color',
			[
				'label' => __( 'Text Color', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} ul.products li.product .price .woocommerce-Price-amount' => 'color: {{VALUE}};',
				],
				'condition' => [
					'displayPrice' => 'yes'
				],
			]
		);

		$this->add_control(
			'price_sale_color',
			[
				'label' => __( 'Text Color (sale)', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} ul.products li.product .price del .woocommerce-Price-amount' => 'color: {{VALUE}};',
				],
				'condition' => [
					'displayPrice' => 'yes'
				],
			]
		);

		$this->add_control(
			'rating_heading',
			[
				'label' => esc_html__( 'Rating', 'botiga-pro' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'displayRating' => 'yes'
				],
			]
		);

		$this->add_control(
			'rating_alignment',
			[
				'label' 		=> __( 'Alignment', 'botiga-pro'),
				'type' => Controls_Manager::CHOOSE,
				'default' => 'left',
				'options' => [
					'left' => [
						'title' => __( 'Left', 'botiga-pro' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => __( 'Center', 'botiga-pro' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => __( 'Right', 'botiga-pro' ),
						'icon' => 'eicon-text-align-right',
					],
				],
				'condition' => [
					'displayRating' => 'yes'
				],
			]
		);

		$this->add_control(
			'rating_color',
			[
				'label' => __( 'Icon Color', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} ul.products li.product .star-rating span::before' => 'color: {{VALUE}};',
				],
				'condition' => [
					'displayRating' => 'yes'
				],
			]
		);

		$this->add_control(
			'short_desc_heading',
			[
				'label' => esc_html__( 'Short Description', 'botiga-pro' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'displayShortDescription' => 'yes'
				],
			]
		);

		$this->add_control(
			'short_desc_alignment',
			[
				'label' 		=> __( 'Alignment', 'botiga-pro'),
				'type' => Controls_Manager::CHOOSE,
				'default' => 'left',
				'options' => [
					'left' => [
						'title' => __( 'Left', 'botiga-pro' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => __( 'Center', 'botiga-pro' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => __( 'Right', 'botiga-pro' ),
						'icon' => 'eicon-text-align-right',
					],
				],
				'selectors' => [
					'{{WRAPPER}} ul.products li.product .product-description' => 'text-align: {{VALUE}};',
				],
				'condition' => [
					'displayShortDescription' => 'yes'
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' 		=> 'short_desc_typography',
				'selector' 	=> '{{WRAPPER}} ul.products li.product .product-description',
				'scheme' 	=> Core\Schemes\Typography::TYPOGRAPHY_3,
				'condition' => [
					'displayShortDescription' => 'yes'
				],
			]
		);

		$this->add_control(
			'short_desc_color',
			[
				'label' => __( 'Text Color', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} ul.products li.product .product-description' => 'color: {{VALUE}};',
				],
				'condition' => [
					'displayShortDescription' => 'yes'
				],
			]
		);

		$this->add_control(
			'addtocart_heading',
			[
				'label' => esc_html__( 'Add To Cart', 'botiga-pro' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'displayAddToCart' => 'yes'
				],
			]
		);

		$this->add_control(
			'addtocart_alignment',
			[
				'label' 		=> __( 'Alignment', 'botiga-pro'),
				'type' => Controls_Manager::CHOOSE,
				'default' => 'left',
				'options' => [
					'left' => [
						'title' => __( 'Left', 'botiga-pro' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => __( 'Center', 'botiga-pro' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => __( 'Right', 'botiga-pro' ),
						'icon' => 'eicon-text-align-right',
					],
				],
				'condition' => [
					'displayAddToCart' => 'yes',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' 		=> 'addtocart_typography',
				'selector' 	=> '{{WRAPPER}} ul.products li.product .add_to_cart_button',
				'scheme' 	=> Core\Schemes\Typography::TYPOGRAPHY_3,
				'condition' => [
					'displayAddToCart' => 'yes'
				],
			]
		);

		$this->add_control(
			'addtocart_background_color',
			[
				'label' => __( 'Background Color', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} ul.products li.product .add_to_cart_button' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'displayAddToCart' => 'yes'
				],
			]
		);

		$this->add_control(
			'addtocart_background_color_hover',
			[
				'label' => __( 'Background Color (hover)', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} ul.products li.product .add_to_cart_button:hover' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'displayAddToCart' => 'yes'
				],
			]
		);

		$this->add_control(
			'addtocart_color',
			[
				'label' => __( 'Text Color', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} ul.products li.product .add_to_cart_button' => 'color: {{VALUE}};',
				],
				'condition' => [
					'displayAddToCart' => 'yes'
				],
			]
		);

		$this->add_control(
			'addtocart_color_hover',
			[
				'label' => __( 'Text Color (hover)', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} ul.products li.product .add_to_cart_button:hover' => 'color: {{VALUE}};',
				],
				'condition' => [
					'displayAddToCart' => 'yes'
				],
			]
		);

		$this->add_control(
			'addtocart_border_color',
			[
				'label' => __( 'Border Color', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} ul.products li.product .add_to_cart_button' => 'border-color: {{VALUE}};',
				],
				'condition' => [
					'displayAddToCart' => 'yes'
				],
			]
		);

		$this->add_control(
			'addtocart_border_color_hover',
			[
				'label' => __( 'Border Color (hover)', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} ul.products li.product .add_to_cart_button:hover' => 'border-color: {{VALUE}};',
				],
				'condition' => [
					'displayAddToCart' => 'yes'
				],
			]
		);

		$this->add_control(
			'wishlist_heading',
			[
				'label' => esc_html__( 'Wishlist', 'botiga-pro' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'displayWishlist' => 'yes'
				],
			]
		);

		$this->add_control(
			'wishlist_vertical_alignment',
			[
				'label' 		=> __( 'Vertical Alignment', 'botiga-pro'),
				'type' => Controls_Manager::CHOOSE,
				'default' => 'top',
				'options' => [
					'top' => [
						'title' => __( 'Top', 'botiga-pro' ),
						'icon' => 'eicon-v-align-top',
					],
					'bottom' => [
						'title' => __( 'Bottom', 'botiga-pro' ),
						'icon' => 'eicon-v-align-bottom',
					],
				],
				'condition' => [
					'displayWishlist' => 'yes'
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' 		=> 'wishlist_typography',
				'selector' 	=> '{{WRAPPER}} ul.products li.product .botiga-wishlist-button.botiga-wishlist-button-tooltip:before',
				'scheme' 	=> Core\Schemes\Typography::TYPOGRAPHY_3,
				'condition' => [
					'displayWishlist' => 'yes'
				],
			]
		);

		$this->add_control(
			'wishlist_icon_color',
			[
				'label' => __( 'Icon Color', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} ul.products li.product .botiga-wishlist-button svg path' => 'stroke: {{VALUE}};',
				],
				'condition' => [
					'displayWishlist' => 'yes'
				],
			]
		);

		$this->add_control(
			'wishlist_icon_color_active',
			[
				'label' => __( 'Icon Color (hover/active)', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} ul.products li.product .botiga-wishlist-button:hover svg path' => 'fill: {{VALUE}}; stroke: {{VALUE}};',
					'{{WRAPPER}} ul.products li.product .botiga-wishlist-button.active svg path' => 'fill: {{VALUE}}; stroke: {{VALUE}};'
				],
				'condition' => [
					'displayWishlist' => 'yes'
				],
			]
		);

		$this->add_control(
			'wishlist_tooltip_backgrund_color',
			[
				'label' => __( 'Tooltip Background Color', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} ul.products li.product .botiga-wishlist-button.botiga-wishlist-button-tooltip:before' => 'background-color: {{VALUE}};',
					'{{WRAPPER}} ul.products li.product .botiga-wishlist-button.botiga-wishlist-button-tooltip:after'  => 'border-left-color: {{VALUE}};',
				],
				'condition' => [
					'displayWishlist' => 'yes'
				],
			]
		);

		$this->add_control(
			'wishlist_tooltip_background_color',
			[
				'label' => __( 'Tooltip Text Color', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} ul.products li.product .botiga-wishlist-button.botiga-wishlist-button-tooltip:before' => 'color: {{VALUE}};',
				],
				'condition' => [
					'displayWishlist' => 'yes'
				],
			]
		);

		$this->add_control(
			'quickview_heading',
			[
				'label' => esc_html__( 'Quick View', 'botiga-pro' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'displayQuickView' => 'yes'
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' 		=> 'quickview_typography',
				'selector' 	=> '{{WRAPPER}} ul.products li.product .botiga-quick-view',
				'scheme' 	=> Core\Schemes\Typography::TYPOGRAPHY_3,
				'condition' => [
					'displayQuickView' => 'yes'
				],
			]
		);

		$this->add_control(
			'quickview_background_color',
			[
				'label' => __( 'Background Color', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} ul.products li.product .botiga-quick-view' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'displayQuickView' => 'yes'
				],
			]
		);

		$this->add_control(
			'quickview_background_color_hover',
			[
				'label' => __( 'Background Color (hover)', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} ul.products li.product .botiga-quick-view:hover' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'displayQuickView' => 'yes'
				],
			]
		);

		$this->add_control(
			'quickview_color',
			[
				'label' => __( 'Text Color', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} ul.products li.product .botiga-quick-view' => 'color: {{VALUE}};',
				],
				'condition' => [
					'displayQuickView' => 'yes'
				],
			]
		);

		$this->add_control(
			'quickview_color_hover',
			[
				'label' => __( 'Text Color (hover)', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} ul.products li.product .botiga-quick-view:hover' => 'color: {{VALUE}};',
				],
				'condition' => [
					'displayQuickView' => 'yes'
				],
			]
		);

		$this->add_control(
			'quickview_border_color',
			[
				'label' => __( 'Border Color', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} ul.products li.product .botiga-quick-view' => 'border-color: {{VALUE}};',
				],
				'condition' => [
					'displayQuickView' => 'yes'
				],
			]
		);

		$this->add_control(
			'quickview_border_color_hover',
			[
				'label' => __( 'Border Color (hover)', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} ul.products li.product .botiga-quick-view:hover' => 'border-color: {{VALUE}};',
				],
				'condition' => [
					'displayQuickView' => 'yes'
				],
			]
		);

		$this->add_control(
			'stock_heading',
			[
				'label' => esc_html__( 'Stock', 'botiga-pro' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => [
					'displayStock' => 'yes'
				],
			]
		);

		$this->add_control(
			'stock_alignment',
			[
				'label' 		=> __( 'Alignment', 'botiga-pro'),
				'type' => Controls_Manager::CHOOSE,
				'default' => 'left',
				'options' => [
					'left' => [
						'title' => __( 'Left', 'botiga-pro' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => __( 'Center', 'botiga-pro' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => __( 'Right', 'botiga-pro' ),
						'icon' => 'eicon-text-align-right',
					],
				],
				'selectors' => [
					'{{WRAPPER}} ul.products li.product .botiga-product-stock' => 'text-align: {{VALUE}};',
				],
				'condition' => [
					'displayStock' => 'yes'
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' 		=> 'stock_typography',
				'selector' 	=> '{{WRAPPER}} ul.products li.product .botiga-product-stock',
				'scheme' 	=> Core\Schemes\Typography::TYPOGRAPHY_3,
				'condition' => [
					'displayStock' => 'yes'
				],
			]
		);

		$this->add_control(
			'stock_color',
			[
				'label' => __( 'Text Color', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} ul.products li.product .botiga-product-stock' => 'color: {{VALUE}};',
				],
				'condition' => [
					'displayStock' => 'yes'
				],
			]
		);
		
		$this->end_controls_section();

	}

	/**
	 * Get the first product found in the database
	 * 
	 * @access protected
	 */
	protected function get_first_post() {
		if( is_singular( 'product' ) ) {
			global $post;
			return $post;
		}

		$first_product = wc_get_products( array(
			'limit' => 1,
			'orderby' => 'date',
			'order' => 'DESC',
			'return' => 'ids'
		) );

		$post = get_post( $first_product[0] );

		return $post;
	}

	/**
	 * Render icon list widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$wrapper_selector = $this->get_html_wrapper_class();
		$element_id       = $this->get_id();

		$edit_mode = \Elementor\Plugin::$instance->editor->is_edit_mode();

		$custom_css = '';

		// Align last block
		if( $settings['alignLastBlock'] === 'yes' ) {
			$custom_css .= "
				.${wrapper_selector}[data-id=\"$element_id\"] ul.products li.product *:last-child:not(.product-data-wrapper) {
					margin-top: auto !important;
				}
			";
		}

		// Sale Badge
		if( $settings['displaySaleBadge'] === 'yes' ) {
			if( 'right' === $settings['sale_badge_alignment'] ) {
				$custom_css .= "
					.${wrapper_selector}[data-id=\"$element_id\"] ul.products li.product .onsale {
						left: auto !important;
						right: 20px !important;
					}
				";
			}
		}

		// Rating
		if( $settings['displayRating'] === 'yes' ) {
			if( 'left' === $settings['rating_alignment'] ) {
				$custom_css .= "
					.${wrapper_selector}[data-id=\"$element_id\"] ul.products li.product .star-rating {
						margin: 0 0 10px;
						margin-right: auto; 
					}
				";
			} elseif( 'right' === $settings['rating_alignment'] ) {
				$custom_css .= "
					.${wrapper_selector}[data-id=\"$element_id\"] ul.products li.product .star-rating {
						margin: 0 0 10px;
						margin-left: auto; 
					}
				";
			}
		}

		// Add To Cart
		if( $settings['displayAddToCart'] === 'yes' ) {
			if( 'left' === $settings['addtocart_alignment'] ) {
				$custom_css .= "
					.${wrapper_selector}[data-id=\"$element_id\"] ul.products li.product .add_to_cart_button, .${wrapper_selector}[data-id=\"$element_id\"] ul.products li.product .product_type_grouped, .${wrapper_selector}[data-id=\"$element_id\"] ul.products li.product .product_type_variation {
						margin: 0;
						margin-right: auto; 
					}
					.${wrapper_selector}[data-id=\"$element_id\"] ul.products li.product .added_to_cart {
						text-align: left;
						margin-top: 10px;
						display: block; 
						width: 100%;
					}
				";
			} elseif( 'right' === $settings['addtocart_alignment'] ) {
				$custom_css .= "
					.${wrapper_selector}[data-id=\"$element_id\"] ul.products li.product .add_to_cart_button, .${wrapper_selector}[data-id=\"$element_id\"] ul.products li.product .product_type_grouped, .${wrapper_selector}[data-id=\"$element_id\"] ul.products li.product .product_type_variation {
						margin: 0;
						margin-left: auto; 
					}
					.${wrapper_selector}[data-id=\"$element_id\"] ul.products li.product .added_to_cart {
						text-align: right;
						margin-top: 10px;
						display: block; 
						width: 100%;
					}
				";
			}
		}

		// Wishlist
		if( $settings['displayWishlist'] === 'yes' ) {

			// Vertical alignment
			if( 'bottom' === $settings['wishlist_vertical_alignment'] ) {
				$custom_css .= "
					.${wrapper_selector}[data-id=\"$element_id\"] ul.products li.product .botiga-wishlist-button {
						bottom: 15px;
						top: auto;
					}
				";
			}
			
		}

		ob_start();
    
		$post    = $this->get_first_post();

		$limit   = absint( $settings['columns'] ) * absint( $settings['rows'] );
		$orderby = $settings['queryOrderBy'];
		$order   = $settings['queryOrder'];

		if( $settings[ 'queryType' ] === 'all-products' ) {
			$args = array(
				'post_type' => 'product',
				'posts_per_page' => $limit,
				'orderby' => $orderby,
				'order' => $order,		
				'paged' => get_query_var( 'paged' ) ? get_query_var( 'paged' ) : 1,
				'fields' => 'ids'
			);

			global $taxonomy;
			if( $taxonomy !== NULL ) {
				$args['tax_query'] = array(
					array(
						'taxonomy' => $taxonomy,
						'field'    => 'slug',
						'terms'    => get_queried_object()->slug,
					),
				);
			}

			$products_query = new \WP_Query( $args );
			$products       = $products_query->posts;
		} else if( $settings[ 'queryType' ] === 'related-products' ) {
			$related = wc_products_array_orderby( array_filter( array_map( 'wc_get_product', wc_get_related_products( $post->ID, -1 ) ), 'wc_products_array_filter_visible' ), $orderby, $order );
			$related = $limit > 0 ? array_slice( $related, 0, $limit ) : $related;

			$products = $related;
		} else if( $settings[ 'queryType' ] === 'upsell-products' ) {
			$product = wc_get_product( $post->ID );

			$upsells = wc_products_array_orderby( array_filter( array_map( 'wc_get_product', $product->get_upsell_ids() ), 'wc_products_array_filter_visible' ), $orderby, $order );
			$upsells = $limit > 0 ? array_slice( $upsells, 0, $limit ) : $upsells;

			$products = $upsells;
		}

		if( ! empty( $products ) ) { 
			
			// Sorting area
			$sorting_area_output = array();

			if( $settings[ 'queryType' ] === 'all-products' ) {

				// Results Count
				if( $settings[ 'displayResultCount' ] === 'yes' ) {
					$args = array(
						'total'    => $products_query->found_posts,
						'per_page' => $limit,
						'current'  => $products_query->get( 'paged' ) ? $products_query->get( 'paged' ) : 1
					);

					ob_start();
					botiga_woocommerce_result_count( $args );
					$sorting_area_output[] = ob_get_clean();
				}

				// Ordering
				if( $settings[ 'displayOrdering' ] === 'yes' ) {
					ob_start();
					botiga_woocommerce_catalog_ordering();
					$sorting_area_output[] = ob_get_clean();
				}

				// Grid & List View Toggle
				if( $settings[ 'displayGridListToggle' ] === 'yes' ) {
					wp_enqueue_script( 'botiga-grid-list-view', BOTIGA_PRO_URI . 'assets/js/botiga-grid-list-view.min.js', array(), BOTIGA_PRO_VERSION, true );

					global $product;
					ob_start();
					botiga_shop_grid_list_view_buttons( $product );
					$sorting_area_output[] = ob_get_clean();
				}

			}

			if( ! empty( $sorting_area_output ) ) { 
				echo '<div class="woocommerce-sorting-wrapper">';
					echo '<div class="row">';
						foreach( $sorting_area_output as $sort_item ) {

							echo '<div class="col-auto">';
								echo $sort_item;
							echo '</div>';

						}
					echo '</div>';
				echo '</div>';
			}
			
			?>

			<ul class="products columns-<?php echo esc_attr( $settings[ 'columns' ] ) ?>">

			<?php
			foreach( $products as $product_id ) {
				$_product = wc_get_product( $product_id ); ?>
		
				<li <?php wc_product_class( '', $_product ); ?>>
					<?php if( $settings[ 'displayImage' ] === 'yes' ) : ?>
						<div class="loop-image-wrap botiga-add-to-cart-button-layout2 botiga-quick-view-button-layout2">
							<a href="<?php echo esc_url( get_the_permalink( $product_id ) ); ?>" class="woocommerce-LoopProduct-link woocommerce-loop-product__link">
								<?php echo wp_kses_post( $_product->get_image() ); ?>
							</a>
							<?php 
							if( $settings[ 'displayQuickView' ] === 'yes' ) {
								botiga_quick_view_button( $_product );
							}

							if( $settings[ 'displaySaleBadge' ] === 'yes' ) {
								if ( $_product->is_on_sale() ) {
									echo apply_filters( 'woocommerce_sale_flash', '<span class="onsale">' . esc_html__( 'Sale!', 'botiga-pro' ) . '</span>', $post, $_product );
								}
							}

							if( $settings[ 'displayWishlist' ] === 'yes' ) {
								botiga_wishlist_button( $_product );
							}
							
							?>
						</div>
					<?php endif;

					if( $settings[ 'displayTitle' ] === 'yes' ) {
						echo '<h2 class="' . esc_attr( apply_filters( 'woocommerce_product_loop_title_classes', 'woocommerce-loop-product__title' ) ) . '">' . $_product->get_name() . '</h2>'; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
					}

					if( $settings[ 'displayPrice' ] === 'yes' ) {
						if ( $price_html = $_product->get_price_html() ) : ?>
							<span class="price"><?php echo $price_html; ?></span>
						<?php endif;
					}

					if( $settings[ 'displayCategories' ] === 'yes' ) {
						echo '<div class="product-category">' . wc_get_product_category_list( $_product->get_id() ) . '</div>'; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
					}
					
					if( $settings[ 'displayRating' ] === 'yes' ) {
						if ( wc_review_ratings_enabled() ) {

							$rating_count = $_product->get_rating_count();
							$average      = $_product->get_average_rating();
							
							if ( $rating_count > 0 ) : ?>
							
								<div class="woocommerce-product-rating">
									<?php echo wc_get_rating_html( $average, $rating_count ); // WPCS: XSS ok. ?>
								</div>
							
							<?php endif;
						}
					}

					if( $settings[ 'displayShortDescription' ] === 'yes' ) {
						echo '<div class="product-description">' . wp_kses_post( wp_trim_words( get_the_excerpt( $_product->get_id() ), 12, '&hellip;' ) ) . '</div>';
					}

					if( $settings[ 'displayStock' ] === 'yes' ) {
						botiga_add_product_stock_to_loop_items( $_product );
					}

					if( $settings[ 'displayAddToCart' ] === 'yes' ) {
						if ( $_product ) {
							$defaults = array(
								'quantity'   => 1,
								'class'      => implode(
									' ',
									array_filter(
										array(
											'button',
											'button', // escaped in the template.
											'product_type_' . $_product->get_type(),
											$_product->is_purchasable() && $_product->is_in_stock() ? 'add_to_cart_button' : '',
											$_product->supports( 'ajax_add_to_cart' ) && $_product->is_purchasable() && $_product->is_in_stock() ? 'ajax_add_to_cart' : '',
										)
									)
								),
								'attributes' => array(
									'data-product_id'  => $_product->get_id(),
									'data-product_sku' => $_product->get_sku(),
									'aria-label'       => $_product->add_to_cart_description(),
									'rel'              => 'nofollow',
								),
							);
				
							$args = apply_filters( 'woocommerce_loop_add_to_cart_args', $defaults, $_product );
				
							if ( isset( $args['attributes']['aria-label'] ) ) {
								$args['attributes']['aria-label'] = wp_strip_all_tags( $args['attributes']['aria-label'] );
							}
				
							echo sprintf(
								'<a href="%s" data-quantity="%s" class="%s" %s>%s</a>',
								esc_url( $_product->add_to_cart_url() ),
								esc_attr( isset( $args['quantity'] ) ? $args['quantity'] : 1 ),
								esc_attr( isset( $args['class'] ) ? $args['class'] : 'button' ),
								isset( $args['attributes'] ) ? wc_implode_html_attributes( $args['attributes'] ) : '',
								esc_html( $_product->add_to_cart_text() )
							);
						}
					} ?>
					
				</li>
		
				<?php
			} ?>

			</ul>

			<?php
			if( $settings[ 'queryType' ] === 'all-products' ) {

				// Pagination
				if( $settings[ 'displayPagination' ] === 'yes' ) {
					$args = array(
						'total'   => $products_query->max_num_pages,
						'current' => $products_query->get( 'paged' ) ? $products_query->get( 'paged' ) : 1,
						'base'    => str_replace( 999999999, '%#%', esc_url( get_pagenum_link( 999999999 ) ) ),
						'format'  => '/page/%#%',
					);

					botiga_woocommerce_pagination( $args );
				}
			
			}

		}

		$output = ob_get_clean();

		echo sprintf(
			'<div class="botiga-tb-el-block botiga-tb-el-wc-products"><style>%2$s</style>%1$s</div>',
			$output,
			$custom_css
		);
	}

	/**
	 * Render icon list widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function content_template() {	
	}
}
Plugin::instance()->widgets_manager->register_widget_type( new Botiga_Elementor_Shop_Products() );