<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor icon list widget.
 *
 * Elementor widget that displays a bullet list with any chosen icons and texts.
 *
 * @since 1.0.0
 */
class Botiga_Elementor_Single_Product_Price extends Widget_Base {

	/**
	 * Constructor
	 */
	public function __construct( $data=[], $args=null ){
		parent::__construct( $data, $args );
	}

	/**
	 * Load styles dependencies
	 */
	public function get_style_depends(){
		return [ 'botiga-builder-widgets-styles' ];	
	}

	public function get_help_url() {
		return 'https://docs.athemes.com/article/pro-templates-builder-overview/';
	}
		
	/**
	 * Get widget name.
	 *
	 * Retrieve icon list widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'botiga-elementor-single-product-price';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve icon list widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Single Product Price', 'botiga-pro' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve icon list widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-product-price';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the icon list widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'botiga-woocommerce' ];
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'price', 'product price', 'product', 'shop', 'botiga' ];
	}	


	/**
	 * Register icon list widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'section_style_general',
			[
				'label' => __( 'General', 'botiga-pro' ),
				'tab' 	=> Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'alignment',
			[
				'label' 		=> __( 'Alignment', 'botiga-pro'),
				'type' => Controls_Manager::CHOOSE,
				'default' => 'left',
				'options' => [
					'left' => [
						'title' => __( 'Left', 'botiga-pro' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => __( 'Center', 'botiga-pro' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => __( 'Right', 'botiga-pro' ),
						'icon' => 'eicon-text-align-right',
					],
				],
			]
		);

		$this->add_control(
			'regular_price_heading',
			[
				'label' => esc_html__( 'Regular Price', 'botiga-pro' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' 		=> 'regular_price_typography',
				'selector' 	=> '{{WRAPPER}} .price',
				'scheme' 	=> Core\Schemes\Typography::TYPOGRAPHY_3,
			]
		);

		$this->add_control(
			'regular_price_color',
			[
				'label' => __( 'Text Color', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .price .woocommerce-Price-amount' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'sale_price_heading',
			[
				'label' => esc_html__( 'Sale Price', 'botiga-pro' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' 		=> 'sale_price_typography',
				'selector' 	=> '{{WRAPPER}} .price del',
				'scheme' 	=> Core\Schemes\Typography::TYPOGRAPHY_3,
			]
		);

		$this->add_control(
			'sale_price_color',
			[
				'label' => __( 'Text Color', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .price del .woocommerce-Price-amount' => 'color: {{VALUE}};',
					'{{WRAPPER}} .price del' => 'color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_section();

	}

	/**
	 * Get the first product found in the database
	 * 
	 * @access protected
	 */
	protected function get_first_post() {
		if( is_singular( 'product' ) ) {
			global $post;
			return $post;
		}

		$first_product = wc_get_products( array(
			'limit' => 1,
			'orderby' => 'date',
			'order' => 'DESC',
			'return' => 'ids'
		) );

		$post = get_post( $first_product[0] );

		return $post;
	}

	/**
	 * Render icon list widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$wrapper_selector = $this->get_html_wrapper_class();
		$element_id       = $this->get_id();

		$edit_mode = \Elementor\Plugin::$instance->editor->is_edit_mode();

		$custom_css = '';

		if( $settings['alignment'] ) {
			if( 'center' === $settings['alignment'] ) {
				$custom_css .= "
					.${wrapper_selector}[data-id=\"$element_id\"] .product-gallery-summary {
						justify-content: center;
					}
				";
			} elseif( 'right' === $settings['alignment'] ) {
				$custom_css .= "
					.${wrapper_selector}[data-id=\"$element_id\"] .product-gallery-summary {
						justify-content: flex-end;
					}
				";
			}
		}

		echo '<div class="botiga-tb-el-block botiga-tb-el-sp-price">';
			if( $custom_css ) {
				echo '<style type="text/css">'. $custom_css .'</style>';
			}

			echo '<div class="product-gallery-summary" style="margin-bottom: 0;">';

				global $post;
				$post = $this->get_first_post();
				setup_postdata( $post );

				woocommerce_template_single_price();

				wp_reset_postdata();

			echo '</div>';
		echo '</div>';
	}

	/**
	 * Render icon list widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function content_template() {	
	}
}
Plugin::instance()->widgets_manager->register_widget_type( new Botiga_Elementor_Single_Product_Price() );