<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor icon list widget.
 *
 * Elementor widget that displays a bullet list with any chosen icons and texts.
 *
 * @since 1.0.0
 */
class Botiga_Elementor_Single_Product_Image extends Widget_Base {

	/**
	 * Constructor
	 */
	public function __construct( $data=[], $args=null ){
		parent::__construct( $data, $args );
	}

	/**
	 * Load styles dependencies
	 */
	public function get_style_depends(){
		return [ 'botiga-builder-widgets-styles' ];	
	}

	public function get_help_url() {
		return 'https://docs.athemes.com/article/pro-templates-builder-overview/';
	}
		
	/**
	 * Get widget name.
	 *
	 * Retrieve icon list widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'botiga-elementor-single-product-image';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve icon list widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Single Product Image', 'botiga-pro' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve icon list widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-product-images';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the icon list widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'botiga-woocommerce' ];
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'image', 'product image', 'product', 'shop', 'botiga' ];
	}	


	/**
	 * Register icon list widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'section_style_product_image',
			[
				'label' => __( 'Product Image', 'botiga-pro' ),
				'tab' 	=> Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name' => 'product_image_border',
				'selector' => '{{WRAPPER}} .flex-viewport',
				'fields_options' => [
					'border' => [
						'default' => '',
					],
					'width' => [
						'default' => [
							'top' => '1',
							'right' => '1',
							'bottom' => '1',
							'left' => '1',
							'isLinked' => true,
						],
					],
					'color' => [
						'default' => '#212121',
					],
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_product_image_thumbs',
			[
				'label' => __( 'Product Image Thumbnails', 'botiga-pro' ),
				'tab' 	=> Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name' => 'product_image_thumbs_border',
				'selector' => '{{WRAPPER}} .flex-control-thumbs li img, .flex-control-thumbs > div img',
				'fields_options' => [
					'border' => [
						'default' => '',
					],
					'width' => [
						'default' => [
							'top' => '1',
							'right' => '1',
							'bottom' => '1',
							'left' => '1',
							'isLinked' => true,
						],
					],
					'color' => [
						'default' => '#212121',
					],
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name' => 'product_image_thumbs_border_hover',
				'selector' => '{{WRAPPER}} .flex-control-thumbs li img.flex-active, {{WRAPPER}} .flex-control-thumbs li img:hover, {{WRAPPER}} .flex-control-thumbs > div:nth-child(1) img',
				'fields_options' => [
					'border' => [
						'default' => '',
					],
					'width' => [
						'default' => [
							'top' => '1',
							'right' => '1',
							'bottom' => '1',
							'left' => '1',
							'isLinked' => true,
						],
					],
					'color' => [
						'default' => '#212121',
					],
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_sale_badge',
			[
				'label' => __( 'Sale Badge', 'botiga-pro' ),
				'tab' 	=> Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' 		=> 'sale_badge_typography',
				'selector' 	=> '{{WRAPPER}} .onsale',
				'scheme' 	=> Core\Schemes\Typography::TYPOGRAPHY_3,
			]
		);

		$this->add_control(
			'sale_badge_text_color',
			[
				'label' => __( 'Text Color', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .onsale' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'sale_badge_background_color',
			[
				'label' => __( 'Background Color', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .onsale' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_section();

	}

	/**
	 * Get the first product found in the database
	 * 
	 * @access protected
	 */
	protected function get_first_product() {
		$first_product = wc_get_products( array(
			'limit' => 1,
			'orderby' => 'date',
			'order' => 'DESC',
			'return' => 'ids'
		) );

		$product = wc_get_product( $first_product[0] );

		return $product;
	}

	/**
	 * Render icon list widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		
		$settings = $this->get_settings_for_display();

		$edit_mode = \Elementor\Plugin::$instance->editor->is_edit_mode();

		echo '<div class="botiga-tb-el-block botiga-tb-el-sp-image">';

			if ( !$edit_mode && ! wp_doing_ajax() ) {
				if( function_exists( 'botiga_single_product_wrap_before' ) ) { 
					botiga_single_product_wrap_before();
				}
			
				woocommerce_show_product_images();
			
				if( function_exists( 'botiga_single_product_wrap_after' ) ) {
					botiga_single_product_wrap_after();
				}
			} else {
				$product = $this->get_first_product(); ?>

				<div class="image-placeholder flex-viewport">
					<?php 

					// Sale badge
					echo botiga_sale_badge( '', '', $product );

					// Product main image
					echo $product->get_image( 'botiga-large' ); ?>
				</div>
				<?php
				
				// Product image thumbnails/gallery
				$attachment_ids = $product->get_gallery_image_ids();
				if( $attachment_ids ) {
					echo '<div class="image-placeholder-thumbnails flex-control-thumbs">';
					foreach ( $attachment_ids as $attachment_id ) {
						echo wc_get_gallery_image_html( $attachment_id );
					}
					echo '</div>';
				}
			}

		echo '</div>';
	}

	/**
	 * Render icon list widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function content_template() {	
	}
}
Plugin::instance()->widgets_manager->register_widget_type( new Botiga_Elementor_Single_Product_Image() );