<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor icon list widget.
 *
 * Elementor widget that displays a bullet list with any chosen icons and texts.
 *
 * @since 1.0.0
 */
class Botiga_Elementor_Shop_Archive_Categories_Links extends Widget_Base {

	/**
	 * Constructor
	 */
	public function __construct( $data=[], $args=null ){
		parent::__construct( $data, $args );
	}

	/**
	 * Load styles dependencies
	 */
	public function get_style_depends(){
		return [ 'botiga-builder-widgets-styles' ];	
	}

	public function get_help_url() {
		return 'https://docs.athemes.com/article/pro-templates-builder-overview/';
	}
		
	/**
	 * Get widget name.
	 *
	 * Retrieve icon list widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'botiga-elementor-shop-archive-categories-links';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve icon list widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Archive Categories Links', 'botiga-pro' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve icon list widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-button';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the icon list widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'botiga-woocommerce' ];
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'archive', 'catalog', 'category', 'categories', 'links', 'product', 'shop', 'botiga' ];
	}	


	/**
	 * Register icon list widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {
		$this->start_controls_section(
			'section_style',
			[
				'label' => __( 'General', 'botiga-pro' ),
				'tab' 	=> Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' 		=> 'typography',
				'selector' 	=> '{{WRAPPER}} .woocommerce-page-header .categories-wrapper .category-button',
				'scheme' 	=> Core\Schemes\Typography::TYPOGRAPHY_3,
			]
		);

		$this->add_control(
			'alignment',
			[
				'label' 		=> __( 'Alignment', 'botiga-pro'),
				'type' => Controls_Manager::CHOOSE,
				'default' => 'left',
				'options' => [
					'flex-start' => [
						'title' => __( 'Left', 'botiga-pro' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => __( 'Center', 'botiga-pro' ),
						'icon' => 'eicon-text-align-center',
					],
					'flex-end' => [
						'title' => __( 'Right', 'botiga-pro' ),
						'icon' => 'eicon-text-align-right',
					],
				],
				'selectors' => [
					'{{WRAPPER}} .woocommerce-page-header .categories-wrapper' => 'justify-content: {{VALUE}};',
				],				
			]
		);

		$this->add_control(
			'border_radius',
			[
				'label' => __( 'Border Radius', 'botiga-pro' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'{{WRAPPER}} .woocommerce-page-header .categories-wrapper .category-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'color',
			[
				'label' 	=> __( 'Text Color', 'botiga-pro' ),
				'type' 		=> Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .woocommerce-page-header .categories-wrapper .category-button' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'color_hover',
			[
				'label' 	=> __( 'Text Color (hover)', 'botiga-pro' ),
				'type' 		=> Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .woocommerce-page-header .categories-wrapper .category-button:hover' => 'color: {{VALUE}} !important;',
				],
			]
		);
		
		$this->add_control(
			'background_color',
			[
				'label' 	=> __( 'Background Color', 'botiga-pro' ),
				'type' 		=> Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .woocommerce-page-header .categories-wrapper .category-button' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'background_color_hover',
			[
				'label' 	=> __( 'Background Color (hover)', 'botiga-pro' ),
				'type' 		=> Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .woocommerce-page-header .categories-wrapper .category-button:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'border_color',
			[
				'label' 	=> __( 'Border Color', 'botiga-pro' ),
				'type' 		=> Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .woocommerce-page-header .categories-wrapper .category-button' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'border_color_hover',
			[
				'label' 	=> __( 'Border Color (hover)', 'botiga-pro' ),
				'type' 		=> Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .woocommerce-page-header .categories-wrapper .category-button:hover' => 'border-color: {{VALUE}} !important;',
				],
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render icon list widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		if( ! function_exists( 'botiga_shop_page_header_category_links' ) ) {
			return;
		}
		
		$settings = $this->get_settings_for_display();

		$edit_mode = \Elementor\Plugin::$instance->editor->is_edit_mode();

		echo '<div class="botiga-tb-el-block botiga-tb-el-sa-categories-links">';
			echo '<div class="woocommerce-page-header" style="padding: 0; margin: 0 -15px;">';

					if ( !$edit_mode && ! wp_doing_ajax() ) {
						botiga_shop_page_header_category_links();
						botiga_shop_page_header_sub_category_links();
					} else { ?>
						<div class="container">
							<div class="categories-wrapper">
								<a href="javascript:void(0);" class="category-button" role="button"><?php echo esc_html__( 'Category One (3)', 'botiga-pro' ); ?></a>
								<a href="javascript:void(0);" class="category-button" role="button"><?php echo esc_html__( 'Category Two (1)', 'botiga-pro' ); ?></a>
								<a href="javascript:void(0);" class="category-button" role="button"><?php echo esc_html__( 'Category Three (5)', 'botiga-pro' ); ?></a>			
							</div>
						</div>
					<?php	
					}
					
			echo '</div>';
		echo '</div>';
	}

	/**
	 * Render icon list widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function content_template() {	
	}
}
Plugin::instance()->widgets_manager->register_widget_type( new Botiga_Elementor_Shop_Archive_Categories_Links() );