<?php
/**
 * Custom templates
 *
 * @package Botiga
 */


if ( !class_exists( 'Botiga_Templates' ) ) :
	Class Botiga_Templates {

		/**
		 * Instance
		 */		
		private static $instance;

		/**
		 * Initiator
		 */
		public static function get_instance() {
			if ( ! isset( self::$instance ) ) {
				self::$instance = new self;
			}
			return self::$instance;
		}

		/**
		 * Constructor
		 */
		public function __construct() {
			
			// Modal Popup Customizer Options
			add_filter( 'botiga_customizer_modal_popup_content_type_options', array( $this, 'modal_popup_customizer_options' ) );

			if ( is_admin() ) {
				
				// Mega menu options
				add_filter( 'botiga_mega_menu_content_type_select_options', array( $this, 'mega_menu_content_type_select_options' ) );

				// Elementor admin assets
				add_action( 'elementor/editor/before_enqueue_scripts', array( $this, 'botiga_elementor_editor_styles' ) );

				// Gutenberg Assets
				add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_gutenberg_assets' ) );

				// Reset some blocks css
				add_action( 'admin_enqueue_scripts', array( $this, 'reset_some_editor_blocks_styles' ) );

				// Gutenberg Blocks Custom Category
				if ( version_compare( get_bloginfo( 'version' ), '5.8', '>=' ) ) {
					add_filter( 'block_categories_all', array( $this, 'register_custom_blocks_inserter_categories' ) );
				} else {
					add_filter( 'block_categories', array( $this, 'register_custom_blocks_inserter_categories' ) );
				}

			} else {

				add_action( 'template_redirect', array( $this, 'render_custom_content' ) );
				add_action( 'template_redirect', array( $this, 'render_custom_shop_single' ) );
				add_action( 'template_redirect', array( $this, 'render_custom_shop_archive' ) );
				add_action( 'template_redirect', array( $this, 'render_404_page' ) );
				add_action( 'template_redirect', array( $this, 'redirect_single_ahf' ) );		
				add_action( 'single_template', array( $this, 'load_canvas_template' ) );		
				
				// Elementor Assets
				add_action( 'elementor/frontend/after_register_styles', array( $this, 'register_elementor_assets' ) );
				
				// Gutenberg Assets
				add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_gutenberg_assets' ) );

				// Load gutenberg blocks assets only if the block is in use on the page
				add_filter( 'should_load_separate_core_block_assets', '__return_true' );
	
				//Flush rewrite rules
				add_action( 'init', array( $this, 'flush_rules' ), 999 );

			}
		}
		
		/**
		 * Extend mega menu content type select options
		 */
		public function mega_menu_content_type_select_options( $options ) {
			$options[ 'content-block' ] = __( 'Custom Content Block', 'botiga-pro' );

			return $options;
		}

		/**
		 * Extend the customizer modal popup with new options
		 */
		public function modal_popup_customizer_options( $options ) {
			$options[ 'content-block' ] = __( 'Custom Content Block', 'botiga-pro' );

			return $options;
		}

		/**
		 * Enqueue Elementor editor style
		 */
		public function botiga_elementor_editor_styles() {
			wp_enqueue_style( 'botiga-elementor-editor-styles', BOTIGA_PRO_URI . 'inc/modules/templates-builder/assets/css/botiga-tb-elementor-editor.min.css' );
		}

		/**
		 * Render the custom content into the selected action hook
		 */
		public function render_custom_content() {

			$ids = $this->get_template_id( 'content' );

			if ( empty( $ids ) ) {
				return;
			}
				
			foreach ( $ids as $id ) {
				$hook 		= get_post_meta( $id, '_ahf_hook_type', true);
				$priority 	= get_post_meta( $id, '_ahf_hook_priority', true);

				if( 'none' === $hook ) {
					continue;
				}

				if ( !$priority ) {
					$priority = 10;
				}

				if ( $this->is_built_with_elementor( $id ) ) {
					add_action( $hook, function() use ( $id )  { echo '<div class="botiga-custom-content">' . Elementor\Plugin::$instance->frontend->get_builder_content_for_display( $id ) . '</div>'; }, $priority );
				} else {
					add_action( $hook, function() use ( $id )  { $thepost = get_post( $id ); echo '<div class="botiga-custom-content">' . do_blocks( $thepost->post_content ) . '</div>'; }, $priority );
				}

			}
		}

		/**
		 * Render 404 page
		 */
		public function render_404_page() {

			$args = array(
				'numberposts'      => -1,
				'meta_key'         => '_ahf_template_type',
				'meta_value'       => 'page404',
				'post_type'        => 'athemes_hf',
			);	
	
			$templates = get_posts( $args );

			$ids = array();

			foreach ( $templates as $template ) {
				$ids[] = $template->ID;
			}

			if ( empty( $ids ) ) {
				return;
			}

			$id = end( $ids );

			// Add body class identifier
			if( is_404() ) {
				add_filter( 'body_class', function( $classes ) use ( $id ) {
					$classes[] = 'botiga-custom-404 botiga-custom-template-'. esc_attr( $id );
					return $classes;
				} );
			}

			remove_action( 'botiga_404_content', 'botiga_404_page_content' );

			if ( $this->is_built_with_elementor( $id ) ) {
				add_action( 'botiga_404_content', function() use ( $id )  { echo '<div class="botiga-custom-404">' . Elementor\Plugin::$instance->frontend->get_builder_content_for_display( $id ) . '</div>'; } );
			} else {
				add_action( 'botiga_404_content', function() use ( $id )  { $thepost = get_post( $id ); echo '<div class="botiga-custom-404">' . do_blocks( $thepost->post_content ) . '</div>'; } );						
			}
		}
		
		/**
		 * Render custom shop single
		 */
		public function render_custom_shop_single() {

			if( ! class_exists( 'Woocommerce' ) ) {
				return;
			}

			if( ! is_singular( 'product' ) ) {
				return;
			}

			$ids = $this->get_template_id( 'shop-single' );

			if ( !$ids || empty( $ids ) ) {
				return;
			}

			$id = end( $ids );

			// Add body class identifier
			if( is_singular( 'product' ) ) {
				add_filter( 'body_class', function( $classes ) use ( $id ) {
					$classes[] = 'botiga-custom-shop-single botiga-custom-template-'. esc_attr( $id );
					return $classes;
				} );
			}

			// Remove woocommerce actions
			remove_action( 'woocommerce_before_main_content', 'woocommerce_breadcrumb', 20 );
			remove_action( 'woocommerce_before_main_content', 'botiga_single_product_navigation', 15 );
			remove_action( 'woocommerce_before_single_product', 'woocommerce_output_all_notices', 10 );
			remove_all_actions( 'woocommerce_before_single_product_summary' );
			remove_all_actions( 'woocommerce_after_single_product_summary' );
			remove_all_actions( 'woocommerce_single_product_summary' );

			if ( $this->is_built_with_elementor( $id ) ) {
				add_action( 'woocommerce_before_single_product_summary', function() use ( $id )  { echo '<div class="botiga-custom-shop-single">' . Elementor\Plugin::$instance->frontend->get_builder_content_for_display( $id ) . '</div>'; } );
			} else {
				add_action( 'woocommerce_before_single_product_summary', function() use ( $id )  { $thepost = get_post( $id ); echo '<div class="botiga-custom-shop-single blocks-content">' . do_blocks( $thepost->post_content ) . '</div>'; } );
			}
		}

		/**
		 * Render custom shop archive
		 */
		public function render_custom_shop_archive() {

			if( ! class_exists( 'Woocommerce' ) ) {
				return;
			}

			if( ! is_shop() && ! is_product_category() && ! is_product_tag() && ! is_product_taxonomy() ) {
				return;
			}
			
			$args = array(
				'numberposts'      => -1,
				'meta_key'         => '_ahf_template_type',
				'meta_value'       => 'shop-archive',
				'post_type'        => 'athemes_hf',
			);	
	
			$templates = get_posts( $args );

			$ids = array();

			foreach ( $templates as $template ) {
				$ids[] = $template->ID;
			}

			if ( empty( $ids ) ) {
				return;
			}

			$id = end( $ids );

			// Add body class identifier
			if( is_shop() || is_product_category() || is_product_tag() || is_product_taxonomy() ) {
				add_filter( 'body_class', function( $classes ) use ( $id ) {
					$classes[] = 'botiga-custom-shop-archive botiga-custom-template-'. esc_attr( $id );
					return $classes;
				} );
			}

			add_action( 'woocommerce_before_main_content', function(){
				do_action( 'botiga_templates_builder_wc_shop_archive' );
				get_footer( 'shop' );
				die();
			} );
			add_filter( 'botiga_show_woo_page_header_breadcrumbs', '__return_false' );
			remove_action( 'botiga_page_header', 'botiga_woocommerce_page_header' );
			remove_all_actions( 'woocommerce_before_shop_loop' );
			remove_all_actions( 'woocommerce_after_shop_loop' );

			if ( $this->is_built_with_elementor( $id ) ) {
				add_action( 'botiga_templates_builder_wc_shop_archive', function() use ( $id )  { echo '<div class="botiga-custom-shop-archive">' . Elementor\Plugin::$instance->frontend->get_builder_content_for_display( $id ) . '</div>'; } );
			} else {
				add_action( 'botiga_templates_builder_wc_shop_archive', function() use ( $id )  { $thepost = get_post( $id ); echo '<div class="botiga-custom-shop-archive blocks-content">' . do_blocks( $thepost->post_content ) . '</div>'; } );
			}
		}

		/**
		 * Template ID
		 */
		public function get_template_id( $template_type ) {

			$meta_query_args = array(
				array(
					'key'   => '_ahf_template_type',
					'value' => $template_type,
				),
			);
			
			//Get templates CPT
			$args = array(
				'numberposts' 	=> -1,
				'post_type'   	=> 'athemes_hf',
				'meta_query'	=> $meta_query_args,
				'fields'        => 'ids'
			);	

			$templates = get_posts( $args );

			$ids = array();

			foreach ( $templates as $template_id ) {

				$metas = get_post_meta( $template_id, 'ahf_display_rules', true);
	
				//Get current post type
				$current_type = '';
				if ( is_singular() || is_date() ) {
					$current_type 	= get_post_type();
				} elseif ( is_post_type_archive() ) {
					$current_type 	= get_queried_object()->name;
				} elseif ( is_tax() || is_category() || is_tag() ) {
					$current_type 	= get_taxonomy( get_queried_object()->taxonomy )->object_type[0];
					$taxonomies 	= get_object_taxonomies( $current_type );	
				}

				if ( ! empty( $metas ) && '' !== $metas ) {
					foreach ( $metas as $meta ) {
						switch ( $meta['select'] ) {
							case is_int( intval( $meta['select'] ) ) && 0 !== intval( $meta['select'] ):
								if ( intval( $meta['select'] ) === get_the_id() && is_singular() ) {
									$ids[] = $template_id;
								}
							break;
						}
					}
				}

				switch ( $current_type ) {

					case 'website':
					case 'all':
						$ids[] = $template_id;
					break;

					case 'homepage':
						if ( is_front_page() ) {
							$ids[] = $template_id;
						}
					break;

					case 'search':
						if ( is_search() ) {
							$ids[] = $template_id;
						}
					break;
						
					case 'blog':
						if ( is_home() ) {
							$ids[] = $template_id;
						}
					break;
						
					case 'shop':
						if ( function_exists( 'is_shop' ) && is_shop() ) {
							$ids[] = $template_id;
						}
					break;

					case 'page-404':
						if ( is_404() ) {
							$ids[] = $template_id;
						}
					break;

					case $current_type;
						if ( is_singular( $current_type ) ) {
							$ids[] = $template_id;
						}
					break;
						
					case $current_type . '-archives';
						if ( 'post' === $current_type ) {
							if ( is_category() || is_tag() || is_date() ) {
								$ids[] = $template_id;
							}
						} elseif ( is_post_type_archive( $current_type ) ) {
							$ids[] = $template_id;
						} elseif ( is_tax() ) {
							foreach ( $taxonomies as $taxonomy ) {
								if ( is_tax( $taxonomy ) ) {
									$ids[] = $template_id;
								}
							}
						}
					break;

				}

			}
	
			return $ids;
		}

		/**
		 * Check if template is built with Elementor
		 */
		public function is_built_with_elementor( $id ) {
			if ( class_exists( 'Elementor\Plugin' ) && Elementor\Plugin::$instance->documents->get( $id )->is_built_with_elementor() ) {
				return true;
			} 

			return false;
		}

		/**
		 * Redirect single template view to home
		 */
		public function redirect_single_ahf() {
			if ( is_singular( 'athemes_hf' ) && ! current_user_can( 'edit_posts' ) ) {
				wp_redirect( site_url(), 301 );
				die;
			}
		}

		/**
		 * Load Elementor canvas template
		 */
		function load_canvas_template( $single_template ) {

			if ( !class_exists( 'Elementor\Plugin' ) ) {
				return $single_template;
			}

			global $post;

			if ( 'athemes_hf' == $post->post_type ) {
				$elementor_2_0_canvas = ELEMENTOR_PATH . '/modules/page-templates/templates/canvas.php';

				if ( file_exists( $elementor_2_0_canvas ) ) {
					return $elementor_2_0_canvas;
				} else {
					return ELEMENTOR_PATH . '/includes/page-templates/canvas.php';
				}
			}

			return $single_template;
		}	
		
		/**
		 * Enqueue Elementor assets
		 */
		public function register_elementor_assets() {
			wp_register_style( 'botiga-builder-widgets-styles', BOTIGA_PRO_URI . 'inc/modules/templates-builder/assets/css/botiga-tb-elementor-widgets.min.css' );
		}

		/**
		 * Enqueue Gutenberg assets
		 */
		public function enqueue_gutenberg_assets() {
			wp_enqueue_style( 'botiga-tb-blocks-styles', BOTIGA_PRO_URI . 'inc/modules/templates-builder/assets/css/botiga-tb-gutenberg-blocks.min.css' );
		}

		/**
		 * Reset some blocks styles
		 */
		public function reset_some_editor_blocks_styles() {
			$css = "
				body.post-type-athemes_hf .editor-styles-wrapper h1,
				body.post-type-athemes_hf .editor-styles-wrapper h2,
				body.post-type-athemes_hf .editor-styles-wrapper h3,
				body.post-type-athemes_hf .editor-styles-wrapper h4,
				body.post-type-athemes_hf .editor-styles-wrapper h5,
				body.post-type-athemes_hf .editor-styles-wrapper h6 {
					margin-top: 0;
					margin-bottom: 0;
				}

				body.post-type-athemes_hf .block-editor-block-list__block:not(.wp-block-columns) {
					margin-top: 0;
					margin-bottom: 0;
				}
			";

			wp_add_inline_style( 'botiga-block-editor-styles', $css );
		}

		/**
		 * Register custom blocks inserter categories
		 */
		public function register_custom_blocks_inserter_categories( $categories ) {
			$categories[] = array(
				'slug'  => 'botiga-woocommerce',
				'title' => __( 'Botiga WooCommerce', 'botiga-pro' )
			);
		
			return $categories;
		}

		/**
		 * Flush rewrite rules
		 */
		public function flush_rules() {
			if ( !get_transient( 'botiga_templates_flushed_rules') ) {
				flush_rewrite_rules();
				set_transient( 'botiga_templates_flushed_rules', true, 0 );
			}
		}

	}

	/**
	 * Initialize class
	 */
	Botiga_Templates::get_instance();

endif;