<?php
/**
 * Mega Menu
 * 
 * @package Botiga_Pro
 */

if( ! class_exists( 'Botiga_Modules' ) ) {
	return;
}

if ( ! Botiga_Modules::is_module_active( 'mega-menu' ) ) {
	return;
}

class Botiga_Mega_Menu {
    /**
     * Instance
     */		
    private static $instance;

    /**
     * Initiator
     */
    public static function get_instance() {
        if ( ! isset( self::$instance ) ) {
            self::$instance = new self;
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    public function __construct() {
        global $pagenow;

        if( is_admin() ) {
            add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_scripts' ) );
            add_action( 'wp_nav_menu_item_custom_fields', array( $this, 'menu_options_button' ), 10, 4 );

            if( 'nav-menus.php' === $pagenow ) {
                add_action( 'admin_footer', array( $this, 'popup_wrapper' ) );
            }

            // Ajax actions
            add_action( 'wp_ajax_botiga_render_popup_content', array( $this, 'popup_content' ) );
            add_action( 'wp_ajax_botiga_save_popup_options', array( $this, 'save_options' ) );
        }

        add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );

        // Custom Walker for nav menus
        require 'class-mega-menu-nav-walker.php';
        require 'class-mobile-mega-menu-nav-walker.php';
        $menu_locations = array( 'primary', 'secondary' );

        foreach( $menu_locations as $location ) {
            add_filter( "botiga_{$location}_wp_nav_menu_walker", function(){
                return new Botiga_Custom_Nav_Walker();
            } );
        }

        add_filter( "botiga_mobile_primary_wp_nav_menu_walker", function(){
            return new Botiga_Mobile_Custom_Nav_Walker();
        } );
        
        // Filter theme custom css to append mega menu css while in the customizer edit page
        // that's needed because the theme custom css is not enqueued on customizer.
        if( is_customize_preview() ) {
            add_filter( 'botiga_custom_css_output', array( $this, 'get_mega_menu_css' ) );
        }

        // Update theme custom CSS with mega menu css when switch themes
        if( is_admin() ) {
            add_action( 'after_switch_theme', array( $this, 'update_custom_css_file' ) );
        }
    }

    /**
     * Enqueue Admin Scripts
     */
    public function admin_enqueue_scripts() {
        global $pagenow;

        if( 'nav-menus.php' === $pagenow ) {
            wp_enqueue_style( 'wp-color-picker' );
            wp_enqueue_script( 'wp-color-picker' );

            wp_enqueue_media();

            wp_enqueue_style( 'botiga-mega-menu', BOTIGA_PRO_URI . 'assets/css/admin/botiga-mega-menu.min.css', array(), BOTIGA_PRO_VERSION );
            wp_enqueue_script( 'botiga-mega-menu', BOTIGA_PRO_URI . 'assets/js/admin/botiga-mega-menu.min.js', array(), BOTIGA_PRO_VERSION, true );
        }
    }

    /**
     * Enqueue Front Scripts.
     */
    public function enqueue_scripts() {
        wp_enqueue_style( 'botiga-mega-menu', BOTIGA_PRO_URI . 'assets/css/botiga-mega-menu.min.css', array(), BOTIGA_PRO_VERSION );
    }

    /**
     * Mega Menu Options Button.
     */
    public function menu_options_button( $item_id, $menu_item, $depth, $args ) { 
        $is_mega_menu = get_post_meta( $item_id, '_is_mega_menu', true ) ? true : false; ?>

        <div class="botiga-menu-options">
            <a href="#" class="button button-secondary button-large" data-nonce="<?php echo esc_attr( wp_create_nonce( 'botiga-popup-content' ) ); ?>" data-item-id="<?php echo absint( $item_id ); ?>" data-item-depth="<?php echo absint( $depth ); ?>" data-item-is-mega-menu="<?php echo absint( $is_mega_menu ); ?>">
                <?php esc_html_e( 'Botiga Menu Options', 'botiga-pro' ); ?>
            </a>
        </div>

        <?php
    }

    /**
     * Popup Wrapper
     */
    public function popup_wrapper() { ?>
        <div id="botiga-menu-options-popup" class="botiga-menu-options-popup">
            <div class="botiga-popup-content-wrapper">
                <div class="botiga-popup-header">
                    <h2><?php echo esc_html__( 'Editing:', 'botiga-pro' ); ?><span class="botiga-popup-editing-item"></span></h2>
                    <a href="#" class="botiga-popup-close"><?php botiga_pro_get_svg_icon( 'icon-cancel', true ); ?></a>
                </div>
                <div class="botiga-popup-loader">
                    <svg xmlns="http://www.w3.org/2000/svg" width="25" height="25" viewBox="0 0 512 512" aria-hidden="true" focusable="false">
                        <path d="M288 39.056v16.659c0 10.804 7.281 20.159 17.686 23.066C383.204 100.434 440 171.518 440 256c0 101.689-82.295 184-184 184-101.689 0-184-82.295-184-184 0-84.47 56.786-155.564 134.312-177.219C216.719 75.874 224 66.517 224 55.712V39.064c0-15.709-14.834-27.153-30.046-23.234C86.603 43.482 7.394 141.206 8.003 257.332c.72 137.052 111.477 246.956 248.531 246.667C393.255 503.711 504 392.788 504 256c0-115.633-79.14-212.779-186.211-240.236C302.678 11.889 288 23.456 288 39.056z" />
                    </svg>
                </div>
                <div class="botiga-popup-content"></div>
                <div class="botiga-popup-footer">
                    <a 
                        href="#" 
                        class="botiga-popup-save-button button-primary button-large" 
                        data-default-text="<?php echo esc_attr__( 'Save', 'botiga-pro' ); ?>" 
                        data-loading-text="<?php echo esc_attr__( 'Loading...', 'botiga-pro' ); ?>">
                            <?php echo esc_html__( 'Save', 'botiga-pro' ); ?>
                    </a>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Popup Content
     */
    public function popup_content() { 
        $item_id       = isset( $_POST[ 'item_id' ] ) ? absint( $_POST[ 'item_id' ] ) : 0;
        $item_depth    = isset( $_POST[ 'item_depth' ] ) ? absint( $_POST[ 'item_depth' ] ) : 0;
        $top_parent_id = isset( $_POST[ 'top_parent_id' ] ) ? absint( $_POST[ 'top_parent_id' ] ) : 0;
        
        check_ajax_referer( 'botiga-popup-content', 'nonce' );

        if ( ! current_user_can( 'edit_theme_options' ) ) {
            wp_die();
        }

        // Check if menu item top parent is a mega menu
        $top_parent_has_mega_menu = $top_parent_id && get_post_meta( $top_parent_id, '_is_mega_menu', true ) ? true : false;

        // Fields.
        $fields = array(

            // Depth 0 fields.
            array(
                'id'          => '_is_mega_menu',
                'title'       => __( 'Mega Menu', 'botiga-pro' ),
                'description' => __( 'Turn this menu item into a mega menu.', 'botiga-pro' ),
                'type'        => 'checkbox',
                'default'     => 0,
                'depth_cond'  => 0 === $item_depth
            ),
            array(
                'id'          => '_is_mega_menu_width',
                'title'       => __( 'Mega Menu Width', 'botiga-pro' ),
                'description' => __( 'Controls the width from mega menu.', 'botiga-pro' ),
                'type'        => 'select',
                'default'     => 'contained',
                'options'     => array(
                    'contained'            => __( 'Contained', 'botiga-pro' ),
                    'contained-full-width' => __( 'Contained & Full Width Background', 'botiga-pro' ),
                    'full-width'           => __( 'Full Width', 'botiga-pro' )
                ),
                'depth_cond'  => 0 === $item_depth,
                'display_cond' => array(
                    'target_id'    => '_is_mega_menu',
                    'target_value' => 'on'
                )
            ),
            array(
                'id'          => '_is_mega_menu_wrapper_top_offset',
                'title'       => __( 'Mega Menu Dropdown Top Offset', 'botiga-pro' ),
                'description' => __( 'Controls the distance between main menu items and the mega menu dropdown when you mouse over the item. Important: This value is not in pixels.', 'botiga-pro' ),
                'type'        => 'number',
                'default'     => 3,
                'min'         => 2,
                'max'         => 10,
                'step'        => 0.1,
                'depth_cond'  => 0 === $item_depth,
                'display_cond' => array(
                    'target_id'    => '_is_mega_menu',
                    'target_value' => 'on'
                )
            ),
            array(
                'id'          => '_is_mega_menu_styling_divider',
                'title'       => __( 'Styling Options', 'botiga-pro' ),
                'type'        => 'divider',
                'depth_cond'  => 0 === $item_depth,
                'display_cond' => array(
                    'target_id'    => '_is_mega_menu',
                    'target_value' => 'on'
                )
            ),
            array(
                'id'          => '_is_mega_menu_style_background_type',
                'title'       => __( 'Background Type', 'botiga-pro' ),
                'description' => __( 'Controls the type of mega menu background.', 'botiga-pro' ),
                'type'        => 'select',
                'default'     => 'color',
                'options'     => array(
                    'color' => __( 'Color', 'botiga-pro' ),
                    'image' => __( 'Image', 'botiga-pro' )
                ),
                'depth_cond'  => 0 === $item_depth,
                'display_cond' => array(
                    'target_id'    => '_is_mega_menu',
                    'target_value' => 'on'
                )
            ),
            array(
                'id'           => '_is_mega_menu_style_background_color',
                'title'        => __( 'Background Color', 'botiga-pro' ),
                'description'  => __( 'Controls the color from mega menu background.', 'botiga-pro' ),
                'type'         => 'colorpicker',
                'default'      => '#FFF',
                'depth_cond'   => 0 === $item_depth,
                'display_cond' => array(
                    'target_id'    => '_is_mega_menu,_is_mega_menu_style_background_type',
                    'target_value' => 'on,color',
                    'multiple'     => true
                )
            ),
            array(
                'id'          => '_is_mega_menu_style_background_image',
                'title'       => __( 'Background Image', 'botiga-pro' ),
                'description' => __( 'Controls the image from mega menu background.', 'botiga-pro' ),
                'type'        => 'media_upload',
                'default'     => '',
                'depth_cond'  => 0 === $item_depth,
                'display_cond' => array(
                    'target_id'    => '_is_mega_menu,_is_mega_menu_style_background_type',
                    'target_value' => 'on,image',
                    'multiple'     => true
                )
            ),
            array(
                'id'          => '_is_mega_menu_style_background_image_size',
                'title'       => __( 'Background Image Size', 'botiga-pro' ),
                'description' => __( 'Controls the size from mega menu background image.', 'botiga-pro' ),
                'type'        => 'select',
                'default'     => 'cover',
                'options'     => array(
                    'cover'   => __( 'Cover', 'botiga-pro' ),
                    'contain' => __( 'Contain', 'botiga-pro' )
                ),
                'depth_cond'  => 0 === $item_depth,
                'display_cond' => array(
                    'target_id'    => '_is_mega_menu,_is_mega_menu_style_background_type',
                    'target_value' => 'on,image',
                    'multiple'     => true
                )
            ),
            array(
                'id'          => '_is_mega_menu_style_background_image_position',
                'title'       => __( 'Background Image Position', 'botiga-pro' ),
                'description' => __( 'Controls the position from mega menu background image.', 'botiga-pro' ),
                'type'        => 'select',
                'default'     => 'center',
                'options'     => array(
                    'top'    => __( 'Top', 'botiga-pro' ),
                    'center' => __( 'Center', 'botiga-pro' ),
                    'bottom' => __( 'Bottom', 'botiga-pro' ),
                    'left'   => __( 'Left', 'botiga-pro' ),
                    'right'  => __( 'Right', 'botiga-pro' ),
                ),
                'depth_cond'  => 0 === $item_depth,
                'display_cond' => array(
                    'target_id'    => '_is_mega_menu,_is_mega_menu_style_background_type',
                    'target_value' => 'on,image',
                    'multiple'     => true
                )
            ),
            array(
                'id'          => '_is_mega_menu_style_background_image_repeat',
                'title'       => __( 'Background Image Repeat', 'botiga-pro' ),
                'description' => __( 'Controls the repeat behavior from mega menu background image.', 'botiga-pro' ),
                'type'        => 'select',
                'default'     => 'no-repeat',
                'options'     => array(
                    'repeat'    => __( 'Repeat', 'botiga-pro' ),
                    'no-repeat' => __( 'No Repeat', 'botiga-pro' )
                ),
                'depth_cond'  => 0 === $item_depth,
                'display_cond' => array(
                    'target_id'    => '_is_mega_menu,_is_mega_menu_style_background_type',
                    'target_value' => 'on,image',
                    'multiple'     => true
                )
            ),
            array(
                'id'          => '_is_mega_menu_wrapper_padding',
                'title'       => __( 'Padding', 'botiga-pro' ),
                'description' => __( 'Controls the padding from mega menu wrapper.', 'botiga-pro' ),
                'type'        => 'spacing',
                'options'     => array( // values here acts as defaults as well
                    'top'    => 30,
                    'right'  => 30,
                    'bottom' => 30,
                    'left'   => 30
                ),
                'min'         => 0,
                'max'         => 200,
                'depth_cond'  => 0 === $item_depth,
                'display_cond' => array(
                    'target_id'    => '_is_mega_menu',
                    'target_value' => 'on'
                )
            ),
            array(
                'id'           => '_is_mega_menu_style_text_color',
                'title'        => __( 'Text Color', 'botiga-pro' ),
                'description'  => __( 'Controls the color from mega menu items.', 'botiga-pro' ),
                'type'         => 'colorpicker',
                'default'      => '#777',
                'depth_cond'   => 0 === $item_depth,
                'display_cond' => array(
                    'target_id'    => '_is_mega_menu',
                    'target_value' => 'on'
                )
            ),
            array(
                'id'           => '_is_mega_menu_style_text_color_hover',
                'title'        => __( 'Text Color Hover', 'botiga-pro' ),
                'description'  => __( 'Controls the hover color from mega menu items.', 'botiga-pro' ),
                'type'         => 'colorpicker',
                'default'      => '#212121',
                'depth_cond'   => 0 === $item_depth,
                'display_cond' => array(
                    'target_id'    => '_is_mega_menu',
                    'target_value' => 'on'
                )
            ),
            array(
                'id'          => '_is_mega_menu_style_heading_divider_color',
                'title'       => __( 'Headiings divider color', 'botiga-pro' ),
                'description' => __( 'Controls the headings divider color.', 'botiga-pro' ),
                'type'        => 'colorpicker',
                'default'     => '',
                'depth_cond'  => 0 === $item_depth,
                'display_cond' => array(
                    'target_id'    => '_is_mega_menu',
                    'target_value' => 'on'
                )
            ),
            array(
                'id'          => '_is_mega_menu_style_heading_text_color',
                'title'       => __( 'Headings text color', 'botiga-pro' ),
                'description' => __( 'Controls the headings text color.', 'botiga-pro' ),
                'type'        => 'colorpicker',
                'default'     => '',
                'depth_cond'  => 0 === $item_depth,
                'display_cond' => array(
                    'target_id'    => '_is_mega_menu',
                    'target_value' => 'on'
                )
            ),
            array(
                'id'          => '_is_mega_menu_style_heading_text_color_hover',
                'title'       => __( 'Headings text color hover', 'botiga-pro' ),
                'description' => __( 'Controls the headings text color when mouse over.', 'botiga-pro' ),
                'type'        => 'colorpicker',
                'default'     => '',
                'depth_cond'  => 0 === $item_depth,
                'display_cond' => array(
                    'target_id'    => '_is_mega_menu',
                    'target_value' => 'on'
                )
            ),

            // Depth === 1
            array(
                'id'          => '_is_mega_menu_column_width',
                'title'       => __( 'Column Width', 'botiga-pro' ),
                'description' => __( 'Controls the with from this mega menu column.', 'botiga-pro' ),
                'type'        => 'select',
                'default'     => 'auto',
                'options'     => array(
                    'auto'    => __( 'Auto', 'botiga-pro' ),
                    'custom'  => __( 'Custom', 'botiga-pro' )
                ),
                'depth_cond'  => 1 === $item_depth
            ),
            array(
                'id'          => '_is_mega_menu_column_width_custom_value',
                'title'       => __( 'Column Width Value (px)', 'botiga-pro' ),
                'description' => __( 'Controls the column width value in pixels.', 'botiga-pro' ),
                'type'        => 'number',
                'min'         => 0,
                'max'         => 500,
                'step'        => 1,
                'default'     => 200,
                'depth_cond'  => 1 === $item_depth,
                'display_cond' => array(
                    'target_id'    => '_is_mega_menu_column_width',
                    'target_value' => 'custom'
                )
            ),
            array(
                'id'          => '_is_mega_menu_column_padding',
                'title'       => __( 'Padding', 'botiga-pro' ),
                'description' => __( 'Controls the padding from this mega menu column.', 'botiga-pro' ),
                'type'        => 'spacing',
                'options'     => array( // values here acts as defaults as well
                    'top'    => 0,
                    'right'  => 0,
                    'bottom' => 0,
                    'left'   => 0
                ),
                'min'         => 0,
                'max'         => 200,
                'depth_cond'  => 1 === $item_depth
            ),

            // Depth > 0 fields
            array(
                'id'          => '_is_mega_menu_item_heading',
                'title'       => __( 'Turn this item into a mega menu heading', 'botiga-pro' ),
                'description' => __( 'Removes the link styling from this item and turns it into a heading. Styling options for heading is available on the mega menu top parent item.', 'botiga-pro' ),
                'type'        => 'checkbox',
                'default'     => 0,
                'depth_cond'  => $item_depth > 1
            ),
            array(
                'id'          => '_is_mega_menu_item_heading_divider',
                'title'       => __( 'Display a divider below the heading?', 'botiga-pro' ),
                'description' => __( 'Displays a line below the heading item. Styling options for divider is available on the mega menu top parent item.', 'botiga-pro' ),
                'type'        => 'checkbox',
                'default'     => 0,
                'depth_cond'  => $item_depth > 1,
                'display_cond' => array(
                    'target_id'    => '_is_mega_menu_item_heading',
                    'target_value' => 'on'
                )
            ),
            array(
                'id'          => '_is_mega_menu_item_disable_link',
                'title'       => __( 'Disable link from this item?', 'botiga-pro' ),
                'description' => __( 'Check to render this item without a link.', 'botiga-pro' ),
                'type'        => 'checkbox',
                'default'     => 0,
                'depth_cond'  => $item_depth > 1
            ),
            array(
                'id'          => '_is_mega_menu_item_content_divider',
                'title'       => __( 'Content', 'botiga-pro' ),
                'type'        => 'divider',
                'depth_cond'  => $item_depth > 0
            ),
            array(
                'id'          => '_is_mega_menu_item_content_type',
                'title'       => __( 'Content Type', 'botiga-pro' ),
                'description' => __( 'Which type of content this mega menu item should render.', 'botiga-pro' ),
                'type'        => 'select',
                'default'     => 'default',
                'options'     => apply_filters( 'botiga_mega_menu_content_type_select_options', array(
                    'default'       => __( 'Default', 'botiga-pro' ),
                    'html'          => __( 'Custom HTML/Shortcode', 'botiga-pro' )
                ) ),
                'depth_cond'  => $item_depth > 0
            ),
            array(
                'id'          => '_is_mega_menu_item_content_custom_html',
                'title'       => __( 'Custom HTML/Shortcode Content', 'botiga-pro' ),
                'description' => __( 'This field accepts HTML and Shortcode tags.', 'botiga-pro' ),
                'type'        => 'textarea',
                'default'     => '',
                'depth_cond'  => $item_depth > 0,
                'display_cond' => array(
                    'target_id'    => '_is_mega_menu_item_content_type',
                    'target_value' => 'html'
                )
            ),
            array(
                'id'          => '_is_mega_menu_item_content_custom_content_block',
                'title'       => __( 'Custom Content Block', 'botiga-pro' ),
                'description' => sprintf( __( 'Create your custom content blocks with the <a href="%s">Templates Builder</a> to render inside your mega menu.', 'botiga-pro' ), admin_url( 'edit.php?post_type=athemes_hf' ) ),
                'type'        => 'content-block-dropdown',
                'default'     => '',
                'depth_cond'  => $item_depth > 0,
                'display_cond' => array(
                    'target_id'    => '_is_mega_menu_item_content_type',
                    'target_value' => 'content-block'
                )
            ),

            // All depths.
            array(
                'id'          => '_item_badge_divider',
                'title'       => __( 'Badge/Label', 'botiga-pro' ),
                'type'        => 'divider',
                'depth_cond'  => 'all',
            ),
            array(
                'id'          => '_item_has_badge',
                'title'       => __( 'Display a badge/label on this item?', 'botiga-pro' ),
                'description' => __( 'Check to display a badge/label on this item. E.g: New, Sale, Featured, etc.', 'botiga-pro' ),
                'type'        => 'checkbox',
                'default'     => 0,
                'depth_cond'  => 'all'
            ),
            array(
                'id'          => '_item_badge_title',
                'title'       => __( 'Badge/Label title', 'botiga-pro' ),
                'description' => __( 'Controls the badge/label title.', 'botiga-pro' ),
                'type'        => 'text',
                'default'     => '',
                'depth_cond'  => 'all',
                'display_cond' => array(
                    'target_id'    => '_item_has_badge',
                    'target_value' => 'on'
                )
            ),
            array(
                'id'          => '_item_badge_text_color',
                'title'       => __( 'Badge/Label text color', 'botiga-pro' ),
                'description' => __( 'Controls the badge/label title text color.', 'botiga-pro' ),
                'type'        => 'colorpicker',
                'default'     => '',
                'depth_cond'  => 'all',
                'display_cond' => array(
                    'target_id'    => '_item_has_badge',
                    'target_value' => 'on'
                )
            ),
            array(
                'id'          => '_item_badge_background_color',
                'title'       => __( 'Badge/Label background color', 'botiga-pro' ),
                'description' => __( 'Controls the badge/label background color.', 'botiga-pro' ),
                'type'        => 'colorpicker',
                'default'     => '',
                'depth_cond'  => 'all',
                'display_cond' => array(
                    'target_id'    => '_item_has_badge',
                    'target_value' => 'on'
                )
            )
        );

        $output = '';
        ob_start(); 
        
        foreach( $fields as $field ) : 

            // Hide mega menu fields only on items with depth > 0 and if the
            // top parent menu item isn't a mega menu.
            if( $item_depth !== 0 && $top_parent_has_mega_menu === false ) {
                if( strpos( $field[ 'id' ], '_is_mega_menu' ) !== FALSE ) {
                    continue;
                }
            }

            // Don't render badge fields on column items if top parent is mega menu
            if( $item_depth === 1 && $top_parent_has_mega_menu && strpos( $field[ 'id' ], '_badge' ) !== FALSE ) {
                continue;
            }

            // Conditional Fields
            $extra_attrs = array(); 

            if( isset( $field[ 'display_cond' ] ) ) {
                $extra_attrs[] = 'data-display-cond-target="'. esc_attr( $field[ 'display_cond' ][ 'target_id' ] ) .'"';
                $extra_attrs[] = 'data-display-cond-target-value="'. esc_attr( $field[ 'display_cond' ][ 'target_value' ] ) .'"';

                if( isset( $field[ 'display_cond' ][ 'multiple' ] ) ) {
                    $extra_attrs[] = 'data-display-cond-multiple="1"';
                }
            }

            if( ( $field[ 'depth_cond' ] || $field[ 'depth_cond' ] === 'all' ) && $field[ 'type' ] !== 'divider' ) { ?>

                <div class="botiga-popup-content-row botiga-popup-field-wrapper"<?php echo implode( ' ', $extra_attrs ); ?>>
                    <div class="botiga-popup-content-col">
                        <strong class="botiga-popup-field-title">
                            <?php echo esc_html( $field[ 'title' ] ); ?>
                        </strong>
                        <?php if( isset( $field[ 'description' ] ) && $field[ 'description' ] ) : ?>
                            <p class="botiga-popup-field-description">
                                <?php echo wp_kses_post( $field[ 'description' ] ); ?>
                            </p>
                        <?php endif; ?>
                    </div>
                    <div class="botiga-popup-content-col">
                        <?php echo $this->get_field( $item_id, $field ); ?>
                    </div>
                </div>
        
            <?php 
            } else if( $field[ 'depth_cond' ] && $field[ 'type' ] === 'divider' ) { ?>

                <div class="botiga-popup-content-row botiga-popup-field-wrapper"<?php echo implode( ' ', $extra_attrs ); ?>>
                    <div class="botiga-popup-content-col botiga-popup-content-col-100">
                        <h2><?php echo esc_html( $field[ 'title' ] ); ?></h2>
                        <hr>
                    </div>
                </div>

            <?php
            }
        endforeach;

        // Save nonce
        echo '<input type="hidden" id="botiga-popup-cotent-save-'. esc_attr( $item_id ) .'" class="botiga-popup-content-save-nonce" name="botiga-popup-cotent-save-'. esc_attr( $item_id ) .'" value="'. esc_attr( wp_create_nonce( 'botiga-save-popup-options' ) ) .'" />';

        $output = ob_get_clean();

        wp_send_json_success( $output );
    }

    /**
     * Get value from fields
     */
    public function get_field_value( $item_id, $field ) {
        $value = '';

        if( isset( $field[ 'id' ] ) ) {
            $value = get_post_meta( $item_id, $field[ 'id' ], true );
            
            if( ( 'number' === $field[ 'type' ] || 'hidden' === $field[ 'type' ] ) && ! $value && $value !== '0' && isset( $field[ 'default' ] ) ) {
                $value = $field[ 'default' ];
            }

            if( 'spacing' === $field[ 'type' ] ) {
                $values = array();

                foreach( $field[ 'options' ] as $side => $val ) {
                    $values[ $side ] = get_post_meta( $item_id, $field[ 'id' ] . '_' . $side, true );
                }

                return $values;
            }
        } else {
            $value = false;
        }

        return $value;
    }

    /**
     * Get field output based on type
     */
    public function get_field( $item_id, $field ) {
        if( ! $field ) {
            return '';
        }

        $value = $this->get_field_value( $item_id, $field );

        $output = '';

        switch ( $field[ 'type' ] ) {
            case 'checkbox':
                $output = '<input type="checkbox" class="botiga-popup-field botiga-popup-field-checkbox" name="'. esc_attr( $field[ 'id' ] ) .'" '. checked( (int) $value, 1, false ) .'/>';
                break;

            case 'select':
                if( isset( $field[ 'options' ] ) ) {
                    $output = '<select class="botiga-popup-field botiga-popup-field-select" name="'. esc_attr( $field[ 'id' ] ) .'">';
                    
                        foreach( $field[ 'options' ] as $option_value => $option_name ) {
                            $output .= '<option value="'. esc_attr( $option_value ) .'" '. selected( $value, $option_value, false ) .'>';
                                $output .= esc_html( $option_name );
                            $output .= '</option>';
                        }
    
                    $output .= '</select>';
                }
                break;

            case 'colorpicker':
                $output = '<input data-color-val="'. esc_attr( $value ) .'" name="'. esc_attr( $field[ 'id' ] ) .'" value="'. esc_attr( $value ) .'" class="botiga-popup-field botiga-popup-colorpicker" type="text" style="background-color: '. esc_attr( $value ) .';" />';
                break;

            case 'media_upload':
                $image_id = $value;

                if( $image = wp_get_attachment_image_src( $image_id ) ) {

                    $output = '<a href="#" class="botiga-popup-media-upload"><img src="' . esc_url( $image[0] ) . '" /></a>
                          <a href="#" class="botiga-popup-media-upload-remove">'. esc_html__( 'Remove Image', 'botiga-pro' ) .'</a>
                          <input class="botiga-popup-field botiga-popup-media-upload" type="hidden" name="'. esc_attr( $field[ 'id' ] ) .'" value="' . absint( $image_id ) . '">';
                
                } else {
                
                    $output = '<a href="#" class="botiga-popup-media-upload">'. esc_html__( 'Upload Image', 'botiga-pro' ) .'</a>
                          <a href="#" class="botiga-popup-media-upload-remove" style="display:none">'. esc_html__( 'Remove Image', 'botiga-pro' ) .'</a>
                          <input class="botiga-popup-field botiga-popup-media-upload" type="hidden" name="'. esc_attr( $field[ 'id' ] ) .'" value="">';
                
                } 
                break;

            case 'number':
                $output = '<input type="number" class="botiga-popup-field botiga-popup-input-number" name="'. esc_attr( $field[ 'id' ] ) .'" step="'. esc_attr( $field[ 'step' ] ) .'" min="'. absint( $field[ 'min' ] ) .'" max="'. absint( $field[ 'max' ] ) .'" value="'. absint( $value ) .'" />';
                break;

            case 'spacing':
                $output .= '<div class="botiga-popup-spacing-input-wrapper">';
                    foreach( $field[ 'options' ] as $side => $val ) {
                        $name = $field[ 'id' ] . '_' . $side; 

                        $output .= '<div class="botiga-popup-spacing-input-inner-wrapper">';
                            $output .= '<label>'. ucwords( esc_html( $side ) ) .'</label>'; 
                            $output .= '<input type="number" class="botiga-popup-field botiga-popup-input-spacing" name="'. esc_attr( $name ) .'" min="'. absint( $field[ 'min' ] ) .'" max="'. absint( $field[ 'max' ] ) .'" value="'. absint( $value[ $side ] ) .'" />';
                        $output .= '</div>';
                    }
                    $output .= '<input type="hidden" class="botiga-popup-field botiga-popup-input-spacing" name="'. esc_attr( $field[ 'id' ] ) .'" value="1" />';
                $output .= '</div>';
                break;

            case 'textarea':
                $output = '<textarea class="botiga-popup-field botiga-popup-textarea" name="'. esc_attr( $field[ 'id' ] ) .'">';
                    $output .= esc_textarea( $value );
                $output .= '</textarea>';
                break;

            case 'text':
                $output = '<input type="text" class="botiga-popup-field botiga-popup-input-text" name="'. esc_attr( $field[ 'id' ] ) .'" value="'. esc_attr( $value ) .'" />';
                break;

            case 'hidden':
                $output = '<input type="hidden" class="botiga-popup-field botiga-popup-input-hidden" name="'. esc_attr( $field[ 'id' ] ) .'" value="'. esc_attr( $value ) .'" />';
                break;

            case 'content-block-dropdown':
                $options = '';

                $options = get_posts( array(
                    'post_type'      => 'athemes_hf',
                    'posts_per_page' => -1,
                    'fields'         => 'ids',
                    'meta_query'     => array(
                        array(
                            'key'     => '_ahf_template_type',
                            'value'   => 'content',
                            'compare' => '=',
                        ),
                    ),
                ) );

                if( ! empty( $options ) ) {
                    $output = '<select class="botiga-popup-field botiga-popup-field-select" name="'. esc_attr( $field[ 'id' ] ) .'">';
                    
                        foreach( $options as $template_id ) {
                            $output .= '<option value="'. esc_attr( $template_id ) .'" '. selected( $value, $template_id, false ) .'>';
                                $output .= esc_html( get_the_title( $template_id ) );
                            $output .= '</option>';
                        }
    
                    $output .= '</select>';
                }
                break;
            
        }

        return $output;
    }

    /**
     * Save options
     */
    public function save_options() {
        $item_id = isset( $_POST[ 'item_id' ] ) ? absint( $_POST[ 'item_id' ] ) : 0;
        
        check_ajax_referer( 'botiga-save-popup-options', 'nonce' );

        if ( ! current_user_can( 'edit_theme_options' ) ) {
            wp_die();
        }

        $options = isset( $_POST['options'] ) ? $_POST['options'] : array();
        
        if( ! empty( $options ) ) {
            foreach( $options as $key => $value ) {
                $meta_key   = sanitize_text_field( $key );

                if( $key === '_is_mega_menu_item_content_custom_html' ) {
                    $meta_value = wp_kses_post( wp_unslash( $value ) );
                } else {
                    $meta_value = sanitize_text_field( wp_unslash( $value ) );
                }

                update_post_meta( $item_id, $meta_key, $meta_value );
            }
        }

        // Generate CSS for all mega menu items
        $this->save_mega_menu_css_as_option();
        $this->update_custom_css_file();

        // Success
        wp_send_json_success();
    }

    /**
     * Save mega menu css on database
     * This is needed to properly render CSS for multiple mega menus
     * e.g: if you have primary and secondary locations with mega menu
     * 
     */
    public function save_mega_menu_css_as_option() {
        $css = '';

        $menu_locations = array( 'primary', 'secondary' );
        $locations      = get_nav_menu_locations();
        $mm_body_class  = false;

        foreach( $menu_locations as $menu_location ) {
            if( ! isset( $locations[ $menu_location ] ) ) {
                continue;
            }

            $menu       = wp_get_nav_menu_object( $locations[ $menu_location ] );
            $menu_items = wp_get_nav_menu_items( $menu->term_id );
    
            foreach( $menu_items as $menu_item ) {
                if( $menu_item->menu_item_parent != 0 ) {
                    continue;
                }
    
                $has_mega_menu = get_post_meta( $menu_item->ID, '_is_mega_menu', true );
                if( $has_mega_menu ) {
                    $css .= Botiga_Mega_Menu::mega_menu_custom_css( $menu_item->ID );

                    $mm_body_class = true;
                }
            }
        }

        // Replace position 'relative' from some elements with 'static'
        if( $mm_body_class ) {
            $css .= '
                @media (min-width: 1025px) {
                    .top-bar div[class*="col-"],
                    .top-bar .col,
                    #masthead div[class*="col-"],
                    #masthead .col,
                    .bottom-header-row div[class*="col-"],
                    .bottom-header-row .col,
                    .main-navigation.botiga-dropdown .nav-menu > li.botiga-mega-menu ,
                    .secondary-navigation.botiga-dropdown .nav-menu > li.botiga-mega-menu  {
                        position: static;
                    }
                }
            ';
        }
        
        update_option( 'botiga-mega-menu-css', apply_filters( 'botiga_mega_menu_css', $css ) );
    }

    /**
     * Generates mega menu custom css.
     */
    public static function mega_menu_custom_css( $item_id = 0 ) {

        if( ! $item_id ) {
            return '';
        }

        $css = '@media (min-width: 1025px) {';

            // Mega menu wrapper/dropdown top offset
            $mm_wrapper_top_offset   = esc_attr( get_post_meta( $item_id, '_is_mega_menu_wrapper_top_offset', true ) );
            $mm_wrapper_transoform_y = $mm_wrapper_top_offset - 2;

                $css .= "
                    .menu-item-$item_id.botiga-mega-menu:hover:before { 
                        content: '';
                        position: relative;
                        top: 70%;
                        display: block;
                        width: 100%;
                        height: {$mm_wrapper_top_offset}em;
                        margin-bottom: -{$mm_wrapper_top_offset}em;
                        z-index: 999;
                    }

                    .menu-item-$item_id.botiga-mega-menu>.sub-menu {
                        transform: translate3d(-50%, {$mm_wrapper_transoform_y}em, 0) !important;
                    }
                ";

            // Background
            $mm_background_type = get_post_meta( $item_id, '_is_mega_menu_style_background_type', true );

            if( 'color' === $mm_background_type ) {
                $mm_width            = get_post_meta( $item_id, '_is_mega_menu_width', true );
                $mm_background_color = esc_attr( get_post_meta( $item_id, '_is_mega_menu_style_background_color', true ) );

                if( 'contained-full-width' === $mm_width ) {
                    $css .= "
                        .menu-item-$item_id.botiga-mega-menu.botiga-mega-menu-contained-full-width>.sub-menu:before {
                            background-color: $mm_background_color;
                        }
                    "; 
                }

                $css .= "
                    .menu-item-$item_id.botiga-mega-menu>.sub-menu { 
                        background-color: $mm_background_color;
                    }
                ";
            }

            if( 'image' === $mm_background_type ) {
                $mm_width                      = get_post_meta( $item_id, '_is_mega_menu_width', true );
                $mm_background_image_id        = get_post_meta( $item_id, '_is_mega_menu_style_background_image', true );
                $mm_background_image_size      = get_post_meta( $item_id, '_is_mega_menu_style_background_image_size', true );
                $mm_background_image_position  = get_post_meta( $item_id, '_is_mega_menu_style_background_image_position', true );
                $mm_background_image_repeat    = get_post_meta( $item_id, '_is_mega_menu_style_background_image_repeat', true );
                $mm_background_image_url       = wp_get_attachment_image_src( $mm_background_image_id, 'botiga-extra-large' );
                
                if( 'contained-full-width' === $mm_width ) {
                    $css .= "
                        .menu-item-$item_id.botiga-mega-menu.botiga-mega-menu-contained-full-width>.sub-menu:before {
                            background-image: url($mm_background_image_url[0]);
                            background-size: $mm_background_image_size;
                            background-position: $mm_background_image_position;
                            background-repeat: $mm_background_image_repeat;
                        }
                    "; 
                } else {
                    $css .= ".menu-item-$item_id.botiga-mega-menu>.sub-menu { 
                        background-image: url($mm_background_image_url[0]);
                        background-size: $mm_background_image_size;
                        background-position: $mm_background_image_position;
                        background-repeat: $mm_background_image_repeat;
                    }";
                }
                
            }

            // Padding
            $mm_padding_top    = get_post_meta( $item_id, '_is_mega_menu_wrapper_padding_top', true );
            $mm_padding_right  = get_post_meta( $item_id, '_is_mega_menu_wrapper_padding_right', true );
            $mm_padding_bottom = get_post_meta( $item_id, '_is_mega_menu_wrapper_padding_bottom', true );
            $mm_padding_left   = get_post_meta( $item_id, '_is_mega_menu_wrapper_padding_left', true );

            $css .= ".menu-item-$item_id.botiga-mega-menu>.sub-menu { 
                padding-top: ${mm_padding_top}px; 
                padding-right: ${mm_padding_right}px; 
                padding-bottom: ${mm_padding_bottom}px;
                padding-left: ${mm_padding_left}px;
            }";

            // Text Color
            $mm_text_color         = get_post_meta( $item_id, '_is_mega_menu_style_text_color', true );
            $mm_text_color_hover   = get_post_meta( $item_id, '_is_mega_menu_style_text_color_hover', true );

            $css .= ".menu-item-$item_id.botiga-mega-menu>.sub-menu>li>.sub-menu>li>a {
                color: ${mm_text_color};
            }";
            $css .= ".menu-item-$item_id.botiga-mega-menu>.sub-menu>li>.sub-menu>li:hover>a {
                color: ${mm_text_color_hover};
            }";
            $css .= ".menu-item-$item_id.botiga-mega-menu>.sub-menu>li>.sub-menu>li>a+.dropdown-symbol svg {
                fill: ${mm_text_color};
            }";
            $css .= ".menu-item-$item_id.botiga-mega-menu>.sub-menu>li>.sub-menu>li:hover>a+.dropdown-symbol svg {
                fill: ${mm_text_color_hover};
            }";

            // Item links as heading
            // Divider color
            $mm_item_heading_divider_color = get_post_meta( $item_id, '_is_mega_menu_style_heading_divider_color', true );

            $css .= ".menu-item-$item_id.botiga-mega-menu .botiga-mega-menu-link.has-mega-menu-heading-divider {
                border-bottom-color: ${mm_item_heading_divider_color};
            }";

            // Link as heading text color
            $mm_heading_text_color       = get_post_meta( $item_id, '_is_mega_menu_style_heading_text_color', true );
            $mm_heading_text_color_hover = get_post_meta( $item_id, '_is_mega_menu_style_heading_text_color_hover', true );

            $css .= ".menu-item-$item_id.botiga-mega-menu>.sub-menu>li>.sub-menu>li>a.is-mega-menu-heading {
                color: ${mm_heading_text_color} !important;
            }";
            $css .= ".menu-item-$item_id.botiga-mega-menu>.sub-menu>li>.sub-menu>li:hover>a.is-mega-menu-heading {
                color: ${mm_heading_text_color_hover} !important;
            }";
        $css .= '}';

        return $css;
    }

    /**
     * Get mega menu CSS from 'botiga-mega-menu-css' option.
     */
    public function get_mega_menu_css( $css ) {
        $mm_css = get_option( 'botiga-mega-menu-css' );

        if( ! $mm_css ) {
            return $css;
        }

        $css .= $mm_css;

        return $css;
    }

    /**
     * Wrapper to upgrade custom CSS file
     * 
     */
    public function update_custom_css_file() {

        // Append mega menu CSS to the theme custom CSS file.
        add_filter( 'botiga_custom_css_output', array( $this, 'get_mega_menu_css' ) );

        // Update the custom CSS file to apply mega menu css.
        $this->update_theme_custom_css_file();
    }

    /**
     * Update theme custom CSS file.
     * 
     */
    public function update_theme_custom_css_file() {
        $custom_css = Botiga_Custom_CSS::get_instance();
        $custom_css->update_custom_css_file();
    }

}

/**
 * Initialize class
 */
Botiga_Mega_Menu::get_instance();