import classnames from 'classnames';
import { __ } from '@wordpress/i18n';
import { useState } from '@wordpress/element';
import ServerSideRender from '@wordpress/server-side-render';
import { PanelBody, SelectControl, ToggleControl, RangeControl, Disabled } from '@wordpress/components';
import { BotigaDimensions, BotigaTypography, BotigaColorPicker, BotigaTabsNavigation, BotigaTabsContent, BotigaBorder } from '../_inc/global-components';
import { InspectorControls, useBlockProps } from '@wordpress/block-editor';
import './editor.scss';
import { generateBlockCustomCss } from '../_inc/helpers';

/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {WPElement} Element to render.
 */
export default function Edit( props ) {
	const { name, attributes, setAttributes } = props;

	const { 
		customCss,

		queryType,
		queryOrderBy,
		queryOrder,
		columns,
		rows,
		displayResultCount,
		displayOrdering,
		displayGridListToggle,
		displayPagination,
		paginationType,
		displayImage,
		displaySaleBadge,
		displayTitle,
		displayPrice,
		displayCategories,
		displayRating,
		displayShortDescription,
		displayAddToCart,
		displayWishlist,
		displayQuickView,
		displayStock
	} = attributes;

	// Block ID
	const blockId = `botiga-tb-block-${ props.clientId.substr( 0, 8 ) }`;
	setAttributes( { blockId: blockId } );

	// Block Props
	const blockProps = useBlockProps( {
		className: classnames( 'botiga-tb-block', 'botiga-tb-sp-related-products' )
	} );

	// Tab Navigation
	const [ tab, setTab ] = useState( 'settings' );

	// Custom CSS Defaults
	const defaults = {
		columnsGap: 15,
		saleBadgeAlign: 'left',
		titleAlign: 'center',
		categoriesAlign: 'center',
		priceAlign: 'center',
		ratingAlign: 'center',
		shortDescAlign: 'center',
		addtocartAlign: 'center',
		wishlistVerticalAlign: 'top',
		stockAlign: 'center',
		alignLastBlock: false
	}

	// Replace custom CSS defaults with value if it exists
	for( const key in defaults ) {
		if ( typeof customCss[ key ] !== 'undefined' ) {
			defaults[ key ] = customCss[ key ].value;
		}
	}
	
	// Custom CSS
	generateBlockCustomCss( blockId, customCss );

	// Separator Style
	const separatorStyle = {
		fontSize: 14,
		margin: '15px 0 15px',
		borderBottom: '1px solid #e5e5e5',
		padding: '13px 0px'
	}

	return (
		<div { ...blockProps }>
		
			<InspectorControls>
				<BotigaTabsNavigation
					activeTab={ tab }
					setActiveTab={ setTab } />
					
				<BotigaTabsContent 
					tabid="settings"
					activeTab={ tab }>
					
					<PanelBody title={ __( 'General', 'botiga-pro' ) } initialOpen={ true }>
						<SelectControl
							label={ __( 'Type', 'botiga-pro' ) }
							value={ queryType }
							options={ [
								{ label: __( 'All Products' ), value: 'all-products' },
								{ label: __( 'Related Products' ), value: 'related-products' },
								{ label: __( 'Upsell Products' ), value: 'upsell-products' },
							] }
							onChange={ ( value ) => { 								
								setAttributes( { queryType: value } ) 
							} }
						/>
						{
							( queryType === 'related-products' || queryType === 'upsell-products' ) && (
								<>
									<p style={ { margin: '-15px 0 15px' } }><strong>{ __( 'Important: ', 'botiga-pro' ) }</strong>{ __( 'Related and upsell products works only inside single product pages.', 'botiga-pro' ) }</p>
								</>
							)
						}
						<SelectControl
							label={ __( 'Order By', 'botiga-pro' ) }
							value={ queryOrderBy }
							options={ [
								{ label: __( 'Date', 'botiga-pro' ), value: 'date' },
								{ label: __( 'Price', 'botiga-pro' ), value: 'price' },
								{ label: __( 'Popularity', 'botiga-pro' ), value: 'popularity' },
								{ label: __( 'Rating', 'botiga-pro' ), value: 'rating' },
								{ label: __( 'Random', 'botiga-pro' ), value: 'random' },
								{ label: __( 'Menu Order', 'botiga-pro' ), value: 'menu_order' },
							] }
							onChange={ ( value ) => { 								
								setAttributes( { queryOrderBy: value } ) 
							} }
						/>
						<SelectControl
							label={ __( 'Order', 'botiga-pro' ) }
							value={ queryOrder }
							options={ [
								{ label: __( 'ASC', 'botiga-pro' ), value: 'asc' },
								{ label: __( 'DESC', 'botiga-pro' ), value: 'desc' },
							] }
							onChange={ ( value ) => { 								
								setAttributes( { queryOrder: value } ) 
							} }
						/>
					</PanelBody>
					<PanelBody title={ __( 'Layout', 'botiga-pro' ) } initialOpen={ false }>
						<SelectControl
							label={ __( 'Columns', 'botiga-pro' ) }
							value={ columns }
							options={ [
								{ label: '1', value: '1' },
								{ label: '2', value: '2' },
								{ label: '3', value: '3' },
								{ label: '4', value: '4' },
								{ label: '5', value: '5' },
								{ label: '6', value: '6' },
							] }
							onChange={ ( value ) => { 								
								setAttributes( { columns: value } ) 
							} }
						/>
						<RangeControl
							label={ __( 'Columns Gap (px)', 'botiga-pro' ) }
							labelPosition="edge"
							value={ defaults.columnsGap }
							onChange={ ( value ) => { 
								const css = {
									columnsGap: {
										selector: `{{WRAPPER}} ul.products`,
										props: `gap: ${ value }px !important;`,
										value: value
									}
								};
								setAttributes( { customCss: { ...customCss, ...css } } ) 
							} }
							min={ 0 }
							max={ 50 }
						/>
						<SelectControl
							label={ __( 'Rows', 'botiga-pro' ) }
							value={ rows }
							options={ [
								{ label: '1', value: '1' },
								{ label: '2', value: '2' },
								{ label: '3', value: '3' },
								{ label: '4', value: '4' },
								{ label: '5', value: '5' },
								{ label: '6', value: '6' },
							] }
							onChange={ ( value ) => { 								
								setAttributes( { rows: value } ) 
							} }
						/>
						{
							queryType === 'all-products' && (
								<>
									<ToggleControl
										label={ __( 'Result Count', 'botiga-pro' ) }
										checked={ displayResultCount }
										onChange={ ( value ) => { 								
											setAttributes( { displayResultCount: value } ) 
										} }
									/>
									<ToggleControl
										label={ __( 'Ordering', 'botiga-pro' ) }
										checked={ displayOrdering }
										onChange={ ( value ) => { 								
											setAttributes( { displayOrdering: value } ) 
										} }
									/>
									<ToggleControl
										label={ __( 'Grid/List Toggle', 'botiga-pro' ) }
										checked={ displayGridListToggle }
										onChange={ ( value ) => { 								
											setAttributes( { displayGridListToggle: value } ) 
										} }
									/>
									<ToggleControl
										label={ __( 'Pagination', 'botiga-pro' ) }
										checked={ displayPagination }
										onChange={ ( value ) => { 								
											setAttributes( { displayPagination: value } ) 
										} }
									/>
									{/* {
										displayPagination && (
											<SelectControl
												label={ __( 'Pagination Type', 'botiga-pro' ) }
												value={ paginationType }
												options={ [
													{ label: __( 'Default', 'botiga-pro' ), value: 'default' },
													{ label: __( 'Load More Button', 'botiga-pro' ), value: 'loadmore' },
													{ label: __( 'Infinite Scroll', 'botiga-pro' ), value: 'infinite' }
												] }
												onChange={ ( value ) => {
													setAttributes( { paginationType: value } ) 
												} }
											/>
										)
									} */}
								</>
							)
						}
					</PanelBody>
					<PanelBody title={ __( 'Product Card', 'botiga-pro' ) } initialOpen={ false }>
						<ToggleControl
							label={ __( 'Product Image', 'botiga-pro' ) }
							checked={ displayImage }
							onChange={ ( value ) => { 								
								setAttributes( { displayImage: value } ) 
							} }
						/>
						{
							displayImage && (
								<ToggleControl
									label={ __( 'Sale Badge', 'botiga-pro' ) }
									checked={ displaySaleBadge }
									onChange={ ( value ) => { 								
										setAttributes( { displaySaleBadge: value } ) 
									} }
								/>
							)
						}
						<ToggleControl
							label={ __( 'Title', 'botiga-pro' ) }
							checked={ displayTitle }
							onChange={ ( value ) => { 								
								setAttributes( { displayTitle: value } ) 
							} }
						/>
						<ToggleControl
							label={ __( 'Price', 'botiga-pro' ) }
							checked={ displayPrice }
							onChange={ ( value ) => { 								
								setAttributes( { displayPrice: value } ) 
							} }
						/>
						<ToggleControl
							label={ __( 'Categories', 'botiga-pro' ) }
							checked={ displayCategories }
							onChange={ ( value ) => { 								
								setAttributes( { displayCategories: value } ) 
							} }
						/>
						<ToggleControl
							label={ __( 'Rating', 'botiga-pro' ) }
							checked={ displayRating }
							onChange={ ( value ) => { 								
								setAttributes( { displayRating: value } ) 
							} }
						/>
						<ToggleControl
							label={ __( 'Short Description', 'botiga-pro' ) }
							checked={ displayShortDescription }
							onChange={ ( value ) => { 								
								setAttributes( { displayShortDescription: value } ) 
							} }
						/>
						<ToggleControl
							label={ __( 'Add To Cart', 'botiga-pro' ) }
							checked={ displayAddToCart }
							onChange={ ( value ) => { 								
								setAttributes( { displayAddToCart: value } ) 
							} }
						/>
						{
							displayImage && (
								<>
									<ToggleControl
										label={ __( 'Wishlist', 'botiga-pro' ) }
										checked={ displayWishlist }
										onChange={ ( value ) => { 								
											setAttributes( { displayWishlist: value } ) 
										} }
									/>
									<ToggleControl
										label={ __( 'Quick View', 'botiga-pro' ) }
										checked={ displayQuickView }
										onChange={ ( value ) => { 								
											setAttributes( { displayQuickView: value } ) 
										} }
									/>
								</>
							)
						}
						<ToggleControl
							label={ __( 'Stock', 'botiga-pro' ) }
							checked={ displayStock }
							onChange={ ( value ) => { 								
								setAttributes( { displayStock: value } ) 
							} }
						/>
					</PanelBody>
				</BotigaTabsContent>

				<BotigaTabsContent 
					tabid="style"
					activeTab={ tab }>
					
					<PanelBody title={ __( 'Product Card', 'botiga-pro' ) } initialOpen={ true }>
						<h6 style={ separatorStyle }>{ __( 'Card', 'botiga-pro' ) }</h6>
						<BotigaColorPicker 
							label={ __( 'Background Color', 'botiga-pro' ) }
							idPrefix={ 'card_bg_color' }
							attributes={ attributes }
							setAttributes={ setAttributes }
							cssProp={ 'background-color' }
							customCssSelector={ `{{WRAPPER}} ul.products li.product` }
						/>
						<BotigaBorder 
							label={ __( 'Border', 'botiga-pro' ) }
							idPrefix={ 'card_border' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} ul.products li.product` }
						/>
						<BotigaDimensions
							label={ __( 'Border Radius', 'botiga-pro' ) }
							idPrefix={ 'card_border_radius' }
							type={ 'border-radius' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} ul.products li.product, {{WRAPPER}} ul.products li.product .loop-image-wrap img` } 
						/>
						<BotigaDimensions
							label={ __( 'Padding', 'botiga-pro' ) }
							idPrefix={ 'card_padding' }
							type={ 'padding' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} ul.products li.product` } 
						/>
						<ToggleControl
							label={ __( 'Align all products height', 'botiga-pro' ) }
							checked={ defaults.alignLastBlock }
							onChange={ ( value ) => { 		
								const val = value ? 'auto' : '0';						
								const css = {
									alignLastBlock: {
										selector: `{{WRAPPER}} ul.products li.product *:last-child:not(.product-data-wrapper)`,
										props: `margin-top: ${ val } !important;`,
										value: value
									}
								};
								setAttributes( { customCss: { ...customCss, ...css } } ) 
							} }
						/>

						{
							displayImage && displaySaleBadge && (
								<>
									<h6 style={ separatorStyle }>{ __( 'Sale Badge', 'botiga-pro' ) }</h6>
									<SelectControl
										label={ __( 'Alignment', 'botiga-pro' ) }
										value={ defaults.saleBadgeAlign }
										options={ [
											{ label: __( 'Left', 'botiga-pro' ), value: 'left' },
											{ label: __( 'Right', 'botiga-pro' ), value: 'right' },
										] }
										onChange={ ( value ) => {
											const side = value === 'left' ? 'right' : 'left';
											const css = {
												saleBadgeAlign: {
													selector: `{{WRAPPER}} ul.products li.product .onsale`,
													props: `${ side }: auto !important; ${ value }: 20px !important;`,
													value: value
												}
											};
											setAttributes( { customCss: { ...customCss, ...css } } ) 
										} }
									/>
									<BotigaColorPicker 
										label={ __( 'Background Color', 'botiga-pro' ) }
										idPrefix={ 'sale_badge_bg_color' }
										attributes={ attributes }
										setAttributes={ setAttributes }
										cssProp={ 'background-color' }
										customCssSelector={ `{{WRAPPER}} ul.products li.product .onsale` }
									/>
									<BotigaColorPicker 
										label={ __( 'Text Color', 'botiga-pro' ) }
										idPrefix={ 'sale_badge_color' }
										attributes={ attributes }
										setAttributes={ setAttributes }
										cssProp={ 'color' }
										customCssSelector={ `{{WRAPPER}} ul.products li.product .onsale` }
									/>
								</>
								
							)
						}
						{
							displayTitle && (
								<>
									<h6 style={ separatorStyle }>{ __( 'Title', 'botiga-pro' ) }</h6>
									<SelectControl
										label={ __( 'Alignment', 'botiga-pro' ) }
										idPrefix={ 'title_align' }
										value={ defaults.titleAlign }
										options={ [
											{ label: __( 'Left', 'botiga-pro' ), value: 'left' },
											{ label: __( 'Center', 'botiga-pro' ), value: 'center' },
											{ label: __( 'Right', 'botiga-pro' ), value: 'right' },
										] }
										onChange={ ( value ) => {
											const css = {
												titleAlign: {
													selector: `{{WRAPPER}} ul.products li.product .woocommerce-loop-product__title`,
													props: `text-align: ${ value } !important;`,
													value: value
												}
											};
											setAttributes( { customCss: { ...customCss, ...css } } ) 
										} }
									/>
									<BotigaTypography 
										label={ __( 'Typography', 'botiga-pro' ) }
										idPrefix={ 'title_typography' }
										attributes={ attributes } 
										setAttributes={ setAttributes }
										customCssSelector={ `{{WRAPPER}} ul.products li.product .woocommerce-loop-product__title` }
									/>
									<BotigaColorPicker 
										label={ __( 'Text Color', 'botiga-pro' ) }
										idPrefix={ 'title_color' }
										attributes={ attributes }
										setAttributes={ setAttributes }
										cssProp={ 'color' }
										customCssSelector={ `{{WRAPPER}} ul.products li.product .woocommerce-loop-product__title` }
									/>
								</>
							)
						}
						{
							displayCategories && (
								<>
									<h6 style={ separatorStyle }>{ __( 'Categories', 'botiga-pro' ) }</h6>
									<SelectControl
										label={ __( 'Alignment', 'botiga-pro' ) }
										idPrefix={ 'categories_align' }
										value={ defaults.categoriesAlign }
										options={ [
											{ label: __( 'Left', 'botiga-pro' ), value: 'left' },
											{ label: __( 'Center', 'botiga-pro' ), value: 'center' },
											{ label: __( 'Right', 'botiga-pro' ), value: 'right' },
										] }
										onChange={ ( value ) => {
											const css = {
												categoriesAlign: {
													selector: `{{WRAPPER}} ul.products li.product .product-category`,
													props: `text-align: ${ value } !important;`,
													value: value
												}
											};
											setAttributes( { customCss: { ...customCss, ...css } } ) 
										} }
									/>
									<BotigaTypography 
										label={ __( 'Typography', 'botiga-pro' ) }
										idPrefix={ 'categories_typography' }
										attributes={ attributes } 
										setAttributes={ setAttributes }
										customCssSelector={ `{{WRAPPER}} ul.products li.product .product-category, {{WRAPPER}} ul.products li.product .product-category a` }
									/>
									<BotigaColorPicker 
										label={ __( 'Text Color', 'botiga-pro' ) }
										idPrefix={ 'categories_color' }
										withHover={ true }
										attributes={ attributes }
										setAttributes={ setAttributes }
										cssProp={ 'color' }
										customCssSelector={ `{{WRAPPER}} ul.products li.product .product-category, {{WRAPPER}} ul.products li.product .product-category a` }
									/>
								</>
							)
						}
						{
							displayPrice && (
								<>
									<h6 style={ separatorStyle }>{ __( 'Price', 'botiga-pro' ) }</h6>
									<SelectControl
										label={ __( 'Alignment', 'botiga-pro' ) }
										idPrefix={ 'price_align' }
										value={ defaults.priceAlign }
										options={ [
											{ label: __( 'Left', 'botiga-pro' ), value: 'left' },
											{ label: __( 'Center', 'botiga-pro' ), value: 'center' },
											{ label: __( 'Right', 'botiga-pro' ), value: 'right' },
										] }
										onChange={ ( value ) => {
											const css = {
												priceAlign: {
													selector: `{{WRAPPER}} ul.products li.product .price`,
													props: `text-align: ${ value } !important;`,
													value: value
												}
											};
											setAttributes( { customCss: { ...customCss, ...css } } ) 
										} }
									/>
									<BotigaTypography 
										label={ __( 'Typography', 'botiga-pro' ) }
										idPrefix={ 'price_typography' }
										attributes={ attributes } 
										setAttributes={ setAttributes }
										customCssSelector={ `{{WRAPPER}} ul.products li.product .price` }
									/>
									<BotigaColorPicker 
										label={ __( 'Text Color', 'botiga-pro' ) }
										idPrefix={ 'price_color' }
										withHover={ false }
										attributes={ attributes }
										setAttributes={ setAttributes }
										cssProp={ 'color' }
										customCssSelector={ `{{WRAPPER}} ul.products li.product .price .woocommerce-Price-amount` }
									/>
									<BotigaColorPicker 
										label={ __( 'Text Color (sale)', 'botiga-pro' ) }
										idPrefix={ 'price_sale_color' }
										withHover={ false }
										attributes={ attributes }
										setAttributes={ setAttributes }
										cssProp={ 'color' }
										customCssSelector={ `{{WRAPPER}} ul.products li.product .price del .woocommerce-Price-amount` }
									/>
								</>
							)
						}
						{
							displayRating && (
								<>
									<h6 style={ separatorStyle }>{ __( 'Rating', 'botiga-pro' ) }</h6>
									<SelectControl
										label={ __( 'Alignment', 'botiga-pro' ) }
										idPrefix={ 'price_align' }
										value={ defaults.ratingAlign }
										options={ [
											{ label: __( 'Left', 'botiga-pro' ), value: 'left' },
											{ label: __( 'Center', 'botiga-pro' ), value: 'center' },
											{ label: __( 'Right', 'botiga-pro' ), value: 'right' },
										] }
										onChange={ ( value ) => {
											if( value === 'center' ) {
												const css = {
													ratingAlign: {
														selector: `{{WRAPPER}} ul.products li.product .star-rating`,
														props: `margin: 0 auto 10px !important;`,
														value: value
													}
												};
												setAttributes( { customCss: { ...customCss, ...css } } ) 
											} else {
												const side = value === 'left' ? 'right' : 'left';
	
												const css = {
													ratingAlign: {
														selector: `{{WRAPPER}} ul.products li.product .star-rating`,
														props: `margin: 0 0 10px !important; margin-${ side }: auto !important;`,
														value: value
													}
												};
												setAttributes( { customCss: { ...customCss, ...css } } ) 
											}

										} }
									/>
									<BotigaColorPicker 
										label={ __( 'Icon Color', 'botiga-pro' ) }
										idPrefix={ 'rating_color' }
										withHover={ false }
										attributes={ attributes }
										setAttributes={ setAttributes }
										cssProp={ 'color' }
										customCssSelector={ `{{WRAPPER}} ul.products li.product .star-rating span::before` }
									/>
								</>
							)
						}
						{
							displayShortDescription && (
								<>
									<h6 style={ separatorStyle }>{ __( 'Short Description', 'botiga-pro' ) }</h6>
									<SelectControl
										label={ __( 'Alignment', 'botiga-pro' ) }
										idPrefix={ 'short_desc_align' }
										value={ defaults.shortDescAlign }
										options={ [
											{ label: __( 'Left', 'botiga-pro' ), value: 'left' },
											{ label: __( 'Center', 'botiga-pro' ), value: 'center' },
											{ label: __( 'Right', 'botiga-pro' ), value: 'right' },
										] }
										onChange={ ( value ) => {
											const css = {
												shortDescAlign: {
													selector: `{{WRAPPER}} ul.products li.product .product-description`,
													props: `text-align: ${ value } !important;`,
													value: value
												}
											};
											setAttributes( { customCss: { ...customCss, ...css } } ) 
										} }
									/>
									<BotigaTypography 
										label={ __( 'Typography', 'botiga-pro' ) }
										idPrefix={ 'short_desc_typography' }
										attributes={ attributes } 
										setAttributes={ setAttributes }
										customCssSelector={ `{{WRAPPER}} ul.products li.product .product-description` }
									/>
									<BotigaColorPicker 
										label={ __( 'Text Color', 'botiga-pro' ) }
										idPrefix={ 'short_desc_color' }
										withHover={ false }
										attributes={ attributes }
										setAttributes={ setAttributes }
										cssProp={ 'color' }
										customCssSelector={ `{{WRAPPER}} ul.products li.product .product-description` }
									/>
								</>
							)
						}
						{
							displayAddToCart && (
								<>
									<h6 style={ separatorStyle }>{ __( 'Add To Cart', 'botiga-pro' ) }</h6>
									<SelectControl
										label={ __( 'Alignment', 'botiga-pro' ) }
										idPrefix={ 'addtocart_align' }
										value={ defaults.addtocartAlign }
										options={ [
											{ label: __( 'Left', 'botiga-pro' ), value: 'left' },
											{ label: __( 'Center', 'botiga-pro' ), value: 'center' },
											{ label: __( 'Right', 'botiga-pro' ), value: 'right' },
										] }
										onChange={ ( value ) => {
											if( value === 'center' ) {
												const css = {
													addtocartAlign: {
														selector: `{{WRAPPER}} ul.products li.product .add_to_cart_button, {{WRAPPER}} ul.products li.product .product_type_variation, {{WRAPPER}} ul.products li.product .product_type_grouped`,
														props: `margin: 0 auto !important;`,
														value: value
													},
													addtocartAlignExtra: {
														selector: `{{WRAPPER}} ul.products li.product .added_to_cart`,
														props: `text-align: center !important; margin-top: 10px; display: block; width: 100%;`,
														value: value
													}
												};
												setAttributes( { customCss: { ...customCss, ...css } } )
											} else {
												const side = value === 'left' ? 'right' : 'left';
	
												const css = {
													addtocartAlign: {
														selector: `{{WRAPPER}} ul.products li.product .add_to_cart_button, {{WRAPPER}} ul.products li.product .product_type_variation, {{WRAPPER}} ul.products li.product .product_type_grouped`,
														props: `margin-${ value }: 0 !important; margin-${ side }: auto !important`,
														value: value
													},
													addtocartAlignExtra: {
														selector: `{{WRAPPER}} ul.products li.product .added_to_cart`,
														props: `text-align: ${ value } !important; margin-top: 10px; display: block; width: 100%;`,
														value: value
													}
												};
												setAttributes( { customCss: { ...customCss, ...css } } ) 
											}
										} }
									/>
									<BotigaTypography 
										label={ __( 'Typography', 'botiga-pro' ) }
										idPrefix={ 'addtocart_typography' }
										attributes={ attributes } 
										setAttributes={ setAttributes }
										customCssSelector={ `{{WRAPPER}} ul.products li.product .add_to_cart_button` }
									/>
									<BotigaColorPicker 
										label={ __( 'Background Color', 'botiga-pro' ) }
										idPrefix={ 'addtocart_background_color' }
										withHover={ true }
										attributes={ attributes }
										setAttributes={ setAttributes }
										cssProp={ 'background-color' }
										customCssSelector={ `{{WRAPPER}} ul.products li.product .add_to_cart_button` }
									/>
									<BotigaColorPicker 
										label={ __( 'Text Color', 'botiga-pro' ) }
										idPrefix={ 'addtocart_color' }
										withHover={ true }
										attributes={ attributes }
										setAttributes={ setAttributes }
										cssProp={ 'color' }
										customCssSelector={ `{{WRAPPER}} ul.products li.product .add_to_cart_button` }
									/>
									<BotigaColorPicker 
										label={ __( 'Border Color', 'botiga-pro' ) }
										idPrefix={ 'addtocart_border_color' }
										withHover={ true }
										attributes={ attributes }
										setAttributes={ setAttributes }
										cssProp={ 'border-color' }
										customCssSelector={ `{{WRAPPER}} ul.products li.product .add_to_cart_button` }
									/>
								</>
							)
						}
						{
							displayWishlist && (
								<>
									<h6 style={ separatorStyle }>{ __( 'Wishlist', 'botiga-pro' ) }</h6>
									<SelectControl
										label={ __( 'Alignment', 'botiga-pro' ) }
										idPrefix={ 'wishlist_vertical_align' }
										value={ defaults.wishlistVerticalAlign }
										options={ [
											{ label: __( 'Top', 'botiga-pro' ), value: 'top' },
											{ label: __( 'Bottom', 'botiga-pro' ), value: 'bottom' },
										] }
										onChange={ ( value ) => {

											let css = {
												wishlistVerticalAlign: {
													selector: `{{WRAPPER}} ul.products li.product .botiga-wishlist-button`,
													props: `bottom: 15px !important; top: auto !important;`,
													value: value
												}
											};

											if( value === 'top' ) {
												css = {};
											}

											setAttributes( { customCss: { ...customCss, ...css } } ) 
										} }
									/>
									<BotigaTypography 
										label={ __( 'Typography', 'botiga-pro' ) }
										idPrefix={ 'wishlist_typography' }
										attributes={ attributes } 
										setAttributes={ setAttributes }
										customCssSelector={ `{{WRAPPER}} ul.products li.product .botiga-wishlist-button.botiga-wishlist-button-tooltip:before` }
									/>
									<BotigaColorPicker 
										label={ __( 'Icon Color', 'botiga-pro' ) }
										idPrefix={ 'wishlist_icon_color' }
										withHover={ false }
										attributes={ attributes }
										setAttributes={ setAttributes }
										cssProp={ 'stroke' }
										customCssSelector={ `{{WRAPPER}} ul.products li.product .botiga-wishlist-button svg path` }
									/>
									<BotigaColorPicker 
										label={ __( 'Icon Color (hover/active)', 'botiga-pro' ) }
										idPrefix={ 'wishlist_icon_color_active' }
										withHover={ false }
										attributes={ attributes }
										setAttributes={ setAttributes }
										cssProp={ [ 'fill', 'stroke' ] }
										customCssSelector={ `{{WRAPPER}} ul.products li.product .botiga-wishlist-button:hover svg path, {{WRAPPER}} ul.products li.product .botiga-wishlist-button.active svg path` }
									/>
									<BotigaColorPicker 
										label={ __( 'Tooltip Background Color', 'botiga-pro' ) }
										idPrefix={ 'wishlist_tooltip_background_color' }
										withHover={ false }
										attributes={ attributes }
										setAttributes={ setAttributes }
										cssProp={ [ 'background-color', 'border-left-color' ] }
										customCssSelector={ `{{WRAPPER}} ul.products li.product .botiga-wishlist-button.botiga-wishlist-button-tooltip:before, {{WRAPPER}} ul.products li.product .botiga-wishlist-button.botiga-wishlist-button-tooltip:after` }
									/>
									<BotigaColorPicker 
										label={ __( 'Tooltip Text Color', 'botiga-pro' ) }
										idPrefix={ 'wishlist_tooltip_text_color' }
										withHover={ false }
										attributes={ attributes }
										setAttributes={ setAttributes }
										cssProp={ 'color' }
										customCssSelector={ `{{WRAPPER}} ul.products li.product .botiga-wishlist-button.botiga-wishlist-button-tooltip:before` }
									/>
								</>
							)
						}
						{
							displayQuickView && (
								<>
									<h6 style={ separatorStyle }>{ __( 'Quick View', 'botiga-pro' ) }</h6>
									<BotigaTypography 
										label={ __( 'Typography', 'botiga-pro' ) }
										idPrefix={ 'quickview_typography' }
										attributes={ attributes } 
										setAttributes={ setAttributes }
										customCssSelector={ `{{WRAPPER}} ul.products li.product .botiga-quick-view` }
									/>
									<BotigaColorPicker 
										label={ __( 'Background Color', 'botiga-pro' ) }
										idPrefix={ 'quickview_background_color' }
										withHover={ true }
										attributes={ attributes }
										setAttributes={ setAttributes }
										cssProp={ 'background-color' }
										customCssSelector={ `{{WRAPPER}} ul.products li.product .botiga-quick-view` }
									/>
									<BotigaColorPicker 
										label={ __( 'Text Color', 'botiga-pro' ) }
										idPrefix={ 'quickview_color' }
										withHover={ true }
										attributes={ attributes }
										setAttributes={ setAttributes }
										cssProp={ 'color' }
										customCssSelector={ `{{WRAPPER}} ul.products li.product .botiga-quick-view` }
									/>
									<BotigaColorPicker 
										label={ __( 'Border Color', 'botiga-pro' ) }
										idPrefix={ 'quickview_border_color' }
										withHover={ true }
										attributes={ attributes }
										setAttributes={ setAttributes }
										cssProp={ 'border-color' }
										customCssSelector={ `{{WRAPPER}} ul.products li.product .botiga-quick-view` }
									/>
								</>
							)
						}
						{
							displayStock && (
								<>
									<h6 style={ separatorStyle }>{ __( 'Stock', 'botiga-pro' ) }</h6>
									<SelectControl
										label={ __( 'Alignment', 'botiga-pro' ) }
										idPrefix={ 'stock_align' }
										value={ defaults.stockAlign }
										options={ [
											{ label: __( 'Left', 'botiga-pro' ), value: 'left' },
											{ label: __( 'Center', 'botiga-pro' ), value: 'center' },
											{ label: __( 'Right', 'botiga-pro' ), value: 'right' },
										] }
										onChange={ ( value ) => {
											const css = {
												stockAlign: {
													selector: `{{WRAPPER}} ul.products li.product .botiga-product-stock`,
													props: `text-align: ${ value } !important;`,
													value: value
												}
											};
											setAttributes( { customCss: { ...customCss, ...css } } ) 
										} }
									/>
									<BotigaTypography 
										label={ __( 'Typography', 'botiga-pro' ) }
										idPrefix={ 'stock_typography' }
										attributes={ attributes } 
										setAttributes={ setAttributes }
										customCssSelector={ `{{WRAPPER}} ul.products li.product .botiga-product-stock` }
									/>
									<BotigaColorPicker 
										label={ __( 'Text Color', 'botiga-pro' ) }
										idPrefix={ 'stock_color' }
										withHover={ false }
										attributes={ attributes }
										setAttributes={ setAttributes }
										cssProp={ 'color' }
										customCssSelector={ `{{WRAPPER}} ul.products li.product .botiga-product-stock` }
									/>
								</>
							)
						}
					</PanelBody>
				</BotigaTabsContent>
				<BotigaTabsContent 
					tabid="advanced"
					activeTab={ tab }>
					
					<PanelBody title={ __( 'Layout', 'botiga-pro' ) } initialOpen={ true }>
						<BotigaDimensions
							label={ __( 'Margin', 'botiga-pro' ) }
							idPrefix={ '' }
							type={ 'margin' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}}` } />

						<BotigaDimensions
							label={ __( 'Padding', 'botiga-pro' ) }
							idPrefix={ '' }
							type={ 'padding' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}}` } />
					</PanelBody>
				</BotigaTabsContent>
			</InspectorControls>

			<Disabled>
				<ServerSideRender
					block={ name }
					httpMethod="POST"
					attributes={ attributes }
				/>
			</Disabled>

		</div>
	);
}
