import classnames from 'classnames';
import { __ } from '@wordpress/i18n';
import { useState } from '@wordpress/element';
import ServerSideRender from '@wordpress/server-side-render';
import { PanelBody, SelectControl, __experimentalBoxControl as BoxControl, Disabled } from '@wordpress/components';
import { BotigaDimensions, BotigaTypography, BotigaColorPicker, BotigaTabsNavigation, BotigaTabsContent } from '../_inc/global-components';
import { InspectorControls, useBlockProps } from '@wordpress/block-editor';
import { generateBlockCustomCss } from '../_inc/helpers';

/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {WPElement} Element to render.
 */
export default function Edit( props ) {
	const { name, attributes, setAttributes } = props;

	const { 
		htmlTag,
		customCss
	} = attributes;

	// Block ID
	const blockId = `botiga-tb-block-${ props.clientId.substr( 0, 8 ) }`;
	setAttributes( { blockId: blockId } );

	// Block Props
	const blockProps = useBlockProps( {
		className: classnames( blockId, 'botiga-tb-block', 'botiga-tb-sp-title' )
	} );

	// Tab Navigation
	const [ tab, setTab ] = useState( 'settings' );

	// HTML Tag
	const HtmlTagOutput = htmlTag ? htmlTag : 'h1';

	// Defaults
	const defaults = {
		align: 'left'
	}

	if( typeof customCss.align !== 'undefined' ) {
        defaults.align = customCss.align.value;
    }

	// Custom CSS
	generateBlockCustomCss( blockId, customCss );

	return (
		<div { ...blockProps }>
			<InspectorControls>
				<BotigaTabsNavigation
					activeTab={ tab }
					setActiveTab={ setTab } />

				<BotigaTabsContent 
					tabid="settings"
					activeTab={ tab }>
					
					<PanelBody title={ __( 'General', 'botiga-pro' ) } initialOpen={ true }>
						<SelectControl
							label={ __( 'HTML Tag', 'botiga-pro' ) }
							value={ htmlTag }
							options={ [
								{ label: 'h1', value: 'h1' },
								{ label: 'h2', value: 'h2' },
								{ label: 'h3', value: 'h3' },
								{ label: 'h4', value: 'h4' },
								{ label: 'h5', value: 'h5' },
								{ label: 'h6', value: 'h6' },
							] }
							onChange={ ( value ) => { 
								setAttributes( { htmlTag: value } ) 
							} }
						/>
						<SelectControl
							label={ __( 'Alignment', 'botiga-pro' ) }
							value={ defaults.align }
							options={ [
								{ label: 'Left', value: 'left' },
								{ label: 'Center', value: 'center' },
								{ label: 'Right', value: 'right' },
							] }
							onChange={ ( value ) => { 
								const css = {
									align: {
										selector: `{{WRAPPER}} ${ htmlTag }`,
										props: `text-align: ${ value };`,
										value: value
									}
								};
								setAttributes( { customCss: { ...customCss, ...css } } ) 
							} }
						/>
					</PanelBody>

				</BotigaTabsContent>
				<BotigaTabsContent 
					tabid="style"
					activeTab={ tab }>
					
					<PanelBody title={ __( 'Title', 'botiga-pro' ) } initialOpen={ true }>
						<BotigaTypography 
							label={ __( 'Typography', 'botiga-pro' ) }
							idPrefix={ 'title' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} ${ htmlTag }` }
						/>
						<BotigaColorPicker
							label={ __( 'Text Color', 'botiga-pro' ) }
							idPrefix={ 'title' }
							withHover={ false }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							customCssSelector={ `{{WRAPPER}} ${ htmlTag }` }
						/>
					</PanelBody>

				</BotigaTabsContent>
				<BotigaTabsContent 
					tabid="advanced"
					activeTab={ tab }>
					
					<PanelBody title={ __( 'Layout', 'botiga-pro' ) } initialOpen={ true }>
						<BotigaDimensions
							label={ __( 'Margin', 'botiga-pro' ) }
							idPrefix={ 'title' }
							type={ 'margin' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} ${ htmlTag }` } />

						<BotigaDimensions
							label={ __( 'Padding', 'botiga-pro' ) }
							idPrefix={ 'title' }
							type={ 'padding' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} ${ htmlTag }` } />
					</PanelBody>

				</BotigaTabsContent>
			</InspectorControls>

			<Disabled>
				<ServerSideRender
					block={ name }
					httpMethod="POST"
					attributes={ attributes }
				/>
			</Disabled>
		</div>
	);
}
