import classnames from 'classnames';
import { __ } from '@wordpress/i18n';
import { useState } from '@wordpress/element';
import ServerSideRender from '@wordpress/server-side-render';
import { PanelBody, SelectControl, Disabled } from '@wordpress/components';
import { BotigaDimensions, BotigaTypography, BotigaColorPicker, BotigaTabsNavigation, BotigaTabsContent, BotigaBorder } from '../_inc/global-components';
import { InspectorControls, useBlockProps } from '@wordpress/block-editor';
import './editor.scss';
import { generateBlockCustomCss } from '../_inc/helpers';

/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {WPElement} Element to render.
 */
export default function Edit( props ) {
	const { name, attributes, setAttributes } = props;

	const { 
		customCss
	} = attributes;

	// Block ID
	const blockId = `botiga-tb-block-${ props.clientId.substr( 0, 8 ) }`;
	setAttributes( { blockId: blockId } );

	// Block Props
	const blockProps = useBlockProps( {
		className: classnames( blockId, 'botiga-tb-block', 'botiga-tb-sp-tabs' )
	} );

	// Tab Navigation
	const [ tab, setTab ] = useState( 'style' );

	// Defaults
	const defaults = {
		alignTabsNav: 'left',
		alignTabsContent: 'left'
	}

	if( typeof customCss.alignTabsNav !== 'undefined' ) {
        defaults.alignTabsNav = customCss.alignTabsNav.value;
    }

	if( typeof customCss.alignTabsContent !== 'undefined' ) {
        defaults.alignTabsContent = customCss.alignTabsContent.value;
    }

	// Custom CSS
	generateBlockCustomCss( blockId, customCss );

	// Separator Style
	const separatorStyle = {
		fontSize: 14,
		margin: '15px 0 15px',
		borderBottom: '1px solid #e5e5e5',
		padding: '13px 0px'
	}

	return (
		<div { ...blockProps }>
		
			<InspectorControls>
				<BotigaTabsNavigation
					exclude="settings"
					activeTab={ tab }
					setActiveTab={ setTab } />

				<BotigaTabsContent 
					tabid="style"
					activeTab={ tab }>
					
					<PanelBody title={ __( 'Tabs Navigation', 'botiga-pro' ) } initialOpen={ true }>
						<BotigaTypography 
							label={ __( 'Typography', 'botiga-pro' ) }
							idPrefix={ 'tab_nav_typography' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .woocommerce-tabs ul.tabs li a` }
						/>
						<SelectControl
							label={ __( 'Alignment', 'botiga-pro' ) }
							value={ defaults.alignTabsNav }
							options={ [
								{ label: 'Left', value: 'left' },
								{ label: 'Center', value: 'center' },
								{ label: 'Right', value: 'right' },
							] }
							onChange={ ( value ) => { 
								const css = {
									alignTabsNav: {
										selector: `{{WRAPPER}} .woocommerce-tabs ul.tabs`,
										props: `text-align: ${ value } !important;`,
										value: value
									}
								};
								setAttributes( { customCss: { ...customCss, ...css } } ) 
							} }
						/>
						<BotigaColorPicker 
							label={ __( 'Link Color', 'botiga-pro' ) }
							idPrefix={ 'tab_nav_link_color' }
							withHover={ false }
							attributes={ attributes }
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .woocommerce-tabs ul.tabs li:not(.active) a` }
						/>
						<BotigaColorPicker 
							label={ __( 'Link Color (hover)', 'botiga-pro' ) }
							idPrefix={ 'tab_nav_link_color_hover' }
							withHover={ false }
							attributes={ attributes }
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .woocommerce-tabs ul.tabs li a:hover, {{WRAPPER}} .woocommerce-tabs ul.tabs li.active a` }
						/>
						<BotigaColorPicker 
							label={ __( 'Border Color (active)', 'botiga-pro' ) }
							idPrefix={ 'tab_nav_active_border_color' }
							withHover={ false }
							attributes={ attributes }
							setAttributes={ setAttributes }
							cssProp={ `border-color` }
							customCssSelector={ `{{WRAPPER}} .woocommerce-tabs ul.tabs li.active a` }
						/>
						<BotigaColorPicker 
							label={ __( 'Divider Color', 'botiga-pro' ) }
							idPrefix={ 'tab_nav_divider_color' }
							withHover={ false }
							attributes={ attributes }
							setAttributes={ setAttributes }
							cssProp={ `border-bottom-color` }
							customCssSelector={ `{{WRAPPER}} .woocommerce-tabs ul.tabs` }
						/>						
					</PanelBody>
					<PanelBody title={ __( 'Description', 'botiga-pro' ) } initialOpen={ false }>
						<BotigaTypography 
							label={ __( 'Typography', 'botiga-pro' ) }
							idPrefix={ 'tab_content_desc_typography' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .woocommerce-Tabs-panel--description` }
						/>
						<SelectControl
							label={ __( 'Alignment', 'botiga-pro' ) }
							value={ defaults.alignTabsContent }
							options={ [
								{ label: 'Left', value: 'left' },
								{ label: 'Center', value: 'center' },
								{ label: 'Right', value: 'right' },
							] }
							onChange={ ( value ) => { 
								const css = {
									alignTabsContent: {
										selector: `{{WRAPPER}} .woocommerce-Tabs-panel--description`,
										props: `text-align: ${ value } !important;`,
										value: value
									}
								};
								setAttributes( { customCss: { ...customCss, ...css } } ) 
							} }
						/>
						<BotigaColorPicker 
							label={ __( 'Text Color', 'botiga-pro' ) }
							idPrefix={ 'tab_content_desc_text_color' }
							withHover={ false }
							attributes={ attributes }
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .woocommerce-Tabs-panel--description` }
						/>
						<BotigaColorPicker 
							label={ __( 'Link Color', 'botiga-pro' ) }
							idPrefix={ 'tab_content_desc_link_color' }
							withHover={ true }
							attributes={ attributes }
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .woocommerce-Tabs-panel--description a` }
						/>
					</PanelBody>
					<PanelBody title={ __( 'Additional Information', 'botiga-pro' ) } initialOpen={ false }>
						<BotigaTypography 
							label={ __( 'Typography', 'botiga-pro' ) }
							idPrefix={ 'tab_content_addinfo_typography' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .woocommerce-Tabs-panel--additional_information .woocommerce-product-attributes-item__label, {{WRAPPER}} .woocommerce-Tabs-panel--additional_information .woocommerce-product-attributes-item__value` }
						/>
						<BotigaColorPicker 
							label={ __( 'Text Color', 'botiga-pro' ) }
							idPrefix={ 'tab_content_addinfo_color' }
							attributes={ attributes }
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .woocommerce-Tabs-panel--additional_information .woocommerce-product-attributes-item__label, {{WRAPPER}} .woocommerce-Tabs-panel--additional_information .woocommerce-product-attributes-item__value` }
						/>
						<BotigaColorPicker 
							label={ __( 'Link Color', 'botiga-pro' ) }
							idPrefix={ 'tab_content_addinfo_link_color' }
							withHover={ true }
							attributes={ attributes }
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .woocommerce-Tabs-panel--additional_information .woocommerce-product-attributes-item__value a` }
						/>
						<BotigaBorder 
							label={ __( 'Border', 'botiga-pro' ) }
							idPrefix={ 'tab_content_addinfo_table_border' }
							attributes={ attributes }
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .woocommerce-Tabs-panel--additional_information table.woocommerce-product-attributes.shop_attributes` }
						/>
						<BotigaColorPicker 
							label={ __( 'Background Color', 'botiga-pro' ) }
							idPrefix={ 'tab_content_addinfo_table_background_color' }
							withHover={ false }
							attributes={ attributes }
							setAttributes={ setAttributes }
							cssProp="background-color"
							customCssSelector={ `{{WRAPPER}} .woocommerce-Tabs-panel--additional_information table.woocommerce-product-attributes.shop_attributes` }
						/>
					</PanelBody>
					<PanelBody title={ __( 'Reviews', 'botiga-pro' ) } initialOpen={ false }>
						<h6 style={ separatorStyle }>{ __( 'Author', 'botiga-pro' ) }</h6>
						<BotigaDimensions
							label={ __( 'Image Border Radius', 'botiga-pro' ) }
							idPrefix={ 'author_image_border_radius' }
							type={ 'border-radius' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .avatar` } 
						/>
						<BotigaColorPicker
							label={ __( 'Rating Icon Background Color', 'botiga-pro' ) }
							idPrefix={ 'author_rating_icon_bg' }
							withHover={ false }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							customCssSelector={ `{{WRAPPER}} .star-rating::before` }
						/>
						<BotigaColorPicker
							label={ __( 'Rating Icon Color', 'botiga-pro' ) }
							idPrefix={ 'author_rating_icon_color' }
							withHover={ false }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							customCssSelector={ `{{WRAPPER}} .star-rating span::before` }
						/>
						<BotigaTypography 
							label={ __( 'Typography', 'botiga-pro' ) }
							idPrefix={ 'author_text_typography' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .meta` }
						/>
						<BotigaColorPicker
							label={ __( 'Name Color', 'botiga-pro' ) }
							idPrefix={ 'author_name_color' }
							withHover={ false }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							customCssSelector={ `{{WRAPPER}} .woocommerce-review__author` }
						/>
						<BotigaColorPicker
							label={ __( 'Date Color', 'botiga-pro' ) }
							idPrefix={ 'author_date_color' }
							withHover={ false }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							customCssSelector={ `{{WRAPPER}} .woocommerce-review__dash, {{WRAPPER}} .woocommerce-review__published-date` }
						/>

						<h6 style={ separatorStyle }>{ __( 'Review Text', 'botiga-pro' ) }</h6>
						<BotigaTypography 
							label={ __( 'Typography', 'botiga-pro' ) }
							idPrefix={ 'review_text_typography' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .description` }
						/>
						<BotigaColorPicker
							label={ __( 'Text Color', 'botiga-pro' ) }
							idPrefix={ 'review_text_color' }
							withHover={ false }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							customCssSelector={ `{{WRAPPER}} .description` }
						/>
						<BotigaColorPicker
							label={ __( 'Background Color', 'botiga-pro' ) }
							idPrefix={ 'review_background_color' }
							withHover={ false }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							cssProp={ `background-color` }
							customCssSelector={ `{{WRAPPER}} .description` }
						/>

						<h6 style={ separatorStyle }>{ __( 'Review Box', 'botiga-pro' ) }</h6>
						<BotigaColorPicker
							label={ __( 'Background Color', 'botiga-pro' ) }
							idPrefix={ 'review_box_background_color' }
							withHover={ false }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							cssProp={ `background-color` }
							customCssSelector={ `{{WRAPPER}} #review_form_wrapper` }
						/>
						<BotigaTypography 
							label={ __( 'Title Typography', 'botiga-pro' ) }
							idPrefix={ 'review_box_title_typography' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} #reply-title` }
						/>
						<BotigaColorPicker
							label={ __( 'Title Color', 'botiga-pro' ) }
							idPrefix={ 'review_box_title_color' }
							withHover={ false }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							cssProp={ `color` }
							customCssSelector={ `{{WRAPPER}} #reply-title` }
						/>
						<BotigaColorPicker
							label={ __( 'Rating Text Color', 'botiga-pro' ) }
							idPrefix={ 'review_box_rating_text_color' }
							withHover={ false }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							cssProp={ `color` }
							customCssSelector={ `{{WRAPPER}} .comment-form-rating > label` }
						/>
						<BotigaColorPicker
							label={ __( 'Rating Icon Background Color', 'botiga-pro' ) }
							idPrefix={ 'review_box_rating_icon_bg' }
							withHover={ false }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							customCssSelector={ `{{WRAPPER}} p.stars a::before` }
						/>
						<BotigaColorPicker
							label={ __( 'Rating Icon Color', 'botiga-pro' ) }
							idPrefix={ 'review_box_rating_icon_color' }
							withHover={ false }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							customCssSelector={ `{{WRAPPER}} p.stars.selected a:not(.active)::before, {{WRAPPER}} p.stars.selected a.active::before` }
						/>

						<h6 style={ separatorStyle }>{ __( 'Review Form', 'botiga-pro' ) }</h6>
						<BotigaTypography 
							label={ __( 'Title Typography', 'botiga-pro' ) }
							idPrefix={ 'review_form_title_typography' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .comment-form-comment > label` }
						/>
						<BotigaColorPicker
							label={ __( 'Title Color', 'botiga-pro' ) }
							idPrefix={ 'review_form_title_color' }
							withHover={ false }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							customCssSelector={ `{{WRAPPER}} .comment-form-comment > label` }
						/>
						<BotigaTypography 
							label={ __( 'Textarea Typography', 'botiga-pro' ) }
							idPrefix={ 'review_form_textarea_typography' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} #comment` }
						/>
						<BotigaColorPicker
							label={ __( 'Textarea Color', 'botiga-pro' ) }
							idPrefix={ 'review_form_textarea_color' }
							withHover={ false }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							customCssSelector={ `{{WRAPPER}} #comment` }
						/>
						<BotigaColorPicker
							label={ __( 'Textarea Background Color', 'botiga-pro' ) }
							idPrefix={ 'review_form_textarea_background_color' }
							withHover={ false }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							cssProp={ `background-color` }
							customCssSelector={ `{{WRAPPER}} #comment` }
						/>
						<BotigaColorPicker
							label={ __( 'Textarea Border Color', 'botiga-pro' ) }
							idPrefix={ 'review_form_textarea_border_color' }
							withHover={ false }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							cssProp={ `border-color` }
							customCssSelector={ `{{WRAPPER}} #comment` }
						/>
						<BotigaTypography 
							label={ __( 'Button Typography', 'botiga-pro' ) }
							idPrefix={ 'review_form_button_typography' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} #submit` }
						/>
						<BotigaColorPicker
							label={ __( 'Button Text Color', 'botiga-pro' ) }
							idPrefix={ 'review_form_button_color' }
							withHover={ true }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							cssProp={ `color` }
							customCssSelector={ `{{WRAPPER}} #submit` }
						/>
						<BotigaColorPicker
							label={ __( 'Button Background Color', 'botiga-pro' ) }
							idPrefix={ 'review_form_button_background_color' }
							withHover={ true }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							cssProp={ `background-color` }
							customCssSelector={ `{{WRAPPER}} #submit` }
						/>
						<BotigaBorder 
							label={ __( 'Button Border', 'botiga-pro' ) }
							idPrefix={ 'review_form_button_border' }
							defaults={{
								borderType: 'none',
								borderWidth: {
									top: 0,
									left: 0,
									right: 0,
									bottom: 0
								},
								borderColor: '#CCC'
							}}
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} #submit` }
						/>
						<BotigaBorder 
							label={ __( 'Button Border (hover)', 'botiga-pro' ) }
							idPrefix={ 'review_form_button_border_hover' }
							defaults={{
								borderType: 'none',
								borderWidth: {
									top: 0,
									left: 0,
									right: 0,
									bottom: 0
								},
								borderColor: '#CCC'
							}}
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} #submit:hover` }
						/>
						<BotigaDimensions
							label={ __( 'Button Padding', 'botiga-pro' ) }
							idPrefix={ 'review_form_button_padding' }
							type={ 'padding' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} #submit` } 
						/>

					</PanelBody>

				</BotigaTabsContent>
				<BotigaTabsContent 
					tabid="advanced"
					activeTab={ tab }>
					
					<PanelBody title={ __( 'Layout', 'botiga-pro' ) } initialOpen={ true }>
						<BotigaDimensions
							label={ __( 'Margin', 'botiga-pro' ) }
							idPrefix={ '' }
							type={ 'margin' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}}` } />

						<BotigaDimensions
							label={ __( 'Padding', 'botiga-pro' ) }
							idPrefix={ '' }
							type={ 'padding' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}}` } />
					</PanelBody>

				</BotigaTabsContent>
			</InspectorControls>

			<Disabled>
				<ServerSideRender
					block={ name }
					httpMethod="POST"
					attributes={ attributes }
				/>
			</Disabled>

		</div>
	);
}
