import classnames from 'classnames';
import { __ } from '@wordpress/i18n';
import { useState } from '@wordpress/element';
import ServerSideRender from '@wordpress/server-side-render';
import { PanelBody, SelectControl, RangeControl, Disabled } from '@wordpress/components';
import { BotigaDimensions, BotigaTypography, BotigaColorPicker, BotigaTabsNavigation, BotigaTabsContent } from '../_inc/global-components';
import { InspectorControls, useBlockProps } from '@wordpress/block-editor';
import './editor.scss';
import { generateBlockCustomCss } from '../_inc/helpers';

/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {WPElement} Element to render.
 */
export default function Edit( props ) {
	const { name, attributes, setAttributes } = props;

	const { 
		customCss
	} = attributes;

	// Block ID
	const blockId = `botiga-tb-block-${ props.clientId.substr( 0, 8 ) }`;
	setAttributes( { blockId: blockId } );

	// Block Props
	const blockProps = useBlockProps( {
		className: classnames( 'botiga-tb-block', 'botiga-tb-sp-short-description' )
	} );

	// Tab Navigation
	const [ tab, setTab ] = useState( 'style' );

	// Defaults
	const defaults = {
		align: 'left',
		paragraphsGap: 0,
		paragraphsGapExtra: 20
	}

	if( typeof customCss.align !== 'undefined' ) {
        defaults.align = customCss.align.value;
    }

	if( typeof customCss.paragraphsGap !== 'undefined' ) {
        defaults.paragraphsGap = customCss.paragraphsGap.value;
    }

	if( typeof customCss.paragraphsGapExtra !== 'undefined' ) {
        defaults.paragraphsGapExtra = customCss.paragraphsGapExtra.value;
    }

	// Custom CSS
	generateBlockCustomCss( blockId, customCss );

	return (
		<div { ...blockProps }>
		
			<InspectorControls>
				<BotigaTabsNavigation
					exclude="settings"
					activeTab={ tab }
					setActiveTab={ setTab } />

				<BotigaTabsContent 
					tabid="style"
					activeTab={ tab }>
					
					<PanelBody title={ __( 'General', 'botiga-pro' ) } initialOpen={ true }>
						<BotigaTypography 
							label={ __( 'Typography', 'botiga-pro' ) }
							idPrefix={ 'text_typography' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}}` }
						/>
						<SelectControl
							label={ __( 'Alignment', 'botiga-pro' ) }
							value={ defaults.align }
							options={ [
								{ label: 'Left', value: 'left' },
								{ label: 'Center', value: 'center' },
								{ label: 'Right', value: 'right' },
							] }
							onChange={ ( value ) => { 
								const css = {
									align: {
										selector: `{{WRAPPER}}`,
										props: `text-align: ${ value };`,
										value: value
									}
								};
								setAttributes( { customCss: { ...customCss, ...css } } ) 
							} }
						/>
						<BotigaColorPicker 
							label={ __( 'Text Color', 'botiga-pro' ) }
							idPrefix={ 'text_color' }
							attributes={ attributes }
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}}` }
						/>
						<BotigaColorPicker 
							label={ __( 'Link Color', 'botiga-pro' ) }
							idPrefix={ 'link_color' }
							withHover={ true }
							attributes={ attributes }
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} a` }
						/>
						<RangeControl
							label={ __( 'Paragraphs Gap', 'botiga-pro' ) }
							labelPosition="edge"
							value={ defaults.paragraphsGapExtra }
							onChange={ ( value ) => { 
								const css = {
									paragraphsGap: {
										selector: '{{WRAPPER}} p',
										props: `margin-bottom: 0px !important;`,
										value: `0px`
									},
									paragraphsGapExtra: {
										selector: '{{WRAPPER}} p + p',
										props: `margin-top: ${ value }px !important;`,
										value: value
									}
								};
								setAttributes( { customCss: { ...customCss, ...css } } ) 
							} }
							min={ 1 }
							max={ 50 }
						/>
					</PanelBody>
				</BotigaTabsContent>
				<BotigaTabsContent 
					tabid="advanced"
					activeTab={ tab }>
					
					<PanelBody title={ __( 'Layout', 'botiga-pro' ) } initialOpen={ true }>
						<BotigaDimensions
							label={ __( 'Margin', 'botiga-pro' ) }
							idPrefix={ '' }
							type={ 'margin' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}}` } />

						<BotigaDimensions
							label={ __( 'Padding', 'botiga-pro' ) }
							idPrefix={ '' }
							type={ 'padding' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}}` } />
					</PanelBody>

				</BotigaTabsContent>
			</InspectorControls>

			<Disabled>
				<ServerSideRender
					block={ name }
					httpMethod="POST"
					attributes={ attributes }
				/>
			</Disabled>

		</div>
	);
}
