import classnames from 'classnames';
import { __ } from '@wordpress/i18n';
import { useState } from '@wordpress/element';
import ServerSideRender from '@wordpress/server-side-render';
import { PanelBody, Disabled } from '@wordpress/components';
import { BotigaDimensions, BotigaTypography, BotigaColorPicker, BotigaTabsNavigation, BotigaTabsContent, BotigaBorder } from '../_inc/global-components';
import { InspectorControls, useBlockProps } from '@wordpress/block-editor';
import './editor.scss';
import { generateBlockCustomCss } from '../_inc/helpers';

/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {WPElement} Element to render.
 */
export default function Edit( props ) {
	const { name, attributes, setAttributes } = props;

	const { 
		customCss
	} = attributes;

	// Block ID
	const blockId = `botiga-tb-block-${ props.clientId.substr( 0, 8 ) }`;
	setAttributes( { blockId: blockId } );

	// Block Props
	const blockProps = useBlockProps( {
		className: classnames( 'botiga-tb-block', 'botiga-tb-sp-additional-information' )
	} );

	// Tab Navigation
	const [ tab, setTab ] = useState( 'style' );

	// Custom CSS
	generateBlockCustomCss( blockId, customCss );

	return (
		<div { ...blockProps }>
		
			<InspectorControls>
				<BotigaTabsNavigation
					exclude="settings"
					activeTab={ tab }
					setActiveTab={ setTab } />

				<BotigaTabsContent 
					tabid="style"
					activeTab={ tab }>
					
					<PanelBody title={ __( 'General', 'botiga-pro' ) } initialOpen={ true }>
						<BotigaTypography 
							label={ __( 'Typography', 'botiga-pro' ) }
							idPrefix={ 'text_typography' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .woocommerce-product-attributes-item__label, {{WRAPPER}} .woocommerce-product-attributes-item__value` }
						/>
						<BotigaColorPicker 
							label={ __( 'Text Color', 'botiga-pro' ) }
							idPrefix={ 'text_color' }
							attributes={ attributes }
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .woocommerce-product-attributes-item__label, {{WRAPPER}} .woocommerce-product-attributes-item__value` }
						/>
						<BotigaColorPicker 
							label={ __( 'Link Color', 'botiga-pro' ) }
							idPrefix={ 'link_color' }
							withHover={ true }
							attributes={ attributes }
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .woocommerce-product-attributes-item__value a` }
						/>
					</PanelBody>
					<PanelBody title={ __( 'Table', 'botiga-pro' ) } initialOpen={ false }>
						<BotigaBorder 
							label={ __( 'Border', 'botiga-pro' ) }
							idPrefix={ 'table_border' }
							defaults={{
								borderType: 'none',
								borderWidth: {
									top: 1,
									left: 1,
									right: 1,
									bottom: 1
								},
								borderColor: '#CCC'
							}}
							attributes={ attributes }
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} table.woocommerce-product-attributes.shop_attributes` }
						/>
						<BotigaColorPicker 
							label={ __( 'Background Color', 'botiga-pro' ) }
							idPrefix={ 'table_background_color' }
							withHover={ false }
							attributes={ attributes }
							setAttributes={ setAttributes }
							cssProp="background-color"
							customCssSelector={ `{{WRAPPER}} table.woocommerce-product-attributes.shop_attributes` }
						/>
					</PanelBody>

				</BotigaTabsContent>
				<BotigaTabsContent 
					tabid="advanced"
					activeTab={ tab }>
					
					<PanelBody title={ __( 'Layout', 'botiga-pro' ) } initialOpen={ true }>
						<BotigaDimensions
							label={ __( 'Margin', 'botiga-pro' ) }
							idPrefix={ '' }
							type={ 'margin' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}}` } />

						<BotigaDimensions
							label={ __( 'Padding', 'botiga-pro' ) }
							idPrefix={ '' }
							type={ 'padding' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}}` } />
					</PanelBody>

				</BotigaTabsContent>
			</InspectorControls>

			<Disabled>
				<ServerSideRender
					block={ name }
					httpMethod="POST"
					attributes={ attributes }
				/>
			</Disabled>

		</div>
	);
}
