import classnames from 'classnames';
import { __ } from '@wordpress/i18n';
import { useState } from '@wordpress/element';
import ServerSideRender from '@wordpress/server-side-render';
import { PanelBody, RangeControl, SelectControl, __experimentalBoxControl as BoxControl, Disabled } from '@wordpress/components';
import { BotigaDimensions, BotigaTypography, BotigaColorPicker, BotigaTabsNavigation, BotigaTabsContent, BotigaBorder } from '../_inc/global-components';
import { InspectorControls, useBlockProps } from '@wordpress/block-editor';
import './editor.scss';
import { generateBlockCustomCss } from '../_inc/helpers';

/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {WPElement} Element to render.
 */
export default function Edit( props ) {
	const { name, attributes, setAttributes } = props;

	const { 
		customCss
	} = attributes;

	// Block ID
	const blockId = `botiga-tb-block-${ props.clientId.substr( 0, 8 ) }`;
	setAttributes( { blockId: blockId } );

	// Block Props
	const blockProps = useBlockProps( {
		className: classnames( blockId, 'botiga-tb-block', 'botiga-tb-sp-add-to-cart' )
	} );

	// Tab Navigation
	const [ tab, setTab ] = useState( 'style' );

	// Defaults
	const defaults = {
		variationColorGap: 8,
		variationButtonGap: 8,
		variationImageGap: 8,
		variationPriceAlign: 'left',
	}

	if( typeof customCss.variationColorGap !== 'undefined' ) {
        defaults.variationColorGap = customCss.variationColorGap.value;
    }

	if( typeof customCss.variationButtonGap !== 'undefined' ) {
        defaults.variationButtonGap = customCss.variationButtonGap.value;
    }

	if( typeof customCss.variationImageGap !== 'undefined' ) {
        defaults.variationImageGap = customCss.variationImageGap.value;
    }

	// Custom CSS
	generateBlockCustomCss( blockId, customCss );

	// Separator Style
	const separatorStyle = {
		fontSize: 14,
		margin: '15px 0 15px',
		borderBottom: '1px solid #e5e5e5',
		padding: '13px 0px'
	}

	return (
		<div { ...blockProps }>
			<InspectorControls>
				<BotigaTabsNavigation
					exclude="settings"
					activeTab={ tab }
					setActiveTab={ setTab } />

				<BotigaTabsContent 
					tabid="style"
					activeTab={ tab }>
					
					<PanelBody title={ __( 'Stock', 'botiga-pro' ) } initialOpen={ true }>
						<BotigaTypography 
							label={ __( 'Typography', 'botiga-pro' ) }
							idPrefix={ 'stock' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .stock` }
						/>
						
						<h6 style={ separatorStyle }>{ __( 'In Stock', 'botiga-pro' ) }</h6>

						<BotigaColorPicker
							label={ __( 'Text Color', 'botiga-pro' ) }
							idPrefix={ 'inStock' }
							withHover={ false }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							cssProp={ `color` }
							customCssSelector={ `{{WRAPPER}} .stock.in-stock` }
						/>

						<h6 style={ separatorStyle }>{ __( 'Out of Stock', 'botiga-pro' ) }</h6>

						<BotigaColorPicker
							label={ __( 'Text Color', 'botiga-pro' ) }
							idPrefix={ 'outOfStock' }
							withHover={ false }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							cssProp={ `color` }
							customCssSelector={ `{{WRAPPER}} .stock.out-of-stock` }
						/>
					</PanelBody>
					<PanelBody title={ __( 'Quantity', 'botiga-pro' ) } initialOpen={ false }>
						<BotigaTypography 
							label={ __( 'Typography', 'botiga-pro' ) }
							idPrefix={ 'quantity_typography' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .quantity` }
						/>
						<BotigaColorPicker
							label={ __( 'Background Color', 'botiga-pro' ) }
							idPrefix={ 'quantity_background' }
							withHover={ false }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							cssProp={ `background-color` }
							customCssSelector={ `{{WRAPPER}} .quantity` }
						/>
						<BotigaBorder 
							label={ __( 'Border', 'botiga-pro' ) }
							idPrefix={ 'quantity_border' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							cssProp={ `border-color` }
							customCssSelector={ `{{WRAPPER}} .quantity` }
						/>
						<BotigaColorPicker
							label={ __( 'Text Color', 'botiga-pro' ) }
							idPrefix={ 'quantity_text_color' }
							withHover={ false }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							cssProp={ `color` }
							customCssSelector={ `{{WRAPPER}} .quantity .qty` }
						/>

						<h6 style={ separatorStyle }>{ __( 'Minus Icon', 'botiga-pro' ) }</h6>
						<BotigaColorPicker
							label={ __( 'Icon Color', 'botiga-pro' ) }
							idPrefix={ 'quantity_minus_color' }
							withHover={ true }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							cssProp={ `color` }
							customCssSelector={ `{{WRAPPER}} .quantity .botiga-quantity-minus` }
						/>
						<BotigaColorPicker
							label={ __( 'Background Color', 'botiga-pro' ) }
							idPrefix={ 'quantity_minus_background_color' }
							withHover={ true }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							cssProp={ `background-color` }
							customCssSelector={ `{{WRAPPER}} .quantity .botiga-quantity-minus` }
						/>
						<BotigaBorder 
							label={ __( 'Border', 'botiga-pro' ) }
							idPrefix={ 'quantity_minus_border' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .quantity .botiga-quantity-minus` }
						/>

						<h6 style={ separatorStyle }>{ __( 'Plus Icon', 'botiga-pro' ) }</h6>
						<BotigaColorPicker
							label={ __( 'Icon Color', 'botiga-pro' ) }
							idPrefix={ 'quantity_plus_color' }
							withHover={ true }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							cssProp={ `color` }
							customCssSelector={ `{{WRAPPER}} .quantity .botiga-quantity-plus` }
						/>
						<BotigaColorPicker
							label={ __( 'Background Color', 'botiga-pro' ) }
							idPrefix={ 'quantity_plus_background_color' }
							withHover={ true }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							cssProp={ `background-color` }
							customCssSelector={ `{{WRAPPER}} .quantity .botiga-quantity-plus` }
						/>
						<BotigaBorder 
							label={ __( 'Border', 'botiga-pro' ) }
							idPrefix={ 'quantity_plus_border' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .quantity .botiga-quantity-plus` }
						/>
					</PanelBody>
					<PanelBody title={ __( 'Add To Cart Button', 'botiga-pro' ) } initialOpen={ false }>
						<BotigaTypography 
							label={ __( 'Typography', 'botiga-pro' ) }
							idPrefix={ 'addtocart_typography' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} button:not(.components-button).single_add_to_cart_button` }
						/>
						<BotigaColorPicker
							label={ __( 'Background Color', 'botiga-pro' ) }
							idPrefix={ 'addtocart_background_color' }
							withHover={ true }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							cssProp={ `background-color` }
							customCssSelector={ `{{WRAPPER}} button:not(.components-button).single_add_to_cart_button` }
						/>
						<BotigaColorPicker
							label={ __( 'Text Color', 'botiga-pro' ) }
							idPrefix={ 'addtocart_color' }
							withHover={ true }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							cssProp={ `color` }
							customCssSelector={ `{{WRAPPER}} button:not(.components-button).single_add_to_cart_button` }
						/>
						<BotigaBorder 
							label={ __( 'Border', 'botiga-pro' ) }
							idPrefix={ 'addtocart_border' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} button:not(.components-button).single_add_to_cart_button` }
						/>
					</PanelBody>
					<PanelBody title={ __( 'Grouped Product', 'botiga-pro' ) } initialOpen={ false }>
						<BotigaTypography 
							label={ __( 'Typography', 'botiga-pro' ) }
							idPrefix={ 'grouped_product_typography' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .group_table` }
						/>
						<BotigaColorPicker
							label={ __( 'Text Color', 'botiga-pro' ) }
							idPrefix={ 'grouped_product_color' }
							withHover={ false }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							cssProp={ `color` }
							customCssSelector={ `{{WRAPPER}} .group_table, {{WRAPPER}} .group_table .woocommerce-Price-amount` }
						/>
						<BotigaColorPicker
							label={ __( 'Links Color', 'botiga-pro' ) }
							idPrefix={ 'grouped_product_links_color' }
							withHover={ true }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							cssProp={ `color` }
							customCssSelector={ `{{WRAPPER}} .group_table a` }
						/>

						<h6 style={ separatorStyle }>{ __( 'Buttons', 'botiga-pro' ) }</h6>
						<BotigaTypography 
							label={ __( 'Typography', 'botiga-pro' ) }
							idPrefix={ 'grouped_product_buttons_typography' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .group_table .add_to_cart_button` }
						/>
						<BotigaColorPicker
							label={ __( 'Background Color', 'botiga-pro' ) }
							idPrefix={ 'grouped_product_buttons_background_color' }
							withHover={ true }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							cssProp={ `background-color` }
							customCssSelector={ `{{WRAPPER}} .group_table .add_to_cart_button` }
						/>
						<BotigaColorPicker
							label={ __( 'Text Color', 'botiga-pro' ) }
							idPrefix={ 'grouped_product_buttons_color' }
							withHover={ true }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							cssProp={ `color` }
							customCssSelector={ `{{WRAPPER}} .group_table .add_to_cart_button` }
						/>
						<BotigaBorder 
							label={ __( 'Border', 'botiga-pro' ) }
							idPrefix={ 'grouped_product_buttons_border' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .group_table .add_to_cart_button` }
						/>
					</PanelBody>
					<PanelBody title={ __( 'Variable Product', 'botiga-pro' ) } initialOpen={ false }>
						<h6 style={ separatorStyle }>{ __( 'Attributes Title', 'botiga-pro' ) }</h6>
						<BotigaTypography 
							label={ __( 'Typography', 'botiga-pro' ) }
							idPrefix={ 'variable_product_att_typography' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .variations_form.cart .variations .label label` }
						/>
						<BotigaColorPicker
							label={ __( 'Text Color', 'botiga-pro' ) }
							idPrefix={ 'variable_product_att_color' }
							withHover={ false }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							cssProp={ `color` }
							customCssSelector={ `{{WRAPPER}} .variations_form.cart .variations .label label` }
						/>

						<h6 style={ separatorStyle }>{ __( 'Clear/Reset Button', 'botiga-pro' ) }</h6>
						<BotigaTypography 
							label={ __( 'Typography', 'botiga-pro' ) }
							idPrefix={ 'variable_product_reset_typography' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} table.variations .reset_variations` }
						/>
						<BotigaColorPicker
							label={ __( 'Text Color', 'botiga-pro' ) }
							idPrefix={ 'variable_product_reset_color' }
							withHover={ true }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							cssProp={ `color` }
							customCssSelector={ `{{WRAPPER}} table.variations .reset_variations` }
						/>
						<BotigaColorPicker
							label={ __( 'Background Color', 'botiga-pro' ) }
							idPrefix={ 'variable_product_reset_background_color' }
							withHover={ true }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							cssProp={ `background-color` }
							customCssSelector={ `{{WRAPPER}} table.variations .reset_variations` }
						/>
						<BotigaBorder 
							label={ __( 'Border', 'botiga-pro' ) }
							idPrefix={ 'variable_product_reset_border' }
							defaults={{
								borderType: 'none',
								borderWidth: {
									top: 0,
									left: 0,
									right: 0,
									bottom: 0
								},
								borderColor: '#CCC'
							}}
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} table.variations .reset_variations` }
						/>
						<BotigaBorder 
							label={ __( 'Border (hover)', 'botiga-pro' ) }
							idPrefix={ 'variable_product_reset_border_hover' }
							defaults={{
								borderType: 'none',
								borderWidth: {
									top: 0,
									left: 0,
									right: 0,
									bottom: 0
								},
								borderColor: '#CCC'
							}}
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} table.variations .reset_variations:hover` }
						/>
						<BotigaDimensions
							label={ __( 'Padding', 'botiga-pro' ) }
							idPrefix={ 'variable_product_reset_padding' }
							type={ 'padding' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} table.variations .reset_variations` } 
						/>

						<h6 style={ separatorStyle }>{ __( 'Variable Price', 'botiga-pro' ) }</h6>
						<SelectControl
							label={ __( 'Alignment', 'botiga-pro' ) }
							value={ defaults.variationPriceAlign }
							options={ [
								{ label: 'Left', value: 'left' },
								{ label: 'Center', value: 'center' },
								{ label: 'Right', value: 'right' },
							] }
							onChange={ ( value ) => { 								
								const css = {
									variationPriceAlign: {
										selector: `{{WRAPPER}} .woocommerce-variation-price`,
										props: `text-align: ${ value };`,
										value: value
									}
								};
								setAttributes( { customCss: { ...customCss, ...css } } ) 
							} }
						/>
						
						<h6 style={ separatorStyle }>{ __( 'Regular Price (variable)', 'botiga-pro' ) }</h6>
						<BotigaTypography 
							label={ __( 'Typography', 'botiga-pro' ) }
							idPrefix={ 'variable_regular_price_typography' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .woocommerce-variation-price .price` }
						/>
						<BotigaColorPicker 
							label={ __( 'Color', 'botiga-pro' ) }
							idPrefix={ 'variable_regular_price_color' }
							attributes={ attributes }
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .woocommerce-variation-price .price .woocommerce-Price-amount, {{WRAPPER}} .woocommerce-variation-price .price` }
						/>

						<h6 style={ separatorStyle }>{ __( 'Sale Price (variable)', 'botiga-pro' ) }</h6>
						<BotigaTypography 
							label={ __( 'Typography', 'botiga-pro' ) }
							idPrefix={ 'variable_sale_price_typography' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .woocommerce-variation-price .price del` }
						/>
						<BotigaColorPicker 
							label={ __( 'Color', 'botiga-pro' ) }
							idPrefix={ 'variable_sale_price_color' }
							attributes={ attributes }
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .woocommerce-variation-price .price del .woocommerce-Price-amount, {{WRAPPER}} .woocommerce-variation-price .price del` }
						/>

						<h6 style={ separatorStyle }>{ __( 'Dropdown Swatch', 'botiga-pro' ) }</h6>
						<BotigaTypography 
							label={ __( 'Typography', 'botiga-pro' ) }
							idPrefix={ 'variation_default_typography' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} table.variations td select` }
						/>
						<BotigaColorPicker
							label={ __( 'Background Color', 'botiga-pro' ) }
							idPrefix={ 'variation_default_background_color' }
							withHover={ false }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							cssProp={ `background-color` }
							customCssSelector={ `{{WRAPPER}} table.variations td select` }
						/>
						<BotigaColorPicker
							label={ __( 'Text Color', 'botiga-pro' ) }
							idPrefix={ 'variation_default_color' }
							withHover={ false }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							cssProp={ `color` }
							customCssSelector={ `{{WRAPPER}} table.variations td select` }
						/>
						<BotigaBorder 
							label={ __( 'Border', 'botiga-pro' ) }
							idPrefix={ 'variation_default_border' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} table.variations td select` }
						/>
						<BotigaDimensions
							label={ __( 'Padding', 'botiga-pro' ) }
							idPrefix={ 'variation_default_padding' }
							type={ 'padding' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} table.variations td select` } 
						/>

						<h6 style={ separatorStyle }>{ __( 'Color Swatch', 'botiga-pro' ) }</h6>
						<BotigaBorder 
							label={ __( 'Border', 'botiga-pro' ) }
							idPrefix={ 'variation_color_border' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							cssProp={ `border-color` }
							customCssSelector={ `{{WRAPPER}} .botiga-variations-wrapper .botiga-variation-type-color > a` }
						/>
						<BotigaBorder 
							label={ __( 'Border (hover)', 'botiga-pro' ) }
							idPrefix={ 'variation_color_border_hover' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							cssProp={ `border-color` }
							customCssSelector={ `{{WRAPPER}} .botiga-variations-wrapper .botiga-variation-type-color > a:hover:not(.disabled):before` }
						/>
						<BotigaDimensions
							label={ __( 'Border Radius', 'botiga-pro' ) }
							idPrefix={ 'variation_color_border_radius' }
							type={ 'border-radius' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .botiga-variations-wrapper .botiga-variation-type-color > a, {{WRAPPER}} .botiga-variations-wrapper .botiga-variation-type-color > a:not(.disabled):before` } 
						/>
						<RangeControl
							label={ __( 'Gap', 'botiga-pro' ) }
							labelPosition="edge"
							value={ defaults.variationColorGap }
							onChange={ ( value ) => { 
								const css = {
									variationColorGap: {
										selector: '{{WRAPPER}} .botiga-variations-wrapper .botiga-variation-type-color > a',
										props: `margin-right: ${ value }px !important;`,
										value: value
									},
								};
								setAttributes( { customCss: { ...customCss, ...css } } ) 
							} }
							min={ 0 }
							max={ 50 }
						/>

						<h6 style={ separatorStyle }>{ __( 'Button Swatch', 'botiga-pro' ) }</h6>
						<BotigaTypography 
							label={ __( 'Typography', 'botiga-pro' ) }
							idPrefix={ 'variation_button_typography' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .botiga-variations-wrapper .botiga-variation-type-button > a` }
						/>
						<BotigaColorPicker
							label={ __( 'Text Color', 'botiga-pro' ) }
							idPrefix={ 'variation_button_color' }
							withHover={ true }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							cssProp={ `color` }
							customCssSelector={ `{{WRAPPER}} .botiga-variations-wrapper .botiga-variation-type-button > a` }
						/>
						<BotigaColorPicker
							label={ __( 'Background Color', 'botiga-pro' ) }
							idPrefix={ 'variation_button_background_color' }
							withHover={ true }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							cssProp={ `background-color` }
							customCssSelector={ `{{WRAPPER}} .botiga-variations-wrapper .botiga-variation-type-button > a` }
						/>
						<BotigaBorder 
							label={ __( 'Border', 'botiga-pro' ) }
							idPrefix={ 'variation_button_border' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							cssProp={ `border-color` }
							customCssSelector={ `{{WRAPPER}} .botiga-variations-wrapper .botiga-variation-type-button > a` }
						/>
						<BotigaBorder 
							label={ __( 'Border (hover)', 'botiga-pro' ) }
							idPrefix={ 'variation_button_border_hover' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							cssProp={ `border-color` }
							customCssSelector={ `{{WRAPPER}} .botiga-variations-wrapper .botiga-variation-type-button > a:hover` }
						/>
						<BotigaDimensions
							label={ __( 'Border Radius', 'botiga-pro' ) }
							idPrefix={ 'variation_button_border_radius' }
							type={ 'border-radius' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .botiga-variations-wrapper .botiga-variation-type-button > a` } 
						/>
						<RangeControl
							label={ __( 'Gap', 'botiga-pro' ) }
							labelPosition="edge"
							value={ defaults.variationButtonGap }
							onChange={ ( value ) => { 
								const css = {
									variationButtonGap: {
										selector: '{{WRAPPER}} .botiga-variations-wrapper .botiga-variation-type-button > a',
										props: `margin-right: ${ value }px !important;`,
										value: value
									},
								};
								setAttributes( { customCss: { ...customCss, ...css } } ) 
							} }
							min={ 0 }
							max={ 50 }
						/>
						<BotigaDimensions
							label={ __( 'Padding', 'botiga-pro' ) }
							idPrefix={ 'variation_button_padding' }
							type={ 'padding' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .botiga-variations-wrapper .botiga-variation-type-button > a` } 
						/>

						<h6 style={ separatorStyle }>{ __( 'Image Swatch', 'botiga-pro' ) }</h6>
						<BotigaBorder 
							label={ __( 'Border', 'botiga-pro' ) }
							idPrefix={ 'variation_image_border' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							cssProp={ `border-color` }
							customCssSelector={ `{{WRAPPER}} .botiga-variations-wrapper .botiga-variation-type-image > a` }
						/>
						<BotigaBorder 
							label={ __( 'Border (hover)', 'botiga-pro' ) }
							idPrefix={ 'variation_image_border_hover' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							cssProp={ `border-color` }
							customCssSelector={ `{{WRAPPER}} .botiga-variations-wrapper .botiga-variation-type-image > a:hover:not(.disabled):before` }
						/>
						<BotigaDimensions
							label={ __( 'Border Radius', 'botiga-pro' ) }
							idPrefix={ 'variation_image_border_radius' }
							type={ 'border-radius' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .botiga-variations-wrapper .botiga-variation-type-image > a, {{WRAPPER}} .botiga-variations-wrapper .botiga-variation-type-image > a:not(.disabled):before` } 
						/>
						<RangeControl
							label={ __( 'Gap', 'botiga-pro' ) }
							labelPosition="edge"
							value={ defaults.variationImageGap }
							onChange={ ( value ) => { 
								const css = {
									variationImageGap: {
										selector: '{{WRAPPER}} .botiga-variations-wrapper .botiga-variation-type-image > a',
										props: `margin-right: ${ value }px !important;`,
										value: value
									},
								};
								setAttributes( { customCss: { ...customCss, ...css } } ) 
							} }
							min={ 0 }
							max={ 50 }
						/>

					</PanelBody>

				</BotigaTabsContent>
				<BotigaTabsContent 
					tabid="advanced"
					activeTab={ tab }>
					
					<PanelBody title={ __( 'Layout', 'botiga-pro' ) } initialOpen={ true }>
						<BotigaDimensions
							label={ __( 'Margin', 'botiga-pro' ) }
							idPrefix={ 'title' }
							type={ 'margin' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}}` } 
						/>
						<BotigaDimensions
							label={ __( 'Padding', 'botiga-pro' ) }
							idPrefix={ 'title' }
							type={ 'padding' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}}` } 
						/>
					</PanelBody>

				</BotigaTabsContent>
			</InspectorControls>

			<Disabled>
				<ServerSideRender
					block={ name }
					httpMethod="POST"
					attributes={ attributes }
				/>
			</Disabled>

		</div>
	);
}
