<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor icon list widget.
 *
 * Elementor widget that displays a bullet list with any chosen icons and texts.
 *
 * @since 1.0.0
 */
class Botiga_Elementor_Single_Product_Meta extends Widget_Base {

	/**
	 * Constructor
	 */
	public function __construct( $data=[], $args=null ){
		parent::__construct( $data, $args );
	}

	/**
	 * Load styles dependencies
	 */
	public function get_style_depends(){
		return [ 'botiga-builder-widgets-styles' ];	
	}

	public function get_help_url() {
		return 'https://docs.athemes.com/article/pro-templates-builder-overview/';
	}
		
	/**
	 * Get widget name.
	 *
	 * Retrieve icon list widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'botiga-elementor-single-product-meta';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve icon list widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Single Product Meta', 'botiga-pro' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve icon list widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-product-meta';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the icon list widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'botiga-woocommerce' ];
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'meta', 'category', 'sku', 'product', 'shop', 'botiga' ];
	}	


	/**
	 * Register icon list widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'section_style_content',
			[
				'label' => __( 'Content', 'botiga-pro' ),
				'tab' 	=> Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' 		=> 'typography',
				'selector' 	=> '{{WRAPPER}} .product_meta',
				'scheme' 	=> Core\Schemes\Typography::TYPOGRAPHY_3,
			]
		);

		$this->add_control(
			'text_alignment',
			[
				'label' 		=> __( 'Alignment', 'botiga-pro'),
				'type' => Controls_Manager::CHOOSE,
				'default' => 'left',
				'options' => [
					'left' => [
						'title' => __( 'Left', 'botiga-pro' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => __( 'Center', 'botiga-pro' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => __( 'Right', 'botiga-pro' ),
						'icon' => 'eicon-text-align-right',
					],
				],
			]
		);

		$this->add_control(
			'text_color',
			[
				'label' 	=> __( 'Text Color', 'botiga-pro' ),
				'type' 		=> Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .product_meta' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'link_color',
			[
				'label' 	=> __( 'Link Color', 'botiga-pro' ),
				'type' 		=> Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .product_meta a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'link_color_hover',
			[
				'label' 	=> __( 'Link Color (hover)', 'botiga-pro' ),
				'type' 		=> Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .product_meta a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'paragraphs_gap',
			[
				'label' => esc_html__( 'Items Gap', 'textdomain' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px' ],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 50,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 8,
				],
				'selectors' => [
					'{{WRAPPER}} .product_meta > span' => 'margin-bottom: 0;',
					'{{WRAPPER}} .product_meta > span + span' => 'margin-top: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

	}

	/**
	 * Get the first product found in the database
	 * 
	 * @access protected
	 */
	protected function get_first_post() {
		if( is_singular( 'product' ) ) {
			global $post;
			return $post;
		}
		
		$first_product = wc_get_products( array(
			'limit' => 1,
			'orderby' => 'date',
			'order' => 'DESC',
			'return' => 'ids'
		) );

		$post = get_post( $first_product[0] );

		return $post;
	}

	/**
	 * Render icon list widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$edit_mode = \Elementor\Plugin::$instance->editor->is_edit_mode();

		$wrapper_selector = $this->get_html_wrapper_class();
		$element_id       = $this->get_id();

		$custom_css = '';

		// Alignment
		if( isset( $settings['text_alignment'] ) ) {
			if( 'center' === $settings['text_alignment'] ) {
				$custom_css .= "
					.${wrapper_selector}[data-id=\"$element_id\"] .product-gallery-summary {
						justify-content: center;
					}

					.${wrapper_selector}[data-id=\"$element_id\"] .product-gallery-summary .product_meta {
						text-align: center;
					}
				";
			} elseif( 'right' === $settings['text_alignment'] ) {
				$custom_css .= "
					.${wrapper_selector}[data-id=\"$element_id\"] .product-gallery-summary {
						justify-content: flex-end;
					}

					.${wrapper_selector}[data-id=\"$element_id\"] .product-gallery-summary .product_meta {
						text-align: right;
					}
				";
			}
		}

		echo '<div class="botiga-tb-el-block botiga-tb-el-sp-meta">';
			if( $custom_css ) {
				echo '<style type="text/css">'. $custom_css .'</style>';
			}

			echo '<div class="product-gallery-summary" style="margin-bottom: 0;">';

				global $post;
				$post = $this->get_first_post();
				setup_postdata( $post );

				woocommerce_template_single_meta();

				wp_reset_postdata();

			echo '</div>';
		echo '</div>';
	}

	/**
	 * Render icon list widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function content_template() {	
	}
}
Plugin::instance()->widgets_manager->register_widget_type( new Botiga_Elementor_Single_Product_Meta() );