<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor icon list widget.
 *
 * Elementor widget that displays a bullet list with any chosen icons and texts.
 *
 * @since 1.0.0
 */
class Botiga_Elementor_Single_Product_Additional_Information extends Widget_Base {

	/**
	 * Constructor
	 */
	public function __construct( $data=[], $args=null ){
		parent::__construct( $data, $args );
	}

	/**
	 * Load styles dependencies
	 */
	public function get_style_depends(){
		return [ 'botiga-builder-widgets-styles' ];	
	}

	public function get_help_url() {
		return 'https://docs.athemes.com/article/pro-templates-builder-overview/';
	}
		
	/**
	 * Get widget name.
	 *
	 * Retrieve icon list widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'botiga-elementor-single-product-additional-information';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve icon list widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'Single Product Additional Information', 'botiga-pro' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve icon list widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-product-info';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the icon list widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'botiga-woocommerce' ];
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'additional', 'information', 'info', 'product', 'shop', 'botiga' ];
	}	


	/**
	 * Register icon list widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {

		$this->start_controls_section(
			'section_style_content',
			[
				'label' => __( 'Content', 'botiga-pro' ),
				'tab' 	=> Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' 		=> 'text_typography',
				'selector' 	=> '{{WRAPPER}} .woocommerce-product-attributes-item__label, {{WRAPPER}} .woocommerce-product-attributes-item__value',
				'scheme' 	=> Core\Schemes\Typography::TYPOGRAPHY_3,
			]
		);

		$this->add_control(
			'text_color',
			[
				'label' => __( 'Text Color', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .woocommerce-product-attributes-item__label' => 'color: {{VALUE}};',
					'{{WRAPPER}} .woocommerce-product-attributes-item__value' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'link_color',
			[
				'label' => __( 'Link Color', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .woocommerce-product-attributes-item__value a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'link_color_hover',
			[
				'label' => __( 'Link Color (hover)', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .woocommerce-product-attributes-item__value a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_table',
			[
				'label' => __( 'Table', 'botiga-pro' ),
				'tab' 	=> Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name' => 'table_border',
				'selector' => '{{WRAPPER}} table.woocommerce-product-attributes.shop_attributes',
			]
		);

		$this->add_control(
			'table_background_color',
			[
				'label' => __( 'Background Color', 'botiga-pro' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} table.woocommerce-product-attributes.shop_attributes' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_section();

	}

	/**
	 * Get the first product found in the database
	 * 
	 * @access protected
	 */
	protected function get_first_post() {
		if( is_singular( 'product' ) ) {
			global $post;
			return $post;
		}
		
		$first_product = wc_get_products( array(
			'limit' => 1,
			'orderby' => 'date',
			'order' => 'DESC',
			'return' => 'ids'
		) );

		$post = get_post( $first_product[0] );

		return $post;
	}

	/**
	 * Render icon list widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$edit_mode = \Elementor\Plugin::$instance->editor->is_edit_mode();

		echo '<div class="botiga-tb-el-block botiga-tb-el-sp-additional-information">';

			global $post;
			$post = $this->get_first_post();
			setup_postdata( $post );

			if( ! $edit_mode ) {
				woocommerce_product_additional_information_tab();
			} else {
				global $product;
				wc_display_product_attributes( $product );
			}

			wp_reset_postdata();

		echo '</div>';
	}

	/**
	 * Render icon list widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function content_template() {	
	}
}
Plugin::instance()->widgets_manager->register_widget_type( new Botiga_Elementor_Single_Product_Additional_Information() );