<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor icon list widget.
 *
 * Elementor widget that displays a bullet list with any chosen icons and texts.
 *
 * @since 1.0.0
 */
class aThemes_Elementor_ACF_Field extends Widget_Base {

	/**
	 * Constructor
	 */
	public function __construct( $data=[], $args=null ){
		parent::__construct( $data, $args );
		
	}

	/**
	 * Load styles dependencies
	 */
	public function get_style_depends(){
		if ( Icons_Manager::is_migration_allowed() ) {
			return [
				'botiga-builder-widgets-styles',
				'elementor-icons-fa-regular',
				'elementor-icons-fa-solid',
			];
		}

		return [ 'botiga-builder-widgets-styles' ];	
	}

	public function get_help_url() {
		return 'https://docs.athemes.com/article/435-templates-system-overview';
	}
		
	/**
	 * Get widget name.
	 *
	 * Retrieve icon list widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'athemes-elementor-acf-field';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve icon list widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( 'ACF Field', 'botiga-pro' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve icon list widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-custom';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the icon list widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'botiga-theme-elements' ];
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'acf', 'custom field', 'advanced', 'site', 'botiga-pro' ];
	}	


	/**
	 * Register icon list widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {
		$this->start_controls_section(
			'section_acf_field_settings',
			[
				'label' => __( 'ACF Field', 'botiga-pro' ),
			]
		);

		$this->add_control(
			'field_name_acf',
			[
				'label'     => __( 'Field Name', 'botiga-pro' ),
				'type'      => Controls_Manager::SELECT,
				'options'   => $this->get_acf_fields(),
			]
		);

		$this->add_control(
			'element_tag',
			[
				'label' 		=> __( 'Element tag', 'botiga-pro'),
				'type' => Controls_Manager::SELECT,
				'default' => 'div',
				'options' => [
					'h1' 	=> __('H1', 'botiga-pro'),
					'h2' 	=> __('H2', 'botiga-pro'),
					'h3' 	=> __('H3', 'botiga-pro'),
					'h4' 	=> __('H4', 'botiga-pro'),
					'h5' 	=> __('H5', 'botiga-pro'),
					'h6' 	=> __('H6', 'botiga-pro'),
					'span' 	=> __('Span', 'botiga-pro'),
					'p' 	=> __('P', 'botiga-pro'),
					'div' 	=> __('Div', 'botiga-pro'),
				],
			]
		);

		$this->add_control(
			'alignment',
			[
				'label' 		=> __('Alignment', 'botiga-pro'),
				'type' => Controls_Manager::CHOOSE,
				'default' => 'left',
				'options' => [
					'left' => [
						'title' => __( 'Left', 'botiga-pro' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => __( 'Center', 'botiga-pro' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => __( 'Right', 'botiga-pro' ),
						'icon' => 'eicon-text-align-right',
					],
				],
				'selectors' => [
					'{{WRAPPER}} .botiga-acf-field-wrapper' => 'text-align: {{VALUE}};',
				],				
			]
		);

		$this->add_control(
			'show_icon',
			[
				'label' => esc_html__( 'Show icon before the field', 'botiga-pro' ),
				'type' => Controls_Manager::SWITCHER,
				'separator'			=> 'before',
			]
		);

		$this->add_control(
			'selected_icon',
			[
				'label' => esc_html__( 'Choose Icon', 'botiga-pro' ),
				'type' => Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'condition' => [
					'show_icon' => 'yes',
				],
			]
		);

		$this->add_control(
			'text_before',
			[
				'label' => esc_html__( 'Text before', 'botiga-pro' ),
				'type' => Controls_Manager::TEXT,
				'default' => '',
				'separator' => 'before',
			]
		);

		$this->add_control(
			'text_after',
			[
				'label' => esc_html__( 'Text after', 'botiga-pro' ),
				'type' => Controls_Manager::TEXT,
				'default' => '',
			]
		);


		$this->end_controls_section();

		//Styles
		$this->start_controls_section(
			'section_acf_field_styles',
			[
				'label' => __( 'Field', 'botiga-pro' ),
				'tab' 	=> Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'field_color',
			[
				'label' 		=> __( 'Color', 'botiga-pro' ),
				'type' 			=> Controls_Manager::COLOR,
				'selectors' 	=> [
					'{{WRAPPER}} .botiga-acf-field-wrapper' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'field_typography',
				'selector' => '{{WRAPPER}} .botiga-acf-field-wrapper',
			]
		);		

		$this->end_controls_section();

		$this->start_controls_section(
			'section_acf_text_before_styles',
			[
				'label' => __( 'Text before', 'botiga-pro' ),
				'tab' 	=> Controls_Manager::TAB_STYLE,
			]
		);	

		$this->add_control(
			'text_before_color',
			[
				'label' 		=> __( 'Color', 'botiga-pro' ),
				'type' 			=> Controls_Manager::COLOR,
				'selectors' 	=> [
					'{{WRAPPER}} .botiga-acf-field-wrapper .text-before' => 'color: {{VALUE}};',
				],
			]
		);	
		
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'text_before_typography',
				'selector' => '{{WRAPPER}} .botiga-acf-field-wrapper .text-before',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_acf_text_after_styles',
			[
				'label' => __( 'Text after', 'botiga-pro' ),
				'tab' 	=> Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'text_after_color',
			[
				'label' 		=> __( 'Color', 'botiga-pro' ),
				'type' 			=> Controls_Manager::COLOR,
				'selectors' 	=> [
					'{{WRAPPER}} .botiga-acf-field-wrapper .text-after' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'text_after_typography',
				'selector' => '{{WRAPPER}} .botiga-acf-field-wrapper .text-after',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_acf_icon_styles',
			[
				'label' => __( 'Icon', 'botiga-pro' ),
				'tab' 	=> Controls_Manager::TAB_STYLE,
				'condition' => [
					'show_icon' => 'yes',
				],
			]
		);

		$this->add_control(
			'icon_color',
			[
				'label' 		=> __( 'Color', 'botiga-pro' ),
				'type' 			=> Controls_Manager::COLOR,
				'selectors' 	=> [
					'{{WRAPPER}} .botiga-acf-field-wrapper i' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'icon_size',
			[
				'label' => __( 'Size', 'botiga-pro' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .botiga-acf-field-wrapper i' => 'font-size: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'icon_spacing',
			[
				'label' => __( 'Spacing', 'botiga-pro' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .botiga-acf-field-wrapper i' => 'margin-right: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

	}

	/**
	 * Render icon list widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {

		if ( ! defined( 'ACF_VERSION' ) ) {
			return;
		}

		$settings = $this->get_settings_for_display();

		//$post_type = $settings['post_type'];

		if ( !$settings['field_name_acf'] ) {
			return;
		}

		$value = get_field( $settings['field_name_acf'] );

		$edit_mode = \Elementor\Plugin::$instance->editor->is_edit_mode();
		if ( $edit_mode ) {
			$value = esc_html__( 'Placeholder', 'botiga-pro' );
		}
		
		if ( '' === $value || empty( $value ) ) {
			return;
		}

		if ( is_array( $value) ) { //return if it's a field we can't process
			return;
		}

		echo '<' . esc_attr( $settings['element_tag'] ) . ' class="botiga-acf-field-wrapper">';

		if ( 'yes' === $settings['show_icon'] && $settings['selected_icon'] ) {
			Icons_Manager::render_icon( $settings['selected_icon'], [ 'aria-hidden' => 'true' ] );
		}

		if ( $settings['text_before'] ) {
			echo '<span class="text-before">' . wp_kses_post( $settings['text_before'] ) . '</span>';
		}
		
		echo '<span>' . wp_kses_post( $value ) . '</span>';

		if ( $settings['text_after'] ) {
			echo '<span class="text-after">' . wp_kses_post( $settings['text_after'] ) . '</span>';
		}
		
		echo '</' . esc_attr( $settings['element_tag'] ) . '>';

	}

	/**
	 * Get acf fields.
	 *
	 * @return array
	 */
	private function get_acf_fields() {
		if ( ! defined( 'ACF_VERSION' ) || ! function_exists( 'get_field_objects' ) ) {
			return [];
		}

		
		if ( function_exists( 'acf_get_field_groups' ) ) {
			$groups = acf_get_field_groups();
		} else {
			$groups = apply_filters( 'acf/get_field_groups', [] );
		}

		$options_page_groups_ids = array();

		if ( function_exists( 'acf_options_page' ) ) {
			$pages = acf_options_page()->get_pages();

			foreach ( $pages as $slug => $page ) {
				$options_page_groups = acf_get_field_groups( array(
					'options_page' => $slug,
				) );

				foreach ( $options_page_groups as $options_page_group ) {
					$options_page_groups_ids[] = $options_page_group['ID'];
				}
			}
		}

		$result      = array();
		$whitelisted = $this->whitelisted_fields();

		foreach ( $groups as $group ) {

			// ACF >= 5.0.0
			if ( function_exists( 'acf_get_fields' ) ) {
				$fields = acf_get_fields( $group['ID'] );
			} else {
				$fields = apply_filters( 'acf/field_group/get_fields', [], $group['id'] );
			}

			$options = [];

			if ( ! is_array( $fields ) ) {
				continue;
			}

			$has_option_page_location = in_array( $group['ID'], $options_page_groups_ids, true );
			$is_only_options_page = $has_option_page_location && 1 === count( $group['location'] );

			foreach ( $fields as $field ) {

				if ( ! isset( $whitelisted[ $field['type'] ] ) ) {
					continue;
				}

				if ( $has_option_page_location ) {
					$key = 'options::' . $field['name'];

					$options[ $key ] = array(
						'type'  => $field['type'],
						'label' => __( 'Options', 'jet-engine' ) . ':' . $field['label'],
					);

					if ( $is_only_options_page ) {
						continue;
					}
				}

				$key = $field['key'] . '::' . $field['name'];
				$options[ $key ] = array(
					'type'  => $field['type'],
					'label' => $field['label']
				);

			}

			if ( empty( $options ) ) {
				continue;
			}

			$result[$field['name']] = $field['name'];
		}


		return $result;
	}


	public function whitelisted_fields() {

		return array(
			'text'             => array( 'field', 'link' ),
			'textarea'         => array( 'field' ),
			'number'           => array( 'field' ),
			'range'            => array( 'field' ),
			'email'            => array( 'field', 'link' ),
			'url'              => array( 'field', 'link' ),
			'wysiwyg'          => array( 'field' ),
			'image'            => array( 'link', 'image' ),
			'file'             => array( 'field', 'link' ),
			'gallery'          => array( 'field' ),
			'select'           => array( 'field' ),
			'radio'            => array( 'field' ),
			'checkbox'         => array( 'field' ),
			'button_group'     => array( 'field' ),
			'true_false'       => array( 'field' ),
			'page_link'        => array( 'field', 'link' ),
			'post_object'      => array( 'field', 'link' ),
			'relationship'     => array( 'field', 'link' ),
			'taxonomy'         => array( 'field', 'link' ),
			'date_picker'      => array( 'field', 'link' ),
			'date_time_picker' => array( 'field' ),
			'time_picker'      => array( 'field' ),
			'repeater'         => array( 'repeater' ),
			'oembed'           => array( 'field' ),
		);

	}

	/**
	 * Render icon list widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function content_template() {	
	}
}
Plugin::instance()->widgets_manager->register_widget_type( new aThemes_Elementor_ACF_Field() );