<?php
/**
 * Product Swatches
 * 
 * @package Botiga_Pro
 */
if ( class_exists( 'Botiga_Modules' ) && ! Botiga_Modules::is_module_active( 'product-swatches' ) ) {
    return;
}

/**
 * Product swatches customize options.
 */
function botiga_product_swatches_options( $wp_customize ) {
    require BOTIGA_PRO_DIR . 'inc/modules/product-swatches/customizer/options.php';
}
add_action( 'customize_register', 'botiga_product_swatches_options', 999 );

/**
 * Botiga Product Swatch Class
 */
class Botiga_Product_Swatch {

    public function __construct() {

        if( is_admin() ) {
            add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_scripts' ) );
            add_filter( 'product_attributes_type_selector', array( $this, 'available_attributes_types' ) );
            add_action( 'admin_init', array( $this, 'add_custom_attribute_fields' ) );
        }

        // 'wp' action is needed here to be able to use wp conditional functions.
        add_action( 'wp', function(){
            global $post;

            $quickview_enabled                     = get_theme_mod( 'shop_product_quickview_layout', 'layout1' ) === 'layout1' ? false : true;
            $enable_product_swatch_on_shop_catalog = get_theme_mod( 'product_swatch_on_shop_catalog', 0 );
            
            // Enqueue the script only on pages where it is needed
            if( $enable_product_swatch_on_shop_catalog ) {
                remove_action( 'woocommerce_after_shop_loop_item', 'woocommerce_template_loop_add_to_cart' );
                add_action( 'woocommerce_after_shop_loop_item', array( $this, 'product_swatch_on_shop_catalog' ) );
    
                if( is_shop() || is_product_category() || is_product_tag() || is_product_taxonomy() || is_search() && get_post_type( 'product' ) || is_singular( 'product' ) || is_404() || $this->page_has_woo_blocks( $post ) || $this->page_has_woo_shortcode( $post ) || defined( 'DOKAN_PLUGIN_VERSION' ) && dokan_is_store_page() ) {
                    add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
                }
    
            } else if( is_singular( 'product' ) || ( $quickview_enabled && ( is_shop() || is_product_category() || is_product_tag() || is_product_taxonomy() || is_search() || is_404() || $this->page_has_woo_blocks( $post ) || $this->page_has_woo_shortcode( $post ) ) ) ) {
    
                // If option to display on archive is disabled, then enqueue scripts only on single product pages
                add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
    
            }
        } );
        
        add_filter( 'woocommerce_dropdown_variation_attribute_options_html', array( $this, 'variation_attribute_options_html' ), 10, 2 );
        
        // Custom CSS
        add_filter( 'botiga_custom_css_output', array( $this, 'custom_css' ) );
    }

    public function admin_enqueue_scripts( $hook ) {

        if ( $hook === 'edit-tags.php' || $hook === 'term.php' ) {

            wp_enqueue_media();

            wp_enqueue_style( 'wp-color-picker' );
            wp_enqueue_script( 'wp-color-picker' );
            
            wp_enqueue_style( 'botiga-product-swatches', BOTIGA_PRO_URI . 'assets/css/admin/botiga-product-swatches.min.css', array(), BOTIGA_PRO_VERSION );
            wp_enqueue_script( 'botiga-product-swatches', BOTIGA_PRO_URI . 'assets/js/admin/botiga-product-swatches.min.js', array(), BOTIGA_PRO_VERSION, true );
    
        }

    }

    public function enqueue_scripts() {
        wp_enqueue_script( 'botiga-product-swatch', BOTIGA_PRO_URI . 'assets/js/botiga-product-swatch.min.js', array(), BOTIGA_PRO_VERSION, true );
    }

    /**
     * Custom CSS
     */
    public function custom_css( $css ) {
        $product_swatch_on_shop_catalog = get_theme_mod( 'product_swatch_on_shop_catalog', 0 );

        if( $product_swatch_on_shop_catalog ) {
            $shop_product_alignment = get_theme_mod( 'shop_product_alignment', 'center' );
            $swatches_alignment     = $shop_product_alignment;

            if( $shop_product_alignment === 'left' ) {
                $swatches_alignment = '-webkit-box-pack:start; -ms-flex-pack:start; justify-content:flex-start;';
            }

            if( $shop_product_alignment === 'center' ) {
                $swatches_alignment = '-webkit-box-pack:center; -ms-flex-pack:center; justify-content:center;';
            }

            if( $shop_product_alignment === 'right' ) {
                $swatches_alignment = '-webkit-box-pack:end; -ms-flex-pack:end; justify-content:flex-end;';

                $css .= '
                    ul.wc-block-grid__products li.wc-block-grid__product .botiga-variations-wrapper .botiga-variation-type-image>a, 
                    ul.wc-block-grid__products li.wc-block-grid__product .botiga-variations-wrapper .botiga-variation-type-color>a,
                    ul.wc-block-grid__products li.product .botiga-variations-wrapper .botiga-variation-type-image>a, 
                    ul.wc-block-grid__products li.product .botiga-variations-wrapper .botiga-variation-type-color>a,
                    ul.products li.wc-block-grid__product .botiga-variations-wrapper .botiga-variation-type-image>a, 
                    ul.products li.wc-block-grid__product .botiga-variations-wrapper .botiga-variation-type-color>a,
                    ul.products li.product .botiga-variations-wrapper .botiga-variation-type-image>a, 
                    ul.products li.product .botiga-variations-wrapper .botiga-variation-type-color>a {
                        margin: 5px 0px 5px 8px;
                    }
                    ul.wc-block-grid__products li.wc-block-grid__product .botiga-variations-wrapper .botiga-variation-type-image>a:first-child, 
                    ul.wc-block-grid__products li.wc-block-grid__product .botiga-variations-wrapper .botiga-variation-type-color>a:first-child,
                    ul.wc-block-grid__products li.product .botiga-variations-wrapper .botiga-variation-type-image>a:first-child, 
                    ul.wc-block-grid__products li.product .botiga-variations-wrapper .botiga-variation-type-color>a:first-child,
                    ul.products li.wc-block-grid__product .botiga-variations-wrapper .botiga-variation-type-image>a:first-child, 
                    ul.products li.wc-block-grid__product .botiga-variations-wrapper .botiga-variation-type-color>a:first-child,
                    ul.products li.product .botiga-variations-wrapper .botiga-variation-type-image>a:first-child, 
                    ul.products li.product .botiga-variations-wrapper .botiga-variation-type-color>a:first-child {
                        margin-left: 0;
                    }
                ';
            }

            $css .= '
                ul.wc-block-grid__products li.wc-block-grid__product .botiga-variations-wrapper .botiga-variation-type-image, 
                ul.wc-block-grid__products li.wc-block-grid__product .botiga-variations-wrapper .botiga-variation-type-color,
                ul.wc-block-grid__products li.product .botiga-variations-wrapper .botiga-variation-type-image, 
                ul.wc-block-grid__products li.product .botiga-variations-wrapper .botiga-variation-type-color,
                ul.products li.wc-block-grid__product .botiga-variations-wrapper .botiga-variation-type-image, 
                ul.products li.wc-block-grid__product .botiga-variations-wrapper .botiga-variation-type-color,
                ul.products li.product .botiga-variations-wrapper .botiga-variation-type-image, 
                ul.products li.product .botiga-variations-wrapper .botiga-variation-type-color {'
                    . esc_attr( $swatches_alignment ) .
                '}
            ';
        }

        // Swatches Styling (Select)

        // Text Color
        $css .= Botiga_Custom_CSS::get_color_css( 'product_swatch_select_text_color', '', '.variations select' );

        // Border Color
        $css .= Botiga_Custom_CSS::get_border_color_css( 'product_swatch_select_border_color', '', '.variations select' );

        // Background Color
        $css .= Botiga_Custom_CSS::get_background_color_css( 'product_swatch_select_background_color', '', '.variations select' );

        if( method_exists( 'Botiga_Custom_CSS', 'get_responsive_dimensions_css' ) ) {

            // Padding
            $css .= Botiga_Custom_CSS::get_responsive_dimensions_css( 
                'product_swatch_select_padding',
                array(
                    'desktop' => '{ "unit": "px", "linked": false, "top": "", "right": "", "bottom": "", "left": "" }',
                    'tablet'  => '{ "unit": "px", "linked": false, "top": "", "right": "", "bottom": "", "left": "" }',
                    'mobile'  => '{ "unit": "px", "linked": false, "top": "", "right": "", "bottom": "", "left": "" }',
                ), 
                '.variations select', 
                'padding'
            );

            // Broder Radius
            $css .= Botiga_Custom_CSS::get_dimensions_css( 
                'product_swatch_select_border_radius',
                '{ "unit": "px", "linked": false, "top": "", "right": "", "bottom": "", "left": "" }',
                '.variations select', 
                'border-radius'
            );
        }

        // Swatches Styling (Color)

        // Width
        $css .= Botiga_Custom_CSS::get_responsive_css( 'product_swatch_color_width', array( 'desktop' => 26, 'tablet' => 26, 'mobile' => 26 ), '.botiga-variations-wrapper .botiga-variation-type-color>a', 'width' );
        
        // Height
        $css .= Botiga_Custom_CSS::get_responsive_css( 'product_swatch_color_height', array( 'desktop' => 26, 'tablet' => 26, 'mobile' => 26 ), '.botiga-variations-wrapper .botiga-variation-type-color>a', 'height' );
        
        // Spacing
        $css .= Botiga_Custom_CSS::get_responsive_css( 'product_swatch_color_spacing', array( 'desktop' => 5, 'tablet' => 5, 'mobile' => 5 ), '.botiga-variations-wrapper .botiga-variation-type-color', 'grid-gap' );

        if( method_exists( 'Botiga_Custom_CSS', 'get_responsive_dimensions_css' ) ) {

            // Border Width
            $css .= Botiga_Custom_CSS::get_responsive_dimensions_css( 
                'product_swatch_color_border_width',
                array(
                    'desktop' => '{ "unit": "px", "linked": false, "top": "", "right": "", "bottom": "", "left": "" }',
                    'tablet'  => '{ "unit": "px", "linked": false, "top": "", "right": "", "bottom": "", "left": "" }',
                    'mobile'  => '{ "unit": "px", "linked": false, "top": "", "right": "", "bottom": "", "left": "" }',
                ), 
                '.botiga-variations-wrapper .botiga-variation-type-color > a', 
                'border-width'
            );

            // Border Radius
            $css .= Botiga_Custom_CSS::get_dimensions_css( 
                'product_swatch_color_border_radius',
                '{ "unit": "px", "linked": false, "top": "", "right": "", "bottom": "", "left": "" }', 
                '.botiga-variations-wrapper .botiga-variation-type-color > a, .botiga-variations-wrapper .botiga-variation-type-color > a > span', 
                'border-radius'
            );
        }

        // Border Color
        $css .= Botiga_Custom_CSS::get_border_color_css( 'product_swatch_color_border_color_default', '', '.botiga-variations-wrapper .botiga-variation-type-color > a' );

        // Border Color Hover
        $css .= Botiga_Custom_CSS::get_border_color_css( 'product_swatch_color_border_color_hover', '', '.botiga-variations-wrapper .botiga-variation-type-color > a:hover, .botiga-variations-wrapper .botiga-variation-type-color > a.active' );

        // Swatches Styling (Button)

        // Spacing
        $css .= Botiga_Custom_CSS::get_responsive_css( 'product_swatch_button_spacing', array( 'desktop' => 5, 'tablet' => 5, 'mobile' => 5 ), '.botiga-variations-wrapper .botiga-linked-variations-type-button', 'grid-gap' );

        // Text Color
        $css .= Botiga_Custom_CSS::get_color_css( 'product_swatch_button_text_color_default', '', '.botiga-variations-wrapper .botiga-variation-type-button > a' );

        // Text Color Hover
        $css .= Botiga_Custom_CSS::get_color_css( 'product_swatch_button_text_color_hover', '', '.botiga-variations-wrapper .botiga-variation-type-button > a:hover' );
        $css .= Botiga_Custom_CSS::get_color_css( 'product_swatch_button_text_color_hover', '', '.botiga-variations-wrapper .botiga-variation-type-button > a.active' );

        // Border Color
        $css .= Botiga_Custom_CSS::get_border_color_css( 'product_swatch_button_border_color_default', '', '.botiga-variations-wrapper .botiga-variation-type-button > a' );

        // Border Color Hover
        $css .= Botiga_Custom_CSS::get_border_color_css( 'product_swatch_button_border_color_hover', '', '.botiga-variations-wrapper .botiga-variation-type-button > a:hover' );
        $css .= Botiga_Custom_CSS::get_border_color_css( 'product_swatch_button_border_color_hover', '', '.botiga-variations-wrapper .botiga-variation-type-button > a.active' );

        // Background Color
        $css .= Botiga_Custom_CSS::get_background_color_css( 'product_swatch_button_background_color_default', '', '.botiga-variations-wrapper .botiga-variation-type-button > a' );

        // Background Color Hover
        $css .= Botiga_Custom_CSS::get_background_color_css( 'product_swatch_button_background_color_hover', '', '.botiga-variations-wrapper .botiga-variation-type-button > a:hover, .botiga-variations-wrapper .botiga-variation-type-button > a.active' );

        if( method_exists( 'Botiga_Custom_CSS', 'get_responsive_dimensions_css' ) ) {

            // Padding
            $css .= Botiga_Custom_CSS::get_responsive_dimensions_css( 
                'product_swatch_button_padding',
                array(
                    'desktop' => '{ "unit": "px", "linked": false, "top": "", "right": "", "bottom": "", "left": "" }',
                    'tablet'  => '{ "unit": "px", "linked": false, "top": "", "right": "", "bottom": "", "left": "" }',
                    'mobile'  => '{ "unit": "px", "linked": false, "top": "", "right": "", "bottom": "", "left": "" }',
                ), 
                '.botiga-variations-wrapper .botiga-variation-type-button > a', 
                'padding'
            );

            // Border Width
            $css .= Botiga_Custom_CSS::get_responsive_dimensions_css( 
                'product_swatch_button_border_width',
                array(
                    'desktop' => '{ "unit": "px", "linked": false, "top": "", "right": "", "bottom": "", "left": "" }',
                    'tablet'  => '{ "unit": "px", "linked": false, "top": "", "right": "", "bottom": "", "left": "" }',
                    'mobile'  => '{ "unit": "px", "linked": false, "top": "", "right": "", "bottom": "", "left": "" }',
                ), 
                '.botiga-variations-wrapper .botiga-variation-type-button > a', 
                'border-width'
            );

            // Border Radius
            $css .= Botiga_Custom_CSS::get_dimensions_css( 
                'product_swatch_button_border_radius',
                '{ "unit": "px", "linked": false, "top": "", "right": "", "bottom": "", "left": "" }', 
                '.botiga-variations-wrapper .botiga-variation-type-button > a', 
                'border-radius'
            );

        }

        // Swatches Styleling (Image)

        // Width
        $css .= Botiga_Custom_CSS::get_responsive_css( 'product_swatch_image_width', array( 'desktop' => 26, 'tablet' => 26, 'mobile' => 26 ), '.botiga-variations-wrapper .botiga-variation-type-image>a', 'width' );

        // Height
        $css .= Botiga_Custom_CSS::get_responsive_css( 'product_swatch_image_height', array( 'desktop' => 26, 'tablet' => 26, 'mobile' => 26 ), '.botiga-variations-wrapper .botiga-variation-type-image>a', 'height' );

        // Spacing
        $css .= Botiga_Custom_CSS::get_responsive_css( 'product_swatch_image_spacing', array( 'desktop' => 5, 'tablet' => 5, 'mobile' => 5 ), '.botiga-variations-wrapper .botiga-variation-type-image', 'grid-gap' );

        // Border Color
        $css .= Botiga_Custom_CSS::get_border_color_css( 'product_swatch_image_border_color_default', '', '.botiga-variations-wrapper .botiga-variation-type-image > a' );

        // Border Color Hover
        $css .= Botiga_Custom_CSS::get_border_color_css( 'product_swatch_image_border_color_hover', '', '.botiga-variations-wrapper .botiga-variation-type-image > a:hover' );
        $css .= Botiga_Custom_CSS::get_border_color_css( 'product_swatch_image_border_color_hover', '', '.botiga-variations-wrapper .botiga-variation-type-image > a.active' );

        if( method_exists( 'Botiga_Custom_CSS', 'get_responsive_dimensions_css' ) ) {

            // Border Width
            $css .= Botiga_Custom_CSS::get_responsive_dimensions_css( 
                'product_swatch_image_border_width',
                array(
                    'desktop' => '{ "unit": "px", "linked": false, "top": "", "right": "", "bottom": "", "left": "" }',
                    'tablet'  => '{ "unit": "px", "linked": false, "top": "", "right": "", "bottom": "", "left": "" }',
                    'mobile'  => '{ "unit": "px", "linked": false, "top": "", "right": "", "bottom": "", "left": "" }',
                ), 
                '.botiga-variations-wrapper .botiga-variation-type-image > a', 
                'border-width'
            );

            // Border Radius
            $css .= Botiga_Custom_CSS::get_dimensions_css( 
                'product_swatch_image_border_radius',
                '{ "unit": "px", "linked": false, "top": "", "right": "", "bottom": "", "left": "" }', 
                '.botiga-variations-wrapper .botiga-variation-type-image > a, .botiga-variations-wrapper .botiga-variation-type-image > a > span, .botiga-variations-wrapper .botiga-variation-type-image>a img', 
                'border-radius'
            );

        }

        return $css;
    }

    /**
     * Check if is a page or single post and if post content has woocommerce blocks
     */
    public function page_has_woo_blocks( $post ) {
        if( ! $post || ! is_page() && ! is_singular( 'post' ) ) {
            return false;
        }

        if( isset( $post->post_content ) && strpos( $post->post_content, 'woocommerce/' ) ) {
            return true;
        }

        return false;
    }

    /**
     * Check if is a page or single post and if post content has woocommerce shortcode
     */
    public function page_has_woo_shortcode( $post ) {
        if( ! $post || ! is_page() && ! is_singular( 'post' ) ) {
            return false;
        }

        $shortcodes = array(
            'products',
            'product_page'
        );

        if( isset( $post->post_content ) ) {
            foreach( $shortcodes as $shortcode ) {
                if( has_shortcode( $post->post_content, $shortcode ) ) {
                    return true;
                }
            }
        }

        return false;
    }

    public static function product_swatch_on_shop_catalog() {

        $button_layout = get_theme_mod( 'shop_product_add_to_cart_layout', 'layout3' );

        if ( $button_layout === 'layout1' ) {
            return;
        }

        global $product;

        if ( $product->is_type( 'variable' ) ) {
            
            remove_action( 'woocommerce_single_variation', 'woocommerce_single_variation_add_to_cart_button', 20 );
            add_filter( 'woocommerce_reset_variations_link', '__return_null' );

            echo '<div class="botiga-product-swatches" data-button-layout="'. esc_attr( $button_layout ) .'" data-button-select-text="'. esc_attr( $product->add_to_cart_text() ) .'" data-button-add-text="'. esc_attr( $product->single_add_to_cart_text() ) .'">';

                woocommerce_template_single_add_to_cart();

                $add_to_cart_text  = $product->add_to_cart_text();
                $add_to_cart_url   = $product->add_to_cart_url();
                $add_to_cart_class = 'button botiga-add-to-cart-button add_to_cart_button';

                if ( ! empty( $product->get_default_attributes() ) ) {
                    $add_to_cart_text  = $product->single_add_to_cart_text();
                    $add_to_cart_class = 'button botiga-add-to-cart-button add_to_cart_button ajax_add_to_cart';
                }

                if ( $button_layout !== 'layout3' && $button_layout !== 'layout4' ) {

                    echo sprintf(
                        '<a href="%s" class="%s">%s</a>',
                        esc_url( $add_to_cart_url ),
                        esc_attr( $add_to_cart_class ),
                        esc_html( $add_to_cart_text )
                    );

                }
        
            echo '</div>';
        
        } else {
            
            if ( $button_layout !== 'layout3' && $button_layout !== 'layout4' ) {
        
                woocommerce_template_loop_add_to_cart();
            
            }
        
        }

    }

    public function available_attributes_types() {
        if( ! wp_doing_ajax() && ( ! isset( $_GET['post'] ) && ! isset( $_GET['action'] ) ) ) {
            return array(
                'select' => __( 'Select', 'botiga-pro' ),
                'color'  => __( 'Color', 'botiga-pro' ),
                'image'  => __( 'Image', 'botiga-pro' ),
                'button' => __( 'Button', 'botiga-pro' )
            );
        }
    }

    public function add_custom_attribute_fields() {
        $attribute_taxonomies = wc_get_attribute_taxonomies();

        foreach( $attribute_taxonomies as $att_tax ) {
            add_action( "pa_{$att_tax->attribute_name}_add_form_fields", array( $this, 'add' ) );
            add_action( "pa_{$att_tax->attribute_name}_edit_form_fields", array( $this, 'edit' ) );
            add_action( "created_term", array( $this, 'save' ), 10, 3 );
            add_action( "edit_term", array( $this, 'save' ), 10, 3 );
        }
    }

    public function get_the_term_handled( $tax_slug = '' ) {
        $attribute_taxonomies = wc_get_attribute_taxonomies();

        foreach( $attribute_taxonomies as $tax ) {
            if( $tax_slug === 'pa_' . $tax->attribute_name ) {
                return $tax;
            }
        }

        return false;
    }

    public function add( $tax_slug ) {
        if( $tax = $this->get_the_term_handled( $tax_slug ) ) {
            $this->custom_attribute_field_html_content( $tax->attribute_type, $tax->attribute_label );
        }
    }

    public function edit( $term ) {
        if( $tax = $this->get_the_term_handled( $term->taxonomy ) ) {
            $this->custom_attribute_field_html_content( $tax->attribute_type, $tax->attribute_label, $term );
        }
    }

    public function custom_attribute_field_html_content( $att_type = 'select', $att_label = '', $term = false ) {
        
        // Do nothing with button type for now
        // improvements will come later
        if( $att_type === 'button' || $att_type === 'select' ) {
            return;
        }

        $this->custom_attribute_field_html_open( $term, $att_label );

        switch ( $att_type ) {
            
            case 'color':

                $color_types     = array(
                    'background' => 'Background',
                    'gradient'   => 'Gradient Background',
                    'multiple'   => 'Multiple Colors'
                );

                $directions     = array(
                    'to bottom' => '&#x2B07;',
                    'to right'  => '&#x27A1;',
                    '135deg'    => '&#x2B0A;',
                    '-135deg'   => '&#x2B0B;',
                );

                $color_type         = ( ! empty( $term ) ) ? get_term_meta( $term->term_id, 'product_attribute_color_type', true ) : '';
                $color              = ( ! empty( $term ) ) ? get_term_meta( $term->term_id, 'product_attribute_color', true ) : '';
                $gradient_color_1   = ( ! empty( $term ) ) ? get_term_meta( $term->term_id, 'product_attribute_color_gradient_color_1', true ) : '';
                $gradient_color_2   = ( ! empty( $term ) ) ? get_term_meta( $term->term_id, 'product_attribute_color_gradient_color_2', true ) : '';
                $gradient_direction = ( ! empty( $term ) ) ? get_term_meta( $term->term_id, 'product_attribute_color_gradient_direction', true ) : '';
                $multiple_color_1   = ( ! empty( $term ) ) ? get_term_meta( $term->term_id, 'product_attribute_color_multiple_color_1', true ) : '';
                $multiple_color_2   = ( ! empty( $term ) ) ? get_term_meta( $term->term_id, 'product_attribute_color_multiple_color_2', true ) : '';
                $multiple_color_3   = ( ! empty( $term ) ) ? get_term_meta( $term->term_id, 'product_attribute_color_multiple_color_3', true ) : '';
                $multiple_color_4   = ( ! empty( $term ) ) ? get_term_meta( $term->term_id, 'product_attribute_color_multiple_color_4', true ) : '';
                $multiple_direction = ( ! empty( $term ) ) ? get_term_meta( $term->term_id, 'product_attribute_color_multiple_direction', true ) : '';

                ?>
                <div class="botiga-color-field-wrapper">
                    <div class="botiga-color-field-item">
                        <div class="botiga-color-field-title"><?php esc_html_e( 'Type', 'botiga-pro' ); ?></div>
                        <select name="tag-color-type" class="botiga-color-type">
                            <?php foreach ( $color_types as $key => $value ) : ?>
                                <option value="<?php echo esc_attr( $key ); ?>"<?php selected( $color_type, $key, true ); ?>><?php echo esc_html( $value ); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <?php $hidden = ( $color_type !== 'background' && $color_type !== '' ) ? 'hidden' : ''; ?>

                    <div class="botiga-color-field-items botiga-color-type-background <?php echo sanitize_html_class( $hidden ) ?>">
                        <?php
                        ?>
                        <div class="botiga-color-field-item">
                            <div class="botiga-color-field-title"><?php esc_html_e( 'Color', 'botiga-pro' ); ?></div>
                            <input type="text" name="tag-color" value="<?php echo esc_attr( $color ); ?>" class="botiga-color-field" />
                        </div>
                    </div>

                    <?php $hidden = ( $color_type !== 'gradient' ) ? 'hidden' : ''; ?>

                    <div class="botiga-color-field-items botiga-color-type-gradient <?php echo sanitize_html_class( $hidden ) ?>">
                        <div class="botiga-color-field-item">
                            <div class="botiga-color-field-title"><?php esc_html_e( 'Color 1', 'botiga-pro' ); ?></div>
                            <input type="text" name="tag-color-gradient-color-1" value="<?php echo esc_attr( $gradient_color_1 ); ?>" class="botiga-color-field" />
                        </div>
                        <div class="botiga-color-field-item">
                            <div class="botiga-color-field-title"><?php esc_html_e( 'Color 2', 'botiga-pro' ); ?></div>
                            <input type="text" name="tag-color-gradient-color-2" value="<?php echo esc_attr( $gradient_color_2 ); ?>" class="botiga-color-field" />
                        </div>
                        <div class="botiga-color-field-item">
                            <div class="botiga-color-field-title"><?php esc_html_e( 'Direction', 'botiga-pro' ); ?></div>
                            <select name="tag-color-gradient-direction">
                                <?php foreach ( $directions as $key => $value ) : ?>
                                    <option value="<?php echo esc_attr( $key ); ?>"<?php selected( $gradient_direction, $key, true ); ?>><?php echo esc_html( $value ); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>

                    <?php $hidden = ( $color_type !== 'multiple' ) ? 'hidden' : ''; ?>

                    <div class="botiga-color-field-items botiga-color-type-multiple <?php echo sanitize_html_class( $hidden ) ?>">
                        <div class="botiga-color-field-item">
                            <div class="botiga-color-field-title"><?php esc_html_e( 'Color 1', 'botiga-pro' ); ?></div>
                            <input type="text" name="tag-color-multiple-color-1" value="<?php echo esc_attr( $multiple_color_1 ); ?>" class="botiga-color-field" />
                        </div>
                        <div class="botiga-color-field-item">
                            <div class="botiga-color-field-title"><?php esc_html_e( 'Color 2', 'botiga-pro' ); ?></div>
                            <input type="text" name="tag-color-multiple-color-2" value="<?php echo esc_attr( $multiple_color_2 ); ?>" class="botiga-color-field" />
                        </div>
                        <div class="botiga-color-field-item">
                            <div class="botiga-color-field-title"><?php esc_html_e( 'Color 3', 'botiga-pro' ); ?></div>
                            <input type="text" name="tag-color-multiple-color-3" value="<?php echo esc_attr( $multiple_color_3 ); ?>" class="botiga-color-field" />
                        </div>
                        <div class="botiga-color-field-item">
                            <div class="botiga-color-field-title"><?php esc_html_e( 'Color 4', 'botiga-pro' ); ?></div>
                            <input type="text" name="tag-color-multiple-color-4" value="<?php echo esc_attr( $multiple_color_4 ); ?>" class="botiga-color-field" />
                        </div>
                        <div class="botiga-color-field-item">
                            <div class="botiga-color-field-title"><?php esc_html_e( 'Direction', 'botiga-pro' ); ?></div>
                            <select name="tag-color-multiple-direction">
                                <?php foreach ( $directions as $key => $value ) : ?>
                                    <option value="<?php echo esc_attr( $key ); ?>"<?php selected( $multiple_direction, $key, true ); ?>><?php echo esc_html( $value ); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>

                </div>
            <?php

            break;

            case 'image':

                $image_value = ( ! empty( $term ) ) ? get_term_meta( $term->term_id, 'product_attribute_image', true ) : '';

                echo '<div class="botiga-image-field">';

                    $placeholder   = wc_placeholder_img_src( 'thumbnail' );
                    $remove_hidden = ( empty( $image_value ) ) ? ' hidden' : '';

                    if ( ! empty( $image_value ) ) {
                        $image     = wp_get_attachment_image_src( $image_value, 'thumbnail' );
                        $thumbnail = ( is_array( $image ) && ! empty( $image[0] ) ) ? $image[0] : $placeholder;
                    } else {
                        $thumbnail = $placeholder;
                    }

                    echo '<figure class="botiga-image-field-preview"><img src="'. esc_url( $thumbnail ) .'" data-placeholder="'. esc_url( $placeholder ) .'" /></figure>';
                    
                    echo '<div class="botiga-image-field-button">';
                        echo '<a href="#" class="botiga-image-field-button-upload button">'. esc_html__( 'Upload/Add Image', 'botiga-pro' ) .'</a>';
                        echo '<a href="#" class="botiga-image-field-button-remove button'. esc_attr( $remove_hidden ) .'">'. esc_html__( 'Remove Image', 'botiga-pro' ) .'</a>';
                        echo '<input type="hidden" name="tag-image" value="'. esc_attr( $image_value ) .'" class="botiga-image-field-input" />';
                    echo '</div>';
                
                echo '</div>';
                
            break;

        }

        $this->custom_attribute_field_html_close( $term );
    }
    
    public function custom_attribute_field_html_open( $term, $att_label ) {
        if( ! $term ) {
            echo '<div class="form-field">';
            echo '  <label for="attribute_type">'. esc_html( $att_label ) .'</label>';
        } else {
            echo '<tr class="form-field form-required term-name-wrap">';
            echo '    <th scope="row"><label>'. esc_html( $att_label ) .'</label></th>';
            echo '    <td>';
        }
    }

    public function custom_attribute_field_html_close( $term ) {
        if( ! $term ) {
            echo '</div>';
        } else {
            echo '    </td>';
            echo '</tr>';
        }
    }

    public function save( $term_id, $tid, $tax_slug ) {        

        if ( $tax = $this->get_the_term_handled( $tax_slug ) ) {

            switch ( $tax->attribute_type ) {

                case 'color':

                    $requests = array(
                        'tag-color'                          => 'product_attribute_color',
                        'tag-color-type'               => 'product_attribute_color_type',
                        'tag-color-gradient-color-1'   => 'product_attribute_color_gradient_color_1',
                        'tag-color-gradient-color-2'   => 'product_attribute_color_gradient_color_2',
                        'tag-color-gradient-direction' => 'product_attribute_color_gradient_direction',
                        'tag-color-multiple-color-1'   => 'product_attribute_color_multiple_color_1',
                        'tag-color-multiple-color-2'   => 'product_attribute_color_multiple_color_2',
                        'tag-color-multiple-color-3'   => 'product_attribute_color_multiple_color_3',
                        'tag-color-multiple-color-4'   => 'product_attribute_color_multiple_color_4',
                        'tag-color-multiple-direction' => 'product_attribute_color_multiple_direction',
                    );

                    foreach ( $requests as $post_key => $term_key ) {
                        $term_value = isset( $_POST[ $post_key ] ) ? sanitize_text_field( wp_unslash( $_POST[ $post_key ] ) ) : '';
                        update_term_meta( $term_id, $term_key, $term_value );
                    }

                break;
    
                case 'image':
 
                    $image = isset( $_POST['tag-image'] ) ? absint( $_POST['tag-image'] ) : '';
 
                    update_term_meta( $term_id, 'product_attribute_image', $image );
 
                break;

            }

        }

    }

    public function get_variation_by_term( $variations, $taxonomy, $slug ) {
        
        if ( ! empty( $variations ) ) {
            $attribute_key = 'attribute_'. $taxonomy;
            foreach ( $variations as $variation ) {
                if ( ! empty( $variation['attributes'][ $attribute_key ] ) && $variation['attributes'][ $attribute_key ] === $slug ) {
                    return wc_get_product( $variation['variation_id'] );
                }
            }
        }

        return false;
    
    }

    public function get_variation_id_by_term( $variations, $taxonomy, $slug ) {
        
        if ( ! empty( $variations ) ) {
            $attribute_key = 'attribute_'. $taxonomy;
            foreach ( $variations as $variation ) {
                if ( ! empty( $variation['attributes'][ $attribute_key ] ) && $variation['attributes'][ $attribute_key ] === $slug ) {
                    return $variation['variation_id'];
                }
            }
        }

        return false;
    
    }

    public function variation_attribute_options_html( $html, $args ) {
        $select_html = $html;

        $args = wp_parse_args(
            apply_filters( 'botiga_dropdown_variation_attribute_options_args', $args ),
            array(
                'options'          => false,
                'attribute'        => false,
                'product'          => false,
                'selected'         => false,
                'name'             => '',
                'id'               => '',
                'class'            => '',
                'show_option_none' => __( 'Choose an option', 'botiga-pro' ),
            )
        );

        // Get selected value.
        if ( false === $args['selected'] && $args['attribute'] && $args['product'] instanceof WC_Product ) {
            $selected_key = 'attribute_' . sanitize_title( $args['attribute'] );
            $args['selected'] = isset( $_REQUEST[ $selected_key ] ) ? wc_clean( wp_unslash( $_REQUEST[ $selected_key ] ) ) : $args['product']->get_variation_default_attribute( $args['attribute'] ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
        }

        $options   = $args['options'];
        $product   = $args['product'];
        $attribute = $args['attribute'];
        
        if ( empty( $options ) && ! empty( $product ) && ! empty( $attribute ) ) {
            $attributes = $product->get_variation_attributes();
            $options    = $attributes[ $attribute ];
        }

        $tax = $this->get_the_term_handled( $attribute );
        
        if( ! $tax ) {
            return $html;
        }

        $attribute_type = $tax->attribute_type;

        if( 'select' === $attribute_type ) {
            return $html;
        }

        $mouseover_class  = '';
        $mouseover_enable = get_theme_mod( 'product_swatch_mouseover', 0 );

        if ( $mouseover_enable ) {
            $mouseover_class = ' botiga-variations-mouseover';
        }

        $html = '<div class="botiga-variations-wrapper'. esc_attr( $mouseover_class ) .'" data-type="'. esc_attr( $attribute_type ) .'">';

        if ( ! empty( $options ) ) {
            if ( $product && taxonomy_exists( $attribute ) ) {
                // Get terms if this is a taxonomy - ordered. We need the names too.
                $terms = wc_get_product_terms(
                    $product->get_id(),
                    $attribute,
                    array(
                        'fields' => 'all',
                    )
                );
                
                $tooltip = get_theme_mod( 'product_swatch_tooltip', 0 );
                
                $variations = $product->get_available_variations();

                $html .= '<div class="botiga-variation-type-'. esc_attr( $attribute_type ) .'">';

                    if( 'color' === $attribute_type ) {

                        foreach ( $terms as $term ) {

                            if ( in_array( $term->slug, $options, true ) ) {

                                $variation = $this->get_variation_by_term( $variations, $term->taxonomy, $term->slug );

                                if ( $variation ) {

                                    $type  = ( ! empty( $term ) ) ? get_term_meta( $term->term_id, 'product_attribute_color_type', true ) : '';

                                    $style = '';

                                    if ( $type === 'gradient' ) {

                                        $color_1   = get_term_meta( $term->term_id, 'product_attribute_color_gradient_color_1', true );
                                        $color_2   = get_term_meta( $term->term_id, 'product_attribute_color_gradient_color_2', true );
                                        $direction = get_term_meta( $term->term_id, 'product_attribute_color_gradient_direction', true );

                                        $style = 'background: linear-gradient('. esc_attr( $direction ) .','. esc_attr( $color_1 ) .' 0%,'. esc_attr( $color_2 ) .' 100%);';

                                    } else if ( $type === 'multiple' ) {

                                        $colors    = array();
                                        $colors[]  = get_term_meta( $term->term_id, 'product_attribute_color_multiple_color_1', true );
                                        $colors[]  = get_term_meta( $term->term_id, 'product_attribute_color_multiple_color_2', true );
                                        $colors[]  = get_term_meta( $term->term_id, 'product_attribute_color_multiple_color_3', true );
                                        $colors[]  = get_term_meta( $term->term_id, 'product_attribute_color_multiple_color_4', true );
                                        $direction = get_term_meta( $term->term_id, 'product_attribute_color_multiple_direction', true );

                                        $colors = array_filter( $colors );

                                        if ( count( $colors ) === 1 ) {
                                            $style = sprintf( 'background-color: %s;', $colors[0] );
                                        } else if ( count( $colors ) === 2 ) {
                                            $style = sprintf( 'background: linear-gradient(%s, %s 50%%, %s 50%%);', $direction, $colors[0], $colors[1] );
                                        } else if ( count( $colors ) === 3 ) {
                                            $style = sprintf( 'background: linear-gradient(%s, %s 33.333%%, %s 33.333%% 66.666%%, %s 66.666%% 100%%);', $direction, $colors[0], $colors[1], $colors[2] );
                                        } else if ( count( $colors ) === 4 ) {
                                            $style  = sprintf( 'background: linear-gradient(135deg, %1$s, %1$s), linear-gradient(135deg, %2$s, %2$s), linear-gradient(225deg, %3$s, %3$s), linear-gradient(225deg, %4$s, %4$s);', $colors[0], $colors[1], $colors[2], $colors[3] );
                                            $style .= 'background-position: 0% 0%, 0% 100%, 100% 0%, 100% 100%;';
                                            $style .= 'background-size: 50% 50%;';
                                            $style .= 'background-repeat: no-repeat;';
                                        }

                                    } else {

                                        $color = get_term_meta( $term->term_id, 'product_attribute_color', true );
                                        $style = 'background-color: '. esc_attr( $color ) .';';

                                    }

                                    $html .= '<a href="'. esc_url( $variation->get_permalink() ) .'" role="button" class="botiga-variation-item" value="' . esc_attr( $term->slug ) . '" ' . selected( sanitize_title( $args['selected'] ), $term->slug, false ) . ' data-stock-status="'. esc_attr( $variation->get_stock_status() ) .'">';
                                    $html .= '<span class="botiga-variation-text" style="'. esc_attr( $style ) .'">';
                                    $html .= ( $tooltip ) ? '<span class="botiga-variation-tooltip">'. esc_html( $term->name ) .'</span>' : esc_html( $term->name );
                                    $html .= '</span>';
                                    $html .= '</a>';

                                }

                            }

                        }

                    }

                    if( 'button' === $attribute_type ) {
                        foreach ( $terms as $term ) {
                            if ( in_array( $term->slug, $options, true ) ) {
                                $variation = $this->get_variation_by_term( $variations, $term->taxonomy, $term->slug );
                                
                                if( $variation ) {
                                    $html .= '<a href="'. esc_url( $variation->get_permalink() ) .'" role="button" class="botiga-variation-item" data-stock-status="'. esc_attr( $variation->get_stock_status() ) .'" value="' . esc_attr( $term->slug ) . '" ' . selected( sanitize_title( $args['selected'] ), $term->slug, false ) . '>';
                                    $html .= esc_html( $term->name );
                                    $html .= '</a>';
                                }
                            }
                        }
                    }

                    if( 'image' === $attribute_type ) {
                        foreach ( $terms as $term ) {
                            if ( in_array( $term->slug, $options, true ) ) {
                                $variation = $this->get_variation_by_term( $variations, $term->taxonomy, $term->slug );

                                if( $variation ) {
                                    $image_id  = $variation->get_image_id( 'swatches' ); // use custom image
                                    $image_src = wc_placeholder_img_src( 'thumbnail' );

                                    if ( empty( $image_id ) ) {
                                        $image_id = get_term_meta( $term->term_id, 'product_attribute_image', true ); // use taxonomy default image
                                        if ( empty( $image_id ) ) {
                                            $image_id = $variation->get_image_id(); // use default image
                                        }
                                    }

                                    $image_data = wp_get_attachment_image_src( $image_id, 'thumbnail' );
                                    $image_src  = ( is_array( $image_data ) && ! empty( $image_data[0] ) ) ? $image_data[0] : $image_src;

                                    $html .= '<a href="'. esc_url( $variation->get_permalink() ) .'" role="button" class="botiga-variation-item" data-stock-status="'. esc_attr( $variation->get_stock_status() ) .'" value="' . esc_attr( $term->slug ) . '" ' . selected( sanitize_title( $args['selected'] ), $term->slug, false ) . '>';
                                    $html .= '<span class="botiga-variation-text">';
                                    $html .= '<img src="'. esc_url( $image_src ) .'" alt="'. esc_attr( $term->name ) .'" />';
                                    $html .= ( $tooltip ) ? '<span class="botiga-variation-tooltip">'. esc_html( $term->name ) .'</span>' : '';
                                    $html .= '</span>';
                                    $html .= '</a>';
                                }
                            
                            }
                        }
                    }

                $html .= '</div>';
            }
        }
            $html .= $select_html;
        $html .= '</div>';
                             
        return $html;
    }
}

new Botiga_Product_Swatch();