<?php
/**
* Product Swatch Filter Widget
*/
defined( 'ABSPATH' ) || exit;

if ( class_exists( 'Botiga_Modules' ) && ! Botiga_Modules::is_module_active( 'product-swatches' ) ) {
	return;
}

/**
* Widget active product filters class.
*/
class Botiga_Widget_Product_Swatch_Active_Filter extends WC_Widget {
	
	/**
	* Constructor.
	*/
	public function __construct() {
		
		$this->widget_id          = 'botiga_widget_product_swatch_active_filter';
		$this->widget_cssclass    = 'botiga_widget botiga_widget_product_swatch_active_filter';
		$this->widget_name        = esc_html__( 'Botiga - Active Product Filters', 'botiga-pro' );
		$this->widget_description = esc_html__( 'Show the currently active product filters.', 'botiga-pro' );
		
		parent::__construct();
		
	}
	
	/**
	* Output widget.
	*/
	public function widget( $args, $instance ) {
		
		if ( ! is_shop() && ! is_product_taxonomy() ) {
			return;
		}

		$list_items_output = '';
		$clear      = false;
		$types      = array();
		$filters    = array();
		$labels     = array();
		$attributes = array();
		$taxonomies = wc_get_attribute_taxonomies();
		
		if ( ! empty( $taxonomies ) ) {
			foreach ( $taxonomies as $tax ) {
				$query_type = 'query_type_'. $tax->attribute_name;
				$attribute  = 'filter_'. $tax->attribute_name;
				if ( isset( $_GET[ $attribute ] ) ) {
					$types[ $attribute ]      = $query_type;
					$filters[ $attribute ]    = explode( ',', wc_clean( wp_unslash( $_GET[ $attribute ] ) ) );
					$labels[ $attribute ]     = $tax->attribute_label;
					$attributes[ $attribute ] = wc_attribute_taxonomy_name( $tax->attribute_name );
				}
			}
		}

		// Min-Max filters.
		$min_price = get_query_var( 'min_price' );
		$max_price = get_query_var( 'max_price' );

		if ( $min_price !== '' || $max_price !== '' ) {
			$list_items_output .= '<li><strong>'. esc_html__( 'Price', 'botiga-pro' ) .':</strong></li>';
			if ( $min_price !== '' && $max_price === '' ) {
				$list_items_output .= '<li><a href="'. esc_url( remove_query_arg( 'min_price' ) ) .'"></a><span>'. sprintf( esc_html__( 'From %s', 'botiga-pro' ), wc_price( $min_price ) ) .'</span></li>';
			} else if ( $min_price === '' && $max_price !== '' ) {
				$list_items_output .= '<li><a href="'. esc_url( remove_query_arg( 'max_price' ) ) .'"></a><span>'. sprintf( esc_html__( 'Up to %s', 'botiga-pro' ), wc_price( $max_price ) ) .'</span></li>';
			} else if ( $min_price !== '' && $max_price !== '' ) {
				$list_items_output .= '<li><a href="'. esc_url( remove_query_arg( array( 'min_price', 'max_price' ) ) ) .'"></a><span>'. sprintf( esc_html__( 'Between %s and %s', 'botiga-pro' ), wc_price( $min_price ), wc_price( $max_price ) ) .'</span></li>';
			}
			$clear = true;
		}

		// Stock status filters.
		$filter_stock_status = get_query_var( 'filter_stock_status' );

		if ( $filter_stock_status !== '' ) {
			$list_items_output .= '<li><strong>'. esc_html__( 'Stock Status', 'botiga-pro' ) .':</strong></li>';
			$status = explode( ',', $filter_stock_status );
			foreach ( $status as $status_key => $status_value ) {
				if ( count( $status ) > 1 ) {
					$new_status = $status;
					unset( $new_status[ $status_key ] );
					$query_args = add_query_arg( 'filter_stock_status', implode( ',', $new_status ) );
				} else {
					$query_args = remove_query_arg( 'filter_stock_status' );
				}
				if ( $status_value === 'instock' ) {
					$list_items_output .= '<li><a href="'. esc_url( $query_args ) .'"></a><span>'. esc_html__( 'In Stock', 'botiga-pro' ) .'</span></li>';
				} else if ( $status_value === 'outofstock' ) {
					$list_items_output .= '<li><a href="'. esc_url( $query_args ) .'"></a><span>'. esc_html__( 'Out of Stock', 'botiga-pro' ) .'</span></li>';
				} else if ( $status_value === 'onbackorder' ) {
					$list_items_output .= '<li><a href="'. esc_url( $query_args ) .'"></a><span>'. esc_html__( 'On backorder', 'botiga-pro' ) .'</span></li>';
				}
			}
			$clear = true;
		}

		// Rating filters.
		$rating_filter = get_query_var( 'rating_filter' );

		if ( $rating_filter !== '' ) {
			$list_items_output .= '<li><strong>'. esc_html__( 'Rating', 'botiga-pro' ) .':</strong></li>';
			$ratings = explode( ',', $rating_filter );
			foreach ( $ratings as $rating_key => $rating_value ) {
				if ( count( $ratings ) > 1 ) {
					$new_ratings = $ratings;
					unset( $new_ratings[ $rating_key ] );
					$query_args = add_query_arg( 'rating_filter', implode( ',', $new_ratings ) );
				} else {
					$query_args = remove_query_arg( 'rating_filter' );
				}
				$list_items_output .= '<li><a href="'. esc_url( $query_args ) .'"></a><span>'. sprintf( esc_html__( 'Rated %s out of 5', 'botiga-pro' ), $rating_value ) .'</span></li>';
			}
			$clear = true;
		}

		// Attribute filters.
		if ( ! empty( $filters ) ) {
			foreach ( $filters as $filter => $atts ) {
				if ( ! empty( $atts ) && ! empty( $attributes[ $filter ] ) ) {
					foreach ( $atts as $attr ) {
						$term = get_term_by( 'slug', $attr, $attributes[ $filter ] );
						if ( $term ) {
							$list_items_output .= '<li><strong>'. esc_html( $labels[ $filter ] ) .':</strong></li>';
							$clear = true;
							if ( count( $atts ) > 1 ) {
								$new_atts = array_flip( $atts );
								unset( $new_atts[ $attr ] );
								$new_atts = array_flip( $new_atts );
								$query_args = add_query_arg( $filter, implode( ',', $new_atts ) );
							} else {
								$query_args = remove_query_arg( $filter );
								$query_args = remove_query_arg( $types[ $filter ], $query_args );
							}
							$list_items_output .= '<li><a href="'. esc_url( $query_args ) .'"></a><span>'. esc_html( $term->name ) .'</span></li>';
						}
					}
				}
			}
		}

		if ( $clear ) {
			$list_items_output .= '<li><a href="'. esc_url( remove_query_arg( array_merge( array_values( $types ), array_keys( $attributes ), array( 'min_price', 'max_price', 'rating_filter', 'filter_stock_status' ) ) ) ) .'" class="botiga-clear"><strong>'. esc_html__( 'Clear All', 'botiga-pro' ) .'</strong></a></li>';
		}

		if( empty( $list_items_output ) ) {
			return;
		}
		
		ob_start();
		$this->widget_start( $args, $instance );
		echo '<ul>' . $list_items_output . '</ul>';
		$this->widget_end( $args );
		echo ob_get_clean();
		
	}
	
}

/**
* Widget filter product by attribute class.
*/
class Botiga_Widget_Product_Swatch_Filter extends WC_Widget {
	
	/**
	* Constructor.
	*/
	public function __construct() {
		
		$this->widget_id          = 'botiga_widget_product_swatch_filter';
		$this->widget_cssclass    = 'botiga_widget botiga_widget_product_swatch_filter';
		$this->widget_name        = esc_html__( 'Botiga - Filter Products by Attribute', 'botiga-pro' );
		$this->widget_description = esc_html__( 'Allow customers to filter the grid by product attribute.', 'botiga-pro' );
		
		parent::__construct();
		
	}
	
	/**
	* Updates a particular instance of a widget.
	*/
	public function update( $new_instance, $old_instance ) {
		$this->init_settings();
		return parent::update( $new_instance, $old_instance );
	}
	
	/**
	* Outputs the settings update form.
	*/
	public function form( $instance ) {
		$this->init_settings();
		parent::form( $instance );
	}
	
	/**
	* Init settings after post types are registered.
	*/
	public function init_settings() {
		
		$attribute  = '';
		$attributes = array();
		$taxonomies = wc_get_attribute_taxonomies();
		
		if ( ! empty( $taxonomies ) ) {
			foreach ( $taxonomies as $taxonomy ) {
				if ( taxonomy_exists( wc_attribute_taxonomy_name( $taxonomy->attribute_name ) ) ) {
					$attributes[ $taxonomy->attribute_name ] = $taxonomy->attribute_label;
				}
			}
			$attribute = current( $attributes );
		}
		
		$this->settings = array(
			'title'   => array(
				'type'  => 'text',
				'label' => esc_html__( 'Title', 'botiga-pro' ),
				'std'   => esc_html__( 'Filter by', 'botiga-pro' ),
			),
			'attribute' => array(
				'type'    => 'select',
				'label'   => esc_html__( 'Select the attribute', 'botiga-pro' ),
				'std'     => $attribute,
				'options' => $attributes,
			),
			'allow_selecting' => array(
				'type'       => 'select',
				'label'      => esc_html__( 'Allow selecting multiple options?', 'botiga-pro' ),
				'std'        => 'multiple',
				'options'    => array(
					'multiple' => esc_html__( 'Multiple', 'botiga-pro' ),
					'single'   => esc_html__( 'Single', 'botiga-pro' ),
				),
			),
			'display_available_variations' => array(
				'type'  => 'checkbox',
				'label' => esc_html__( 'Display only available variations', 'botiga-pro' ),
				'std'   => false,
			),
		);
		
	}
	
	/**
	* Get the attribute.
	*/
	public function get_attribute_type( $taxonomy = '' ) {
		
		$attribute_taxonomies = wc_get_attribute_taxonomies();
		
		if ( ! empty( $attribute_taxonomies ) ) {
			foreach ( $attribute_taxonomies as $tax ) {
				if ( wc_attribute_taxonomy_name( $tax->attribute_name ) === $taxonomy ) {
					return $tax->attribute_type;
				}
			}
		}
		
		return '';
		
	}
	
	/**
	* Get the taxonomy.
	*/
	protected function get_instance_taxonomy( $instance ) {
		
		if ( isset( $instance['attribute'] ) ) {
			return wc_attribute_taxonomy_name( $instance['attribute'] );
		}
		
		$taxonomies = wc_get_attribute_taxonomies();
		
		if ( ! empty( $taxonomies ) ) {
			foreach ( $taxonomies as $taxonomy ) {
				if ( taxonomy_exists( wc_attribute_taxonomy_name( $taxonomy->attribute_name ) ) ) {
					return wc_attribute_taxonomy_name( $taxonomy->attribute_name );
				}
			}
		}
		
		return '';
		
	}
	
	/**
	* Output widget.
	*/
	public function widget( $args, $instance ) {
		
		if ( ! is_shop() && ! is_product_taxonomy() ) {
			return;
		}
		
		$taxonomy = $this->get_instance_taxonomy( $instance );
		
		if ( ! taxonomy_exists( $taxonomy ) ) {
			return;
		}

		$display_available_variations = ( isset( $instance['display_available_variations'] ) ) ? $instance['display_available_variations'] : false;
		
		$terms = get_terms( $taxonomy, array( 'hide_empty' => $display_available_variations ) );
		
		if ( empty( $terms ) ) {
			return;
		}
		
		$variations_output = '';
		$tooltip           = get_theme_mod( 'product_swatch_tooltip', 0 );
		$attribute         = ( isset( $instance['attribute'] ) ) ? $instance['attribute'] : '';
		$allow_selecting   = ( isset( $instance['allow_selecting'] ) ) ? $instance['allow_selecting'] : '';
		$attribute_type    = $this->get_attribute_type( $taxonomy );

		global $wp_query;

		$exists_terms = array();

		if ( ! empty( $wp_query->posts ) ) {
			foreach ( $wp_query->posts as $p ) {
				$exists_terms = array_merge( array_column( wp_get_post_terms( $p->ID, $taxonomy ), 'term_id' ), $exists_terms );
			}
		}

		$type   = 'query_type_'. $attribute;
		$filter = 'filter_'. $attribute;

		foreach ( $terms as $term ) {
			
			$query_type = ( isset( $_GET[ $type ] ) ) ? wc_clean( wp_unslash( $_GET[ $type ] ) ) : '';
			$attributes = ( isset( $_GET[ $filter ] ) ) ? explode( ',', wc_clean( wp_unslash( $_GET[ $filter ] ) ) ) : array();
			$is_active  = ( in_array( $term->slug, $attributes ) ) ? ' active' : '';

			if ( $display_available_variations && ( ! $is_active || ! empty( $exists_terms ) ) && ! in_array( $term->term_id, $exists_terms ) ) {
				continue;
			}
			
			if ( $is_active ) {
				if ( count( $attributes ) > 1 && $allow_selecting !== 'single' ) {
					$new_atts = array_flip( $attributes );
					unset( $new_atts[ $term->slug ] );
					$new_atts   = array_flip( $new_atts );
					$query_args = add_query_arg( $filter, implode( ',', $new_atts ) );
				} else {
					$query_args = remove_query_arg( $filter );
					if ( ! empty( $query_type ) ) {
						$query_args = remove_query_arg( $type, $query_args );
					}
				}
			} else {
				if ( count( $attributes ) ) {
					if ( $allow_selecting === 'single' ) {
						$query_args = add_query_arg( $filter, $term->slug );
					} else {
						$attributes[] = $term->slug;
						$query_args = add_query_arg( $filter, implode( ',', $attributes ) );
					}
				} else {
					$query_args = add_query_arg( $filter, $term->slug );
					if ( empty( $query_type ) ) {
						$query_args = add_query_arg( $type, 'or', $query_args );
					}
				}
			}
			
			if ( $attribute_type === 'color' ) {

				$color_type = ( ! empty( $term ) ) ? get_term_meta( $term->term_id, 'product_attribute_color_type', true ) : '';

				$style = '';

				if ( $color_type === 'gradient' ) {

					$color_1   = get_term_meta( $term->term_id, 'product_attribute_color_gradient_color_1', true );
					$color_2   = get_term_meta( $term->term_id, 'product_attribute_color_gradient_color_2', true );
					$direction = get_term_meta( $term->term_id, 'product_attribute_color_gradient_direction', true );

					$style = 'background: linear-gradient('. esc_attr( $direction ) .','. esc_attr( $color_1 ) .' 0%,'. esc_attr( $color_2 ) .' 100%);';

				} else if ( $color_type === 'multiple' ) {

					$colors    = array();
					$colors[]  = get_term_meta( $term->term_id, 'product_attribute_color_multiple_color_1', true );
					$colors[]  = get_term_meta( $term->term_id, 'product_attribute_color_multiple_color_2', true );
					$colors[]  = get_term_meta( $term->term_id, 'product_attribute_color_multiple_color_3', true );
					$colors[]  = get_term_meta( $term->term_id, 'product_attribute_color_multiple_color_4', true );
					$direction = get_term_meta( $term->term_id, 'product_attribute_color_multiple_direction', true );

					$colors = array_filter( $colors );

					if ( count( $colors ) === 1 ) {
						$style = sprintf( 'background-color: %s;', $colors[0] );
					} else if ( count( $colors ) === 2 ) {
						$style = sprintf( 'background: linear-gradient(%s, %s 50%%, %s 50%%);', $direction, $colors[0], $colors[1] );
					} else if ( count( $colors ) === 3 ) {
						$style = sprintf( 'background: linear-gradient(%s, %s 33.333%%, %s 33.333%% 66.666%%, %s 66.666%% 100%%);', $direction, $colors[0], $colors[1], $colors[2] );
					} else if ( count( $colors ) === 4 ) {
						$style  = sprintf( 'background: linear-gradient(135deg, %1$s, %1$s), linear-gradient(135deg, %2$s, %2$s), linear-gradient(225deg, %3$s, %3$s), linear-gradient(225deg, %4$s, %4$s);', $colors[0], $colors[1], $colors[2], $colors[3] );
						$style .= 'background-position: 0% 0%, 0% 100%, 100% 0%, 100% 100%;';
						$style .= 'background-size: 50% 50%;';
						$style .= 'background-repeat: no-repeat;';
					}

				} else {

					$color = get_term_meta( $term->term_id, 'product_attribute_color', true );
					$style = 'background-color: '. esc_attr( $color ) .';';

				}

				if ( ! empty( $style ) ) {

					$variations_output .= '<a href="'. esc_url( $query_args ) .'" role="button" class="botiga-variation-item'. esc_attr( $is_active ) .'">';
					$variations_output .= '<span class="botiga-variation-text" style="'. esc_attr( $style ) .'">';
					$variations_output .= ( $tooltip ) ? '<span class="botiga-variation-tooltip">'. esc_html( $term->name ) .'</span>' : esc_html( $term->name );
					$variations_output .= '</span>';
					$variations_output .= '</a>';

				}

			}
			
			if ( $attribute_type === 'button' || $attribute_type === 'select' ) {
				$variations_output .= '<a href="'. esc_url( $query_args ) .'" role="button" class="botiga-variation-item'. esc_attr( $is_active ) .'">';
				$variations_output .= esc_html( $term->name );
				$variations_output .= '</a>';
			}
			
			if ( $attribute_type === 'image' ) {

				$image_id = get_term_meta( $term->term_id, 'product_attribute_image', true );

				if ( ! empty( $image_id ) ) {

					$image_data = wp_get_attachment_image_src( $image_id, 'thumbnail' );
					$image_src  = ( is_array( $image_data ) && ! empty( $image_data[0] ) ) ? $image_data[0] : '';

					if ( ! empty( $image_src ) ) {

						$variations_output .= '<a href="'. esc_url( $query_args ) .'" role="button" class="botiga-variation-item'. esc_attr( $is_active ) .'">';
						$variations_output .= '<span class="botiga-variation-text">';
						$variations_output .= '<img src="'. esc_url( $image_src ) .'" alt="'. esc_attr( $term->name ) .'" />';
						$variations_output .= ( $tooltip ) ? '<span class="botiga-variation-tooltip">'. esc_html( $term->name ) .'</span>' : '';
						$variations_output .= '</span>';
						$variations_output .= '</a>';

					}

				}

			}
			
		}

		if( empty( $variations_output ) ) {
			return;
		}

		ob_start();
		$this->widget_start( $args, $instance );
		echo '<div class="botiga-widget-variations botiga-variations-wrapper">';
			echo '<div class="botiga-variation-type-'. esc_attr( $attribute_type ) .'">';
				echo $variations_output;
			echo '</div>';
		echo '</div>';
		$this->widget_end( $args );
		
	}
	
}

/**
* Register Widgets.
*/
function botiga_register_woocommerce_widgets() {
	register_widget( 'Botiga_Widget_Product_Swatch_Active_Filter' );
	register_widget( 'Botiga_Widget_Product_Swatch_Filter' );
}
add_action( 'widgets_init', 'botiga_register_woocommerce_widgets' );
