<?php
/**
 * Shopify Checkout
 * 
 * @package Botiga_Pro
 */
class Botiga_Shopify_Checkout{

    public function __construct() {

        add_action( 'wp', array( $this, 'init' ) );
        add_filter( 'woocommerce_update_order_review_fragments', array( $this, 'add_custom_checkout_fragments' ) );
        add_filter( 'woocommerce_checkout_fields', array( $this, 'add_custom_props_to_address_fields' ) );
        add_action( 'woocommerce_checkout_process', array( $this, 'apply_shipping_address_to_billing' ) );
        add_action( 'woocommerce_checkout_create_order', array( $this, 'update_shipping_phone_on_order_create' ) );

    }

    /**
     * Initalize
     */
    public function init() {

        if ( $this->is_checkout_page() || $this->is_order_received_page() ) {
            add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
        }

        if ( $this->is_checkout_page() ) {
            remove_action( 'woocommerce_before_checkout_form', 'woocommerce_checkout_coupon_form' );
            add_shortcode( 'woocommerce_checkout', array( $this, 'checkout_shortcode' ) );
        }

        if ( $this->is_order_received_page() ) {
            add_shortcode( 'woocommerce_checkout', array( $this, 'order_received_shortcode' ) );
        }

    }

    /**
     * Is checkout page and layout4
     */
    public function is_checkout_page() {
        $shop_checkout_layout = get_theme_mod( 'shop_checkout_layout', 'layout1' );
        return ( in_array( $shop_checkout_layout, array( 'layout4', 'layout5' ) ) && is_checkout() && ! is_order_received_page() && ! is_checkout_pay_page() );
    }

    /**
     * Is order received page
     */
    public function is_order_received_page() {
        
        $shop_checkout_layout = get_theme_mod( 'shop_checkout_layout', 'layout1' );

        return ( in_array( $shop_checkout_layout, array( 'layout4', 'layout5' ) ) && is_checkout() && is_order_received_page() && ! is_checkout_pay_page() );

    }

    /**
     * Check for needs shipping (in some cales shipping no needs, for eg. downloadable or virtual products)
     */
    public function is_needs_shipping() {
        return ( WC()->cart->needs_shipping() && WC()->cart->show_shipping() );
    }

    /**
     * Enqueue checkout scripts
     */
    public function enqueue_scripts() {

        wp_enqueue_script( 'jquery-validate', BOTIGA_PRO_URI . 'assets/vendor/jquery-validate/jquery.validate.min.js', array( 'jquery' ), BOTIGA_PRO_VERSION, true );

        wp_enqueue_style( 'botiga-shopify-checkout', BOTIGA_PRO_URI . 'assets/css/botiga-shopify-checkout.min.css', array(), BOTIGA_PRO_VERSION );

        wp_enqueue_script( 'botiga-shopify-checkout', BOTIGA_PRO_URI . 'assets/js/botiga-shopify-checkout.min.js', array( 'jquery' ), BOTIGA_PRO_VERSION, true );

        wp_localize_script( 'botiga-shopify-checkout', 'botiga_shopify_checkout', array(
            'i18n' => array(
                'validation' => array(
                    'required'    => esc_html__( 'This field is required.', 'botiga-pro' ),
                    'remote'      => esc_html__( 'Please fix this field.', 'botiga-pro' ),
                    'email'       => esc_html__( 'Please enter a valid email address.', 'botiga-pro' ),
                    'url'         => esc_html__( 'Please enter a valid URL.', 'botiga-pro' ),
                    'date'        => esc_html__( 'Please enter a valid date.', 'botiga-pro' ),
                    'dateISO'     => esc_html__( 'Please enter a valid date (ISO).', 'botiga-pro' ),
                    'number'      => esc_html__( 'Please enter a valid number.', 'botiga-pro' ),
                    'digits'      => esc_html__( 'Please enter only digits.', 'botiga-pro' ),
                    'equalTo'     => esc_html__( 'Please enter the same value again.', 'botiga-pro' ),
                    'maxlength'   => esc_html__( 'Please enter no more than {0} characters.', 'botiga-pro' ),
                    'minlength'   => esc_html__( 'Please enter at least {0} characters.', 'botiga-pro' ),
                    'rangelength' => esc_html__( 'Please enter a value between {0} and {1} characters long.', 'botiga-pro' ),
                    'range'       => esc_html__( 'Please enter a value between {0} and {1}.', 'botiga-pro' ),
                    'max'         => esc_html__( 'Please enter a value less than or equal to {0}.', 'botiga-pro' ),
                    'min'         => esc_html__( 'Please enter a value greater than or equal to {0}.', 'botiga-pro' ),
                    'step'        => esc_html__( 'Please enter a multiple of {0}.', 'botiga-pro' )
                ),
            ),
        ) );

    }

    /**
     * Add custom checkout fragments
     */
    public function add_custom_checkout_fragments( $fragments ) {

        if ( ! $this->is_checkout_page() ) {
            return $fragments;
        }

        $fragments['.botiga-sc-shipping-methods']     = $this->get_custom_checkout_fragment( 'shipping-methods' );
        $fragments['.botiga-sc-shipping-methods-nav'] = $this->get_custom_checkout_fragment( 'shipping-methods-nav' );
        $fragments['.botiga-sc-payment-methods']      = $this->get_custom_checkout_fragment( 'payment-methods' );
        $fragments['.botiga-sc-cart-total']           = $this->get_custom_checkout_fragment( 'cart-total' );

        return $fragments;

    }

    /**
     * Get a custom fragment
     */
    public function get_custom_checkout_fragment( $fragment ) {

        ob_start();

            switch ( $fragment ) {

                case 'shipping-methods':
                    $this->shipping_methods();
                break;

                case 'shipping-methods-nav':
                    $this->shipping_methods_nav();
                break;

                case 'payment-methods':
                    $this->payment_methods();
                break;

                case 'cart-total':
                    $this->cart_total();
                break;

            }

        return ob_get_clean();

    }

    /**
     * Apply shipping address as billing as well
     */
    public function apply_shipping_address_to_billing() {

        if ( isset( $_POST['ship_to_different_address'] ) && 'same_as_shipping' === $_POST['bill_to_different_address'] && $this->is_checkout_page() ) {
            foreach ( $_POST as $key => $value ) {
                if ( substr( $key, 0, 9 ) === 'shipping_' ) {
                    $new_key = substr_replace( $key, 'billing_', 0, 9 );
                    $_POST[ $new_key ] = $value;
                }
            }
        }

    }

    /**
     * Update shipping phone on order create
     */
    public function update_shipping_phone_on_order_create( $order ) {

        if ( isset( $_POST['ship_to_different_address'] ) && ! empty( $_POST['shipping_phone'] ) && $this->is_checkout_page() ) {
            $order->update_meta_data( '_shipping_phone', wc_clean( wp_unslash( $_POST['shipping_phone'] ) ) );
        }

    }

    /**
     * Add custom props to address fields
     */
    public function add_custom_props_to_address_fields( $fields ) {

        if ( ! $this->is_checkout_page() ) {
            return $fields;
        }

        $props = array(
            'first_name'   => array(
                'class'    => 'col-md-6',
                'priority' => 10,
            ),
            'last_name'    => array(
                'class'    => 'col-md-6',
                'priority' => 20,
            ),
            'address_1'    => array(
                'class'    => 'col-md-12',
                'priority' => 30,
            ),
            'address_2'    => array(
                'class'    => 'col-md-12',
                'priority' => 40,
            ),
            'company'      => array(
                'class'    => 'col-md-12',
                'priority' => 50,
            ),
            'country'      => array(
                'class'    => 'col-md-4',
                'priority' => 60,
            ),
            'postcode'     => array(
                'class'    => 'col-md-4',
                'priority' => 70,
            ),
            'state'        => array(
                'class'    => 'col-md-4',
                'priority' => 80,
            ),
            'city'         => array(
                'class'    => 'col-md-12',
                'priority' => 90,
            ),
            'phone'        => array(
                'class'    => 'col-md-12',
                'priority' => 100,
            ),
        );

        $wc_phone_field = get_option( 'woocommerce_checkout_phone_field', 'required' ); 

        if ( $wc_phone_field !== 'hidden' ) {
            $fields['shipping']['shipping_phone'] = $fields['billing']['billing_phone'];
        }

        foreach ( $fields as $type => $fieldset ) {

            foreach ( $fieldset as $key => $field ) {

                if ( ! is_array( $field ) ) {
                    continue;
                }

                $prop_key = str_replace( array( 'billing_', 'shipping_' ), '', $key );

                if ( ! empty( $props[ $prop_key ] ) ) {
                    $fields[ $type ][ $key ]['class'][]  = $props[ $prop_key ]['class'];
                    $fields[ $type ][ $key ]['priority'] = $props[ $prop_key ]['priority'];
                } else {
                    $fields[ $type ][ $key ]['class'][]  = 'col-md-12';
                }

                $input_class = array( 'botiga-sc-field-'. $prop_key );

                if ( ! empty( $fields[ $type ][ $key ]['input_class'] ) ) {
                    $input_class = array_merge( $fields[ $type ][ $key ]['input_class'], $input_class );
                }

                $fields[ $type ][ $key ]['input_class'] = $input_class;

                if ( $key === 'order_comments' ) {
                    $fields[ $type ][ $key ]['class'] = array_diff( $fields[ $type ][ $key ]['class'], array('notes') );
                }

            }

        }

        return $fields;

    }

    /**
     * Shipping methods
     */
    public function shipping_methods() {
        ?>
            <div class="botiga-sc-shipping-methods">
                <ul>
                    <?php
                        $packages = WC()->shipping()->get_packages();
                        if ( ! empty( $packages ) ) :
                            foreach ( $packages as $index => $package ) :
                                $checked = isset( WC()->session->chosen_shipping_methods[ $index ] ) ? WC()->session->chosen_shipping_methods[ $index ] : '';
                                $methods = $package['rates'];
                                ?>
                                <?php if ( ! empty( $methods ) ) : ?>
                                        <?php foreach ( $methods as $method ) : ?>
                                            <li class="botiga-sc-shipping-method">
                                                <label>
                                                    <?php echo sprintf( '<input type="radio" name="shipping_method[%1$s]" data-index="%1$s" data-label="%2$s" value="%3$s" %4$s />', $index, esc_attr( $method->label ), esc_attr( $method->id ), checked( $method->id, $checked, false ) ); ?>
                                                    <span><?php echo wc_cart_totals_shipping_method_label( $method ); ?></span>
                                                </label>
                                            </li>
                                        <?php endforeach; ?>
                                <?php else : ?>
                                    <li>
                                        <div class="botiga-sc-shipping-method-message"><?php echo esc_html__( 'There are no shipping options available. Please ensure that your address has been entered correctly, or contact us if you need any help.', 'botiga-pro' ); ?></div>
                                    </li>
                                <?php endif; ?>
                                <?php
                            endforeach;
                        endif;
                    ?>
                </ul>
            </div>
        <?php
    }

    /**
     * Shipping methods prev & next navigation
     */
    public function shipping_methods_nav() {

        $continue = false;
        $packages = WC()->shipping()->get_packages();

        if ( ! empty( $packages ) ) {
            foreach ( $packages as $index => $package ) {
                if ( ! empty( $package['rates'] ) ) {
                    $continue = true;
                    break;
                }
            }
        }
        ?>
            <div class="botiga-sc-shipping-methods-nav">
                <div class="botiga-sc-content-nav">
                    <a href="#" class="botiga-sc-prev">&#171; <?php echo esc_html__( 'Return to information', 'botiga-pro' ); ?></a>
                    <?php if ( ! empty( $continue ) ) : ?>
                        <a href="#" class="button botiga-sc-next"><?php echo esc_html__( 'Continue to payment', 'botiga-pro' ); ?></a>
                    <?php endif; ?>
                </div>
            </div>
        <?php
    }

    /**
     * Payment methods
     */
    public function payment_methods() {
        ?>
            <?php if ( WC()->cart->needs_payment() ) : ?>
                <div class="botiga-sc-payment-methods">
                    <ul class="wc_payment_methods payment_methods methods">
                        <?php
                            $gateways = WC()->payment_gateways()->get_available_payment_gateways();
                            WC()->payment_gateways()->set_current_gateway( $gateways );

                            if ( ! empty( $gateways ) ) :
                                foreach ( $gateways as $gateway ) :
                                    wc_get_template( 'checkout/payment-method.php', array( 'gateway' => $gateway ) );
                                endforeach;
                            else :
                            ?>
                                <li>
                                    <?php if ( WC()->customer->get_billing_country() ) : ?>
                                        <div class="woocommerce-message"><?php echo esc_html__( 'Sorry, it seems that there are no available payment methods for your state. Please contact us if you require assistance or wish to make alternate arrangements.', 'botiga-pro' ); ?></div>
                                    <?php else: ?>
                                        <div class="woocommerce-message"><?php echo esc_html__( 'Please fill in your details above to see available payment methods.', 'botiga-pro' ); ?></div>
                                    <?php endif; ?>
                                </li>
                            <?php
                            endif;
                        ?>
                    </ul>
                </div>
                <?php
                    if ( ! wp_doing_ajax() ) {
                        do_action( 'woocommerce_review_order_after_submit' );
                        do_action( 'woocommerce_review_order_after_payment' );
                    }
                ?>
            <?php endif; ?>
        <?php
    }

    /**
     * Cart total
     */
    public function cart_total() {
        ?>
            <div class="botiga-sc-cart-total">
                <table>
                    <tbody>

                        <tr>
                            <td><?php echo esc_html__( 'Subtotal', 'botiga-pro' ); ?></td>
                            <td><?php wc_cart_totals_subtotal_html(); ?></td>
                        </tr>

                        <?php foreach ( WC()->cart->get_coupons() as $code => $coupon ) : ?>
                            <tr>
                                <td><?php wc_cart_totals_coupon_label( $coupon ); ?></td>
                                <td><?php wc_cart_totals_coupon_html( $coupon ); ?></td>
                            </tr>
                        <?php endforeach; ?>

                        <?php if ( $this->is_needs_shipping() ) : ?>
                            <tr>
                                <td><?php echo esc_html__( 'Shipping', 'botiga-pro' ); ?></td>
                                <td>
                                    <?php 
                                        $total = 0;
                                        $packages = WC()->shipping->get_packages();
                                        if ( ! empty( $packages ) ) {
                                            foreach ( $packages as $index => $package ) {
                                                $methods   = ( ! empty( $package['rates'] ) ) ? $package['rates'] : array();
                                                $method_id = ( isset( WC()->session->chosen_shipping_methods[ $index ] ) ) ? WC()->session->chosen_shipping_methods[ $index ] : '';
                                                foreach ( $methods as $key => $method ) {
                                                    if ( $method->id !== $method_id ) {
                                                        continue;
                                                    }
                                                    $total += $method->cost;
                                                    if ( WC()->cart->display_prices_including_tax() ) {
                                                        $total += $method->get_shipping_tax();
                                                    }
                                                }
                                            }
                                        }
                                        echo wc_price( $total );
                                    ?>
                                </td>
                            </tr>
                        <?php endif; ?>

                        <?php foreach ( WC()->cart->get_fees() as $fee ) : ?>
                            <tr>
                                <td><?php echo esc_html( $fee->name ); ?></td>
                                <td><?php wc_cart_totals_fee_html( $fee ); ?></td>
                            </tr>
                        <?php endforeach; ?>

                        <?php if ( wc_tax_enabled() && ! WC()->cart->display_prices_including_tax() ) : ?>
                            <?php if ( get_option( 'woocommerce_tax_total_display' ) === 'itemized' ) : ?>
                                <?php foreach ( WC()->cart->get_tax_totals() as $code => $tax ) : ?>
                                    <tr>
                                        <td><?php echo esc_html( $tax->label ); ?></td>
                                        <td><?php echo wp_kses_post( $tax->formatted_amount ); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php else : ?>
                                <tr>
                                    <td><?php echo esc_html( WC()->countries->tax_or_vat() ); ?></td>
                                    <td><?php wc_cart_totals_taxes_total_html(); ?></td>
                                </tr>
                            <?php endif; ?>
                        <?php endif; ?>

                    </tbody>
                    <tfoot>
                        <tr>
                            <td><?php echo esc_html__( 'Total', 'botiga-pro' ); ?></td>
                            <td><?php wc_cart_totals_order_total_html(); ?></td>
                        </tr>
                    </tfoot>
                </table>
            </div>
        <?php
    }

    /**
     * Checkout shortcode
     */
    public function checkout_shortcode() {

        if ( ! WC()->checkout()->is_registration_enabled() && WC()->checkout()->is_registration_required() && ! is_user_logged_in() ) {
            return sprintf( '<div class="botiga-sc-must-logged-in">%s <a href="%s">%s</a></a>', esc_html__( 'You must be logged in to checkout.', 'botiga-pro' ), esc_html( wc_get_page_permalink( 'myaccount' ) ), esc_html__( 'Login', 'botiga-pro' ) );
        }

        // Get checkout object.
        $checkout = WC()->checkout();

        $shop_checkout_layout = get_theme_mod( 'shop_checkout_layout', 'layout1' );

        ob_start();
            ?>
            <div class="woocommerce">

                <div class="woocommerce-notices-wrapper"></div>

                <div class="botiga-sc botiga-sc-<?php echo esc_attr( $shop_checkout_layout ); ?>">

                    <?php do_action( 'woocommerce_before_checkout_form', $checkout ); ?>

                    <form name="checkout" method="post" class="botiga-sc-form checkout woocommerce-checkout" action="<?php echo esc_url( wc_get_checkout_url() ); ?>" enctype="multipart/form-data">

                        <div class="botiga-sc-order-review">

                            <?php if ( $shop_checkout_layout === 'layout4' ) : ?>
                                <div class="botiga-sc-breadcrumbs">
                                    <ul>
                                        <li><a href="<?php echo esc_url( wc_get_cart_url() ); ?>"><?php echo esc_html__( 'Cart', 'botiga-pro' ); ?></a></li>
                                        <li><a href="#" class="botiga-sc-breadcrumb-item active" data-content-id="information"><?php echo esc_html__( 'Information', 'botiga-pro' ); ?></a></li>
                                        <?php if ( $this->is_needs_shipping() ) : ?>
                                            <li><a href="#" class="botiga-sc-breadcrumb-item" data-content-id="shipping"><?php echo esc_html__( 'Shipping', 'botiga-pro' ); ?></a></li>
                                        <?php endif; ?>
                                        <li><a href="#" class="botiga-sc-breadcrumb-item" data-content-id="payment"><?php echo esc_html__( 'Payment', 'botiga-pro' ); ?></a></li>
                                    </ul>
                                </div>
                            <?php endif; ?>

                            <div class="botiga-sc-contents">

                                <div class="botiga-sc-content-item active" data-content-id="information">

                                    <div class="botiga-sc-content-item-block"><?php do_action( 'woocommerce_checkout_before_customer_details' ); ?></div>
                                    
                                    <div class="botiga-sc-content-item-block">
                                        <h3 class="botiga-sc-heading"><?php echo esc_html__( 'Contact Information', 'botiga-pro' ); ?></h3>
                                        <div class="botiga-sc-fields">
                                            <div class="botiga-sc-contact-information">

                                                <?php if ( is_user_logged_in() ) : ?>

                                                    <div class="botiga-sc-account-info">
                                                        <input type="hidden" name="billing_email" id="billing_email" value="<?php echo esc_attr( wp_get_current_user()->user_email ); ?>" class="botiga-sc-field-email" />
                                                        <?php echo sprintf( esc_html__( 'Welcome, %s (%s).', 'botiga-pro' ), '<strong>'. esc_html( wp_get_current_user()->display_name ) .'</strong>', esc_html( wp_get_current_user()->user_email ) ); ?>
                                                    </div>

                                                <?php else : ?>

                                                    <?php if ( get_option( 'woocommerce_enable_checkout_login_reminder' ) === 'yes' ) : ?>
                                                        <div class="botiga-sc-already-account">
                                                            <?php echo esc_html__( 'Already have an account with us?', 'botiga-pro' ); ?> <a href="<?php echo esc_html( wc_get_page_permalink( 'myaccount' ) ); ?>"><?php echo esc_html__( 'Login', 'botiga-pro' ); ?></a>
                                                        </div>
                                                    <?php endif; ?>

                                                    <div class="botiga-sc-account-email">
                                                        <div class="row">
                                                            <?php
                                                                $billing_fields = WC()->checkout()->get_checkout_fields( 'billing' );
                                                                $billing_email  = $billing_fields['billing_email'];
                                                                woocommerce_form_field( 'billing_email', $billing_email, WC()->checkout()->get_value( 'billing_email' ) );
                                                            ?>
                                                        </div>
                                                    </div>

                                                    <?php if ( WC()->checkout()->is_registration_enabled() ) : ?>
                                                        <div class="botiga-sc-create-account">
                                                            <?php if ( ! WC()->checkout()->is_registration_required() ) : ?>
                                                                <label>
                                                                    <input class="botiga-sc-create-account-input" id="createaccount" type="checkbox" name="createaccount" value="1" <?php checked( ( WC()->checkout()->get_value( 'createaccount' ) === true ), true ); ?>/>
                                                                    <?php echo esc_html__( 'Create an account?', 'botiga-pro' ); ?>
                                                                </label>
                                                            <?php endif; ?>
                                                            <?php if ( WC()->checkout()->get_checkout_fields( 'account' ) ) : ?>
                                                                <div class="botiga-sc-create-account-fields">
                                                                    <div class="row">
                                                                        <?php
                                                                            $account_fields = WC()->checkout()->get_checkout_fields( 'account' );
                                                                            foreach ( $account_fields as $key => $field ) :
                                                                                woocommerce_form_field( $key, $field, WC()->checkout()->get_value( $key ) );
                                                                            endforeach;
                                                                        ?>
                                                                    </div>
                                                                </div>
                                                            <?php endif; ?>
                                                        </div>
                                                    <?php endif; ?>

                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    </div>

                                    <div class="botiga-sc-content-item-block">
                                        <h3 class="botiga-sc-heading">
                                            <?php
                                                if ( wc_ship_to_billing_address_only() && WC()->cart->needs_shipping() ) :
                                                    echo esc_html__( 'Billing &amp; Shipping Address', 'botiga-pro' );
                                                elseif ( ! WC()->cart->needs_shipping_address() ) :
                                                    echo esc_html__( 'Billing Address', 'botiga-pro' ); 
                                                else :
                                                    echo esc_html__( 'Shipping Address', 'botiga-pro' );
                                                endif;
                                            ?>
                                        </h3>
                                        <div class="botiga-sc-fields">
                                            <div class="row">
                                                <?php
                                                    $type   = ( WC()->cart->needs_shipping_address() ) ? 'shipping' : 'billing';
                                                    $fields = WC()->checkout()->get_checkout_fields( $type );
                                                    foreach ( $fields as $key => $field ) :
                                                        if ( $key === 'billing_email' ) :
                                                            continue;
                                                        endif;
                                                        woocommerce_form_field( $key, $field, WC()->checkout()->get_value( $key ) );
                                                    endforeach;
                                                ?>
                                            </div>
                                        </div>
                                    </div>

                                    <?php if ( $shop_checkout_layout === 'layout4' ) : ?>
                                        <div class="botiga-sc-content-item-block">
                                            <div class="botiga-sc-content-nav">
                                                <a href="<?php echo esc_url( wc_get_cart_url() ); ?>" class="botiga-sc-prev botiga-sc-return-cart">&#171; <?php echo esc_html__( 'Return to cart', 'botiga-pro' ); ?></a>
                                                <?php if ( $this->is_needs_shipping() ) : ?>
                                                    <a href="#" class="button botiga-sc-next"><?php echo esc_html__( 'Continue to shipping', 'botiga-pro' ); ?></a>
                                                <?php else : ?>
                                                    <a href="#" class="button botiga-sc-next"><?php echo esc_html__( 'Continue to payment', 'botiga-pro' ); ?></a>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    <?php endif; ?>

                                </div>

                                <?php if ( $this->is_needs_shipping() ) : ?>
                                    <div class="botiga-sc-content-item" data-content-id="shipping">

                                        <?php if ( $shop_checkout_layout === 'layout4' ) : ?>
                                            <div class="botiga-sc-content-item-block">
                                                <div class="botiga-sc-detail-summary">
                                                    <table>
                                                        <tbody>
                                                            <tr>
                                                                <td><?php echo esc_html__( 'Contact', 'botiga-pro' ); ?></td>
                                                                <td><span class="botiga-sc-detail-email"></span></td>
                                                                <td><a href="#" class="botiga-sc-detail-change" data-content-id="information"><?php echo esc_html__( 'Change', 'botiga-pro' ); ?></a></td>
                                                            </tr>
                                                            <tr>
                                                                <td><?php echo esc_html__( 'Ship to', 'botiga-pro' ); ?></td>
                                                                <td><span class="botiga-sc-detail-shipping"></span></td>
                                                                <td><a href="#" class="botiga-sc-detail-change" data-content-id="information"><?php echo esc_html__( 'Change', 'botiga-pro' ); ?></a></td>
                                                            </tr>
                                                        </tbody>
                                                    </table>
                                                </div>
                                            </div>
                                        <?php endif; ?>

                                        <div class="botiga-sc-content-item-block">
                                            <h3 class="botiga-sc-heading"><?php echo esc_html__( 'Shipping Method', 'botiga-pro' ); ?></h3>
                                            <?php $this->shipping_methods(); ?>
                                        </div>

                                        <?php if ( $shop_checkout_layout === 'layout4' ) : ?>
                                            <div class="botiga-sc-content-item-block">
                                                <?php $this->shipping_methods_nav(); ?>
                                            </div>
                                        <?php endif; ?>

                                    </div>
                                <?php endif; ?>

                                <div class="botiga-sc-content-item" data-content-id="payment">

                                    <?php if ( $shop_checkout_layout === 'layout4' ) : ?>
                                        <div class="botiga-sc-content-item-block">
                                            <div class="botiga-sc-detail-summary">
                                                <table>
                                                    <tbody>
                                                        <tr>
                                                            <td><?php echo esc_html__( 'Contact', 'botiga-pro' ); ?></td>
                                                            <td><span class="botiga-sc-detail-email"></span></td>
                                                            <td><a href="#" class="botiga-sc-detail-change" data-content-id="information"><?php echo esc_html__( 'Change', 'botiga-pro' ); ?></a></td>
                                                        </tr>
                                                        <tr>
                                                            <td><?php echo esc_html__( 'Ship to', 'botiga-pro' ); ?></td>
                                                            <td><span class="botiga-sc-detail-shipping"></span></td>
                                                            <td><a href="#" class="botiga-sc-detail-change" data-content-id="information"><?php echo esc_html__( 'Change', 'botiga-pro' ); ?></a></td>
                                                        </tr>
                                                        <?php if ( $this->is_needs_shipping() ) : ?>
                                                            <tr>
                                                                <td><?php echo esc_html__( 'Method', 'botiga-pro' ); ?></td>
                                                                <td><span class="botiga-sc-detail-method"></span></td>
                                                                <td><a href="#" class="botiga-sc-detail-change" data-content-id="shipping"><?php echo esc_html__( 'Change', 'botiga-pro' ); ?></a></td>
                                                            </tr>
                                                        <?php endif; ?>
                                                    </tbody>
                                                </table>
                                            </div>
                                        </div>

                                        <div class="botiga-sc-content-item-block">
                                            <h3 class="botiga-sc-heading"><?php echo esc_html__( 'Payment Method', 'botiga-pro' ); ?></h3>
                                            <h5 class="botiga-sc-heading-sub"><?php echo esc_html__( 'All transactions are secure and encrypted.', 'botiga-pro' ); ?></h5>
                                            <?php $this->payment_methods(); ?>
                                            
                                        </div>
                                    <?php endif; ?>

                                    <?php if ( WC()->cart->needs_shipping_address() ) : ?>
                                        <div class="botiga-sc-content-item-block">
                                            
                                            <h3 class="botiga-sc-heading"><?php echo esc_html__( 'Billing Address', 'botiga-pro' ); ?></h3>
                                            <h5 class="botiga-sc-heading-sub"><?php echo esc_html__( 'Select the address that matches your card or payment method.', 'botiga-pro' ); ?></h5>

                                            <div id="ship-to-different-address">
                                                <input class="botiga-sc-different-address-input" type="checkbox" name="ship_to_different_address" value="1" checked="checked" />
                                            </div>

                                            <div class="botiga-sc-different-address">
                                                <ul>
                                                    <li>
                                                        <label>
                                                            <input type="radio" name="bill_to_different_address" value="same_as_shipping" checked="checked" />
                                                            <?php echo esc_html__( 'Same as shipping address', 'botiga-pro' ); ?>
                                                        </label>
                                                    </li>
                                                    <li>
                                                        <label>
                                                            <input type="radio" name="bill_to_different_address" value="different_from_shipping" /> 
                                                            <?php echo esc_html__( 'Use a different billing address', 'botiga-pro' ); ?>
                                                        </label>
                                                        <div class="botiga-sc-shipping-address">
                                                            <div class="botiga-sc-fields">
                                                                <div class="row">
                                                                    <?php
                                                                        $type = ( WC()->cart->needs_shipping_address() ) ? 'billing' : 'shipping';
                                                                        $fields = WC()->checkout()->get_checkout_fields( $type );
                                                                        foreach ( $fields as $key => $field ) :
                                                                            if ( $key === 'billing_email' ) :
                                                                                continue;
                                                                            endif;
                                                                            woocommerce_form_field( $key, $field, WC()->checkout()->get_value( $key ) );
                                                                        endforeach;
                                                                    ?>
                                                                </div>
                                                            </div>
                                                        </div>
                                                    </li>
                                                </ul>
                                            </div>
                                        </div>
                                    <?php endif; ?>

                                    <?php if ( apply_filters( 'woocommerce_enable_order_notes_field', 'yes' === get_option( 'woocommerce_enable_order_comments', 'yes' ) ) ) : ?>
                                        <div class="botiga-sc-content-item-block">

                                            <?php if ( ! WC()->cart->needs_shipping() || wc_ship_to_billing_address_only() ) : ?>
                                                <h3 class="botiga-sc-heading"><?php echo esc_html__( 'Additional information', 'botiga-pro' ); ?></h3>
                                            <?php endif; ?>

                                            <div class="botiga-sc-fields">
                                                <div class="row">
                                                    <?php foreach ( WC()->checkout()->get_checkout_fields( 'order' ) as $key => $field ) : ?>
                                                        <?php woocommerce_form_field( $key, $field, WC()->checkout()->get_value( $key ) ); ?>
                                                    <?php endforeach; ?>
                                                </div>
                                            </div>

                                        </div>
                                    <?php endif; ?>

                                    <?php if ( wc_privacy_policy_page_id() || wc_terms_and_conditions_page_id() ) : ?>
                                        <div class="botiga-sc-content-item-block">
                                            <?php wc_get_template( 'checkout/terms.php' ); ?>
                                        </div>
                                    <?php endif; ?>

                                    <?php if ( $shop_checkout_layout === 'layout4' ) : ?>
                                        <div class="botiga-sc-content-item-block">
                                            <div class="botiga-sc-content-nav">
                                                <?php if ( WC()->cart->needs_shipping_address() ) : ?>
                                                    <a href="#" class="botiga-sc-prev">&#171; <?php echo esc_html__( 'Return to shipping', 'botiga-pro' ); ?></a>
                                                <?php else : ?>
                                                    <a href="#" class="botiga-sc-prev">&#171; <?php echo esc_html__( 'Return to information', 'botiga-pro' ); ?></a>
                                                <?php endif; ?>
                                                <?php echo sprintf( '<button type="submit" class="button" name="woocommerce_checkout_place_order" id="place_order" value="%1$s" data-value="%1$s">%1$s</button>',  esc_html__( 'Place order', 'botiga-pro' ) ); ?>
                                                <?php wp_nonce_field( 'woocommerce-process_checkout', 'woocommerce-process-checkout-nonce' ); ?>
                                            </div>
                                        </div>
                                    <?php endif; ?>

                                </div>

                            </div>

                        </div>

                        <?php
                            $sticky_enabled = get_theme_mod( 'checkout_sticky_totals_box', 0 );
                            $sticky_class   = ( $sticky_enabled ) ? ' botiga-sc-cart-summary-sticky' : '';
                        ?>

                        <div class="botiga-sc-cart-summary<?php echo esc_attr( $sticky_class ); ?>">

                            <div class="botiga-sc-cart-summary-item">
                                <div class="botiga-sc-cart-products">
                                    <?php foreach ( WC()->cart->get_cart() as $cart_item_key => $cart_item ) : ?>
                                        <div class="botiga-sc-product">
                                            <div class="botiga-sc-product-thumbnail">
                                                <?php echo sprintf( '<figure>%s</figure>', $cart_item['data']->get_image( 'thumbnail' ) ); ?>
                                                <?php echo '<span class="botiga-sc-product-quantity">'. $cart_item['quantity'] .'</span>'; ?>
                                            </div>
                                            <div class="botiga-sc-product-name">
                                                <?php echo wp_kses_post( $cart_item['data']->get_name() ) . ' '; ?>
                                                <?php echo wc_get_formatted_cart_item_data( $cart_item ); ?>
                                            </div>
                                            <div class="botiga-sc-product-subtotal">
                                                <?php echo WC()->cart->get_product_subtotal( $cart_item['data'], $cart_item['quantity'] ); ?>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            </div>

                            <?php if ( wc_coupons_enabled() && get_theme_mod( 'shop_checkout_show_coupon_form', 1 ) ) { ?>
                                <div class="botiga-sc-cart-summary-item">
                                    <div class="botiga-sc-coupon">
                                        <input type="text" name="coupon_code" value="" class="botiga-sc-coupon-code" placeholder="<?php esc_attr_e( 'Coupon code', 'botiga-pro' ); ?>" />
                                        <a href="#" class="button botiga-sc-apply-coupon" value="<?php esc_attr_e( 'Apply coupon', 'botiga-pro' ); ?>"><?php echo esc_html__( 'Apply coupon', 'botiga-pro' ); ?></a>
                                    </div>
                                </div>
                            <?php } ?>

                            <div class="botiga-sc-cart-summary-item">
                                <?php $this->cart_total(); ?>
                            </div>

                            <?php if ( $shop_checkout_layout === 'layout5' ) : ?>
                                <div class="botiga-sc-cart-summary-item">
                                    <h3 class="botiga-sc-heading"><?php echo esc_html__( 'Payment Method', 'botiga-pro' ); ?></h3>
                                    <h5 class="botiga-sc-heading-sub"><?php echo esc_html__( 'All transactions are secure and encrypted.', 'botiga-pro' ); ?></h5>
                                    <?php $this->payment_methods(); ?>
                                    <?php wp_nonce_field( 'woocommerce-process_checkout', 'woocommerce-process-checkout-nonce' ); ?>
                                    <?php echo sprintf( '<button type="submit" class="button botiga-sc-submit-button botiga-sc-submit-button-desktop" name="woocommerce_checkout_place_order" id="place_order" value="%1$s" data-value="%1$s">%1$s</button>',  esc_html__( 'Place order', 'botiga-pro' ) ); ?>
                                </div>
                            <?php endif; ?>

                        </div>

                    </form>

                    <?php do_action( 'woocommerce_after_checkout_form', $checkout ); ?>

                </div>

            </div>
        <?php

        return ob_get_clean();

    }

    /**
     * Order received shortcode
     */
    public function order_received_shortcode() {

        $order_id  = get_query_var( 'order-received' );
        $order_key = ( ! empty( $_GET['key'] ) ) ? wc_clean( wp_unslash( $_GET['key'] ) ) : '';

        $order = wc_get_order( $order_id );

        unset( WC()->session->order_awaiting_payment );

        if ( $order && $order->is_created_via( 'admin' ) ) {
            $order->set_customer_ip_address( WC_Geolocation::get_ip_address() );
            $order->save();
        }

        if ( ! $order || ! hash_equals( $order->get_order_key(), $order_key ) ) {
            return;
        }

        ob_start();
            ?>
            <div class="woocommerce">

                <div class="botiga-sc-order-received">

                    <div class="botiga-sc-order-review">

                        <?php if ( $order->has_status( 'failed' ) ) : ?>

                            <div class="botiga-sc-order-received-item">
                                <div class="botiga-sc-order-failed">
                                    <p class="botiga-sc-order-received-text"><?php echo esc_html__( 'Unfortunately your order cannot be processed as the originating bank/merchant has declined your transaction. Please attempt your purchase again.', 'botiga-pro' ); ?></p>
                                    <p class="botiga-sc-order-received-text">
                                        <a href="<?php echo esc_url( $order->get_checkout_payment_url() ); ?>" class="button"><?php echo esc_html__( 'Pay', 'botiga-pro' ); ?></a>
                                        <?php if ( is_user_logged_in() ) : ?>
                                            <a href="<?php echo esc_url( wc_get_page_permalink( 'myaccount' ) ); ?>" class="button"><?php echo esc_html__( 'My account', 'botiga-pro' ); ?></a>
                                        <?php endif; ?>
                                    </p>
                                </div>
                            </div>

                        <?php else : ?>

                            <div class="botiga-sc-order-received-item">
                                <p class="botiga-sc-order-received-text"><small><?php echo sprintf( esc_html__( 'Order %s', 'botiga-pro' ), '#'. $order->get_order_number() ); ?></small></p>
                                <h3 class="botiga-sc-order-received-title"><?php echo sprintf( esc_html__( 'Thank You %s!', 'botiga-pro' ), $order->get_billing_first_name() ); ?></h3>
                            </div>

                            <div class="botiga-sc-order-received-item">
                                <div class="botiga-sc-order-received-box">
                                    <h3 class="botiga-sc-order-received-title"><?php echo esc_html__( 'Order Updates', 'botiga-pro' ); ?></h3>
                                    <p class="botiga-sc-order-received-text"><?php echo esc_html__( 'You’ll get shipping and delivery updates by email.', 'botiga-pro' ); ?></p>
                                </div>
                            </div>

                            <div class="botiga-sc-order-received-item">
                                <div class="botiga-sc-order-received-box">
                                    <div class="botiga-sc-order-received-detail">
                                        <?php if ( is_user_logged_in() && $order->get_user_id() === get_current_user_id() && $order->get_billing_email() ) : ?>
                                            <div class="row">
                                                <div class="col-md-12">
                                                    <h3 class="botiga-sc-order-received-title"><?php echo esc_html__( 'Contact Information', 'botiga-pro' ); ?></h3>
                                                    <div><?php echo esc_html( $order->get_billing_email() ); ?></div>
                                                </div>
                                            </div>
                                        <?php endif; ?>
                                        <div class="row">
                                            <?php $show_shipping = ( ! wc_ship_to_billing_address_only() && $order->needs_shipping_address() ); ?>
                                            <?php if ( $show_shipping ) : ?>
                                                <div class="col-md-6">
                                            <?php else : ?>
                                                <div class="col-md-12">
                                            <?php endif; ?>
                                                <h3 class="botiga-sc-order-received-title"><?php echo esc_html__( 'Billing Address', 'botiga-pro' ); ?></h3>
                                                <?php echo wp_kses_post( $order->get_formatted_billing_address( esc_html__( 'N/A', 'botiga-pro' ) ) ); ?>
                                                <?php if ( $order->get_billing_phone() ) : ?>
                                                    <br /><?php echo esc_html( $order->get_billing_phone() ); ?>
                                                <?php endif; ?>
                                            </div>
                                            <?php if ( $show_shipping ) : ?>
                                                <div class="col-md-6">
                                                <h3 class="botiga-sc-order-received-title"><?php echo esc_html__( 'Shipping Address', 'botiga-pro' ); ?></h3>
                                                <?php echo wp_kses_post( $order->get_formatted_shipping_address( esc_html__( 'N/A', 'botiga-pro' ) ) ); ?>
                                                <?php if ( $order->get_shipping_phone() ) : ?>
                                                    <br /><?php echo esc_html( $order->get_shipping_phone() ); ?>
                                                <?php endif; ?>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                        <div class="row">
                                            <?php if ( $order->get_payment_method_title() ) : ?>
                                                <?php if ( $order->needs_shipping_address() ) : ?>
                                                    <div class="col-md-6">
                                                <?php else: ?>
                                                    <div class="col-md-12">
                                                <?php endif; ?>
                                                    <h3 class="botiga-sc-order-received-title"><?php echo esc_html__( 'Payment Method', 'botiga-pro' ); ?></h3>
                                                    <p class="botiga-sc-order-received-text"><?php echo wp_kses_post( $order->get_payment_method_title() ); ?></p>
                                                </div>
                                            <?php endif; ?>
                                            <?php if ( $order->needs_shipping_address() ) : ?>
                                                <?php if ( $order->get_payment_method_title() ) : ?>
                                                    <div class="col-md-6">
                                                <?php else: ?>
                                                    <div class="col-md-12">
                                                <?php endif; ?>
                                                    <h3 class="botiga-sc-order-received-title"><?php echo esc_html__( 'Shipping Method', 'botiga-pro' ); ?></h3>
                                                    <p class="botiga-sc-order-received-text"><?php echo wp_kses_post( $order->get_shipping_method() ); ?></p>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <div class="botiga-sc-order-received-item">
                                <div class="botiga-sc-order-received-continue">
                                    <a href="<?php echo esc_url( wc_get_page_permalink( 'shop' ) ); ?>" class="button"><?php echo esc_html__( 'Continue Shopping', 'botiga-pro' ); ?></a>
                                </div>
                            </div>

                        <?php endif; ?>
                    </div>

                    <div class="botiga-sc-cart-summary">

                        <div class="botiga-sc-cart-summary-item">
                            <div class="botiga-sc-cart-products">
                                <?php $order_items = $order->get_items(); ?>
                                <?php foreach ( $order_items as $order_item ) : ?>
                                    <?php $order_product = $order_item->get_product(); ?>
                                    <div class="botiga-sc-product">
                                        <div class="botiga-sc-product-thumbnail">
                                            <?php echo sprintf( '<figure>%s</figure>', $order_product->get_image( 'thumbnail' ) ); ?>
                                            <?php echo '<span class="botiga-sc-product-quantity">'. $order_item->get_quantity() .'</span>'; ?>
                                        </div>
                                        <div class="botiga-sc-product-name">
                                            <?php echo wp_kses_post( $order_product->get_name() ) . ' '; ?>
                                            <?php echo wc_display_item_meta( $order_item ); ?>
                                        </div>
                                        <div class="botiga-sc-product-subtotal">
                                            <?php echo $order->get_formatted_line_subtotal( $order_item ); ?>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        </div>

                        <div class="botiga-sc-cart-summary-item">
                            <div class="botiga-sc-cart-total">
                                <table>
                                    <tbody>
                                        <?php $item_totals = $order->get_order_item_totals(); ?>
                                        <?php if ( ! empty( $item_totals ) ) : ?>
                                            <?php foreach ( $item_totals as $key => $item_total ) : ?>
                                                <?php
                                                    if ( in_array( $key, array( 'payment_method', 'order_total' ) ) ) {
                                                        continue;
                                                    }
                                                    if ( $key === 'shipping' ) {
                                                        $item_total['value'] = preg_replace( '/&nbsp;\<small.*?\>.*?\<\/small\>/', '', $item_total['value'] );
                                                    }
                                                ?>
                                                <tr>
                                                    <td><?php echo wp_kses_post( $item_total['label'] ); ?></td>
                                                    <td><?php echo wp_kses_post( $item_total['value'] ); ?></td>
                                                </tr>
                                            <?php endforeach; ?>
                                        <?php endif; ?>
                                    </tbody>
                                    <tfoot>
                                        <tr>
                                            <td><?php echo esc_html__( 'Total', 'botiga-pro' ); ?></td>
                                            <td><?php echo wc_price( $order->get_total() ); ?></td>
                                        </tr>
                                    </tfoot>
                                </table>
                            </div>
                        </div>

                    </div>

                </div>
            </div>
        <?php

        return ob_get_clean();

    }

}

new Botiga_Shopify_Checkout();
