<?php
/**
 * WooCommerce Single Product Related Products
 * Templates Builder Block
 * 
 * @package Botiga_Pro
 */

/**
 * Registers the block using the metadata loaded from the `block.json` file.
 * Behind the scenes, it registers also all assets so they can be enqueued
 * through the block editor in the corresponding context.
 *
 * @see https://developer.wordpress.org/block-editor/tutorials/block-tutorial/writing-your-first-block-type/
 */

function botiga_register_wc_products_block() {
    register_block_type_from_metadata( 
        BOTIGA_PRO_DIR . '/inc/modules/templates-builder/components/gutenberg/build/wc-products/block.json',
        array(
            'render_callback' => 'botiga_wc_products_block_callback'
        )
    );
}
add_action( 'init', 'botiga_register_wc_products_block', 11 );

/**
 * Render callback
 * 
 */
function botiga_wc_products_block_callback( $block_atts ) {
    $blockId    = $block_atts['blockId'];
    $custom_css = Botiga_Templates_Builder_Gutenberg_Helper::get_block_custom_css( $block_atts );
    $edit_mode  = is_shop() || is_product_category() || is_product_tag() || is_product_taxonomy() || is_singular( 'product' ) || is_singular( 'athemes_hf' ) ? false : true;

    ob_start();
    
    $post    = Botiga_Templates_Builder_Gutenberg_Helper::get_first_post();

    $limit   = absint( $block_atts['columns'] ) * absint( $block_atts['rows'] );
    $orderby = $block_atts['queryOrderBy'];
    $order   = $block_atts['queryOrder'];

    if( $block_atts[ 'queryType' ] === 'all-products' ) {
        $args = array(
            'post_type' => 'product',
            'posts_per_page' => $limit,
            'orderby' => $orderby,
            'order' => $order,		
            'paged' => get_query_var( 'paged' ) ? get_query_var( 'paged' ) : 1,
            'fields' => 'ids'
        );

        global $taxonomy;
        if( $taxonomy !== NULL ) {
            $args['tax_query'] = array(
                array(
                    'taxonomy' => $taxonomy,
                    'field'    => 'slug',
                    'terms'    => get_queried_object()->slug,
                ),
            );
        }

        $products_query = new \WP_Query( $args );
        $products       = $products_query->posts;
    } else if( $block_atts[ 'queryType' ] === 'related-products' ) {
        $related = wc_products_array_orderby( array_filter( array_map( 'wc_get_product', wc_get_related_products( $post->ID, -1 ) ), 'wc_products_array_filter_visible' ), $orderby, $order );
        $related = $limit > 0 ? array_slice( $related, 0, $limit ) : $related;

        $products = $related;
    } else if( $block_atts[ 'queryType' ] === 'upsell-products' ) {
        $product = wc_get_product( $post->ID );

        $upsells = wc_products_array_orderby( array_filter( array_map( 'wc_get_product', $product->get_upsell_ids() ), 'wc_products_array_filter_visible' ), $orderby, $order );
        $upsells = $limit > 0 ? array_slice( $upsells, 0, $limit ) : $upsells;

        $products = $upsells;
    }

    if( ! empty( $products ) ) { 
        
        // Sorting area
        $sorting_area_output = array();

        if( $block_atts[ 'queryType' ] === 'all-products' ) {

            // Results Count
            if( $block_atts[ 'displayResultCount' ] ) {
                $args = array(
                    'total'    => $products_query->found_posts,
                    'per_page' => $limit,
                    'current'  => $products_query->get( 'paged' ) ? $products_query->get( 'paged' ) : 1
                );

                ob_start();
                botiga_woocommerce_result_count( $args );
                $sorting_area_output[] = ob_get_clean();
            }

            // Ordering
            if( $block_atts[ 'displayOrdering' ] ) {
                ob_start();
                botiga_woocommerce_catalog_ordering();
                $sorting_area_output[] = ob_get_clean();
            }

            // Grid & List View Toggle
            if( $block_atts[ 'displayGridListToggle' ] ) {
                wp_enqueue_script( 'botiga-grid-list-view', BOTIGA_PRO_URI . 'assets/js/botiga-grid-list-view.min.js', array(), BOTIGA_PRO_VERSION, true );

                global $product;
                ob_start();
                botiga_shop_grid_list_view_buttons( $product );
                $sorting_area_output[] = ob_get_clean();
            }

        }

        if( ! empty( $sorting_area_output ) ) { 
            echo '<div class="woocommerce-sorting-wrapper">';
                echo '<div class="row">';
                    foreach( $sorting_area_output as $sort_item ) {

                        echo '<div class="col-auto">';
                            echo $sort_item;
                        echo '</div>';

                    }
                echo '</div>';
            echo '</div>';
        }
        
        ?>

        <ul class="products columns-<?php echo esc_attr( $block_atts[ 'columns' ] ) ?>">

        <?php
        foreach( $products as $product_id ) {
            $layout   = isset( $_GET[ 'view' ] ) && $_GET[ 'view' ] === 'product-list' ? 'product-list' : 'grid';
            $_product = wc_get_product( $product_id ); ?>
    
            <li <?php wc_product_class( '', $_product ); ?>>
                <?php if( $block_atts[ 'displayImage' ] ) : ?>
                    <div class="loop-image-wrap botiga-add-to-cart-button-layout2 botiga-quick-view-button-layout2">
                        <a href="<?php echo esc_url( get_the_permalink( $product_id ) ); ?>" class="woocommerce-LoopProduct-link woocommerce-loop-product__link">
                            <?php echo wp_kses_post( $_product->get_image() ); ?>
                        </a>
                        <?php 
                        if( $block_atts[ 'displayQuickView' ] ) {
                            botiga_quick_view_button( $_product );
                        }

                        if( $block_atts[ 'displaySaleBadge' ] ) {
                            if ( $_product->is_on_sale() ) {
                                echo apply_filters( 'woocommerce_sale_flash', '<span class="onsale">' . esc_html__( 'Sale!', 'botiga-pro' ) . '</span>', $post, $_product );
                            }
                        }

                        if( $block_atts[ 'displayWishlist' ] ) {
                            botiga_wishlist_button( $_product );
                        }
                        
                        ?>
                    </div>
                <?php endif;

                if( $layout === 'product-list' ) {
                    echo '<div class="product-data-wrapper">';
                }

                if( $block_atts[ 'displayTitle' ] ) {
                    echo '<h2 class="' . esc_attr( apply_filters( 'woocommerce_product_loop_title_classes', 'woocommerce-loop-product__title' ) ) . '">' . $_product->get_name() . '</h2>'; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
                }

                if( $block_atts[ 'displayPrice' ] ) {
                    if ( $price_html = $_product->get_price_html() ) : ?>
                        <span class="price"><?php echo $price_html; ?></span>
                    <?php endif;
                }

                if( $block_atts[ 'displayCategories' ] ) {
                    echo '<div class="product-category">' . wc_get_product_category_list( $_product->get_id() ) . '</div>'; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
                }
                
                if( $block_atts[ 'displayRating' ] ) {
                    if ( wc_review_ratings_enabled() ) {

                        $rating_count = $_product->get_rating_count();
                        $average      = $_product->get_average_rating();
                        
                        if ( $rating_count > 0 ) : ?>
                        
                            <div class="woocommerce-product-rating">
                                <?php echo wc_get_rating_html( $average, $rating_count ); // WPCS: XSS ok. ?>
                            </div>
                        
                        <?php endif;
                    }
                }

                if( $block_atts[ 'displayShortDescription' ] ) {
                    echo '<div class="product-description">' . wp_kses_post( wp_trim_words( get_the_excerpt( $_product->get_id() ), 12, '&hellip;' ) ) . '</div>';
                }

                if( $block_atts[ 'displayStock' ] ) {
                    botiga_add_product_stock_to_loop_items( $_product );
                }

                if( $block_atts[ 'displayAddToCart' ] ) {
                    if ( $_product ) {
                        $defaults = array(
                            'quantity'   => 1,
                            'class'      => implode(
                                ' ',
                                array_filter(
                                    array(
                                        'button',
                                        'button', // escaped in the template.
                                        'product_type_' . $_product->get_type(),
                                        $_product->is_purchasable() && $_product->is_in_stock() ? 'add_to_cart_button' : '',
                                        $_product->supports( 'ajax_add_to_cart' ) && $_product->is_purchasable() && $_product->is_in_stock() ? 'ajax_add_to_cart' : '',
                                    )
                                )
                            ),
                            'attributes' => array(
                                'data-product_id'  => $_product->get_id(),
                                'data-product_sku' => $_product->get_sku(),
                                'aria-label'       => $_product->add_to_cart_description(),
                                'rel'              => 'nofollow',
                            ),
                        );
            
                        $args = apply_filters( 'woocommerce_loop_add_to_cart_args', $defaults, $_product );
            
                        if ( isset( $args['attributes']['aria-label'] ) ) {
                            $args['attributes']['aria-label'] = wp_strip_all_tags( $args['attributes']['aria-label'] );
                        }
            
                        echo sprintf(
                            '<a href="%s" data-quantity="%s" class="%s" %s>%s</a>',
                            esc_url( $_product->add_to_cart_url() ),
                            esc_attr( isset( $args['quantity'] ) ? $args['quantity'] : 1 ),
                            esc_attr( isset( $args['class'] ) ? $args['class'] : 'button' ),
                            isset( $args['attributes'] ) ? wc_implode_html_attributes( $args['attributes'] ) : '',
                            esc_html( $_product->add_to_cart_text() )
                        );
                    }
                } 
                
                if( $layout === 'product-list' ) {
                    echo '</div>';
                } ?>
                
            </li>
    
            <?php
        } ?>

        </ul>

        <?php
        if( $block_atts[ 'queryType' ] === 'all-products' ) {

            // Pagination
            if( $block_atts[ 'displayPagination' ] ) {
                $args = array(
                    'total'   => $products_query->max_num_pages,
                    'current' => $products_query->get( 'paged' ) ? $products_query->get( 'paged' ) : 1,
                    'base'    => str_replace( 999999999, '%#%', esc_url( get_pagenum_link( 999999999 ) ) ),
                    'format'  => '/page/%#%',
                );

                botiga_woocommerce_pagination( $args );
            }
        
        }

    }

    $output = ob_get_clean();

    return sprintf(
        '<div class="%1$s botiga-tb-block botiga-tb-wc-products"><style>%3$s</style>%2$s</div>',
        $blockId,
        $output,
        $custom_css
    );
}