import classnames from 'classnames';
import { __ } from '@wordpress/i18n';
import { useState } from '@wordpress/element';
import ServerSideRender from '@wordpress/server-side-render';
import { PanelBody, SelectControl, Disabled } from '@wordpress/components';
import { BotigaDimensions, BotigaTypography, BotigaColorPicker, BotigaTabsNavigation, BotigaTabsContent, BotigaBorder } from '../_inc/global-components';
import { InspectorControls, useBlockProps } from '@wordpress/block-editor';
import './editor.scss';
import { generateBlockCustomCss } from '../_inc/helpers';

/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {WPElement} Element to render.
 */
export default function Edit( props ) {
	const { name, attributes, setAttributes } = props;

	const { 
		customCss
	} = attributes;

	// Block ID
	const blockId = `botiga-tb-block-${ props.clientId.substr( 0, 8 ) }`;
	setAttributes( { blockId: blockId } );

	// Block Props
	const blockProps = useBlockProps( {
		className: classnames( 'botiga-tb-block', 'botiga-tb-wc-notices' )
	} );

	// Tab Navigation
	const [ tab, setTab ] = useState( 'style' );

	// Defaults
	const defaults = {
		align: 'left',
	}

	if( typeof customCss.align !== 'undefined' ) {
        defaults.align = customCss.align.value;
    }

	// Custom CSS
	generateBlockCustomCss( blockId, customCss );

	return (
		<div { ...blockProps }>
		
			<InspectorControls>
				<BotigaTabsNavigation
					exclude="settings"
					activeTab={ tab }
					setActiveTab={ setTab } />

				<BotigaTabsContent 
					tabid="style"
					activeTab={ tab }>
					
					<PanelBody title={ __( 'General', 'botiga-pro' ) } initialOpen={ true }>
						<BotigaTypography 
							label={ __( 'Typography', 'botiga-pro' ) }
							idPrefix={ 'typography' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .woocommerce-notices-wrapper` }
						/>
						<SelectControl
							label={ __( 'Alignment', 'botiga-pro' ) }
							value={ defaults.align }
							options={ [
								{ label: 'Left', value: 'left' },
								{ label: 'Center', value: 'center' },
								{ label: 'Right', value: 'right' },
							] }
							onChange={ ( value ) => { 
								const css = {
									align: {
										selector: `{{WRAPPER}} .woocommerce-notices-wrapper`,
										props: `text-align: ${ value };`,
										value: value
									},
								};
								setAttributes( { customCss: { ...customCss, ...css } } ) 
							} }
						/>
						<BotigaColorPicker 
							label={ __( 'Text Color', 'botiga-pro' ) }
							idPrefix={ 'text_color' }
							attributes={ attributes }
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .woocommerce-notices-wrapper .woocommerce-message` }
						/>
						<BotigaColorPicker 
							label={ __( 'Background Color', 'botiga-pro' ) }
							idPrefix={ 'background_color' }
							attributes={ attributes }
							setAttributes={ setAttributes }
							cssProp={ `background-color` }
							customCssSelector={ `{{WRAPPER}} .woocommerce-notices-wrapper .woocommerce-message, {{WRAPPER}} .woocommerce-notices-wrapper .woocommerce-error` }
						/>
					</PanelBody>
					<PanelBody title={ __( 'Buttons', 'botiga-pro' ) } initialOpen={ false }>
						<BotigaTypography 
							label={ __( 'Typography', 'botiga-pro' ) }
							idPrefix={ 'button_typography' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .button` }
						/>
						<BotigaColorPicker
							label={ __( 'Text Color', 'botiga-pro' ) }
							idPrefix={ 'button_color' }
							withHover={ true }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							cssProp={ `color` }
							customCssSelector={ `{{WRAPPER}} .button` }
						/>
						<BotigaColorPicker
							label={ __( 'Background Color', 'botiga-pro' ) }
							idPrefix={ 'button_background_color' }
							withHover={ true }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							cssProp={ `background-color` }
							customCssSelector={ `{{WRAPPER}} .button` }
						/>
						<BotigaBorder 
							label={ __( 'Border', 'botiga-pro' ) }
							idPrefix={ 'button_border' }
							defaults={{
								borderType: 'none',
								borderWidth: {
									top: 0,
									left: 0,
									right: 0,
									bottom: 0
								},
								borderColor: '#CCC'
							}}
							attributes={ attributes } 
							setAttributes={ setAttributes }
							cssProp={ `border-color` }
							customCssSelector={ `{{WRAPPER}} .button` }
						/>
						<BotigaBorder 
							label={ __( 'Border (hover)', 'botiga-pro' ) }
							idPrefix={ 'button_border_hover' }
							defaults={{
								borderType: 'none',
								borderWidth: {
									top: 0,
									left: 0,
									right: 0,
									bottom: 0
								},
								borderColor: '#CCC'
							}}
							attributes={ attributes } 
							setAttributes={ setAttributes }
							cssProp={ `border-color` }
							customCssSelector={ `{{WRAPPER}} .button:hover` }
						/>
						<BotigaDimensions
							label={ __( 'Border Radius', 'botiga-pro' ) }
							idPrefix={ 'button_border_radius' }
							type={ 'border-radius' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .button` } 
						/>
						<BotigaDimensions
							label={ __( 'Padding', 'botiga-pro' ) }
							idPrefix={ 'button_padding' }
							type={ 'padding' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .button` } 
						/>
					</PanelBody>
				</BotigaTabsContent>
				<BotigaTabsContent 
					tabid="advanced"
					activeTab={ tab }>
					
					<PanelBody title={ __( 'Layout', 'botiga-pro' ) } initialOpen={ true }>
						<BotigaDimensions
							label={ __( 'Margin', 'botiga-pro' ) }
							idPrefix={ '' }
							type={ 'margin' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}}` } />

						<BotigaDimensions
							label={ __( 'Padding', 'botiga-pro' ) }
							idPrefix={ '' }
							type={ 'padding' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}}` } />
					</PanelBody>

				</BotigaTabsContent>
			</InspectorControls>

			<Disabled>
				<ServerSideRender
					block={ name }
					httpMethod="POST"
					attributes={ attributes }
				/>
			</Disabled>

		</div>
	);
}
