import classnames from 'classnames';
import { __ } from '@wordpress/i18n';
import { useState } from '@wordpress/element';
import ServerSideRender from '@wordpress/server-side-render';
import { PanelBody, SelectControl, __experimentalBoxControl as BoxControl, Disabled } from '@wordpress/components';
import { BotigaDimensions, BotigaTypography, BotigaColorPicker, BotigaTabsNavigation, BotigaTabsContent } from '../_inc/global-components';
import { InspectorControls, useBlockProps } from '@wordpress/block-editor';
import './editor.scss';
import { generateBlockCustomCss } from '../_inc/helpers';

/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {WPElement} Element to render.
 */
export default function Edit( props ) {
	const { name, attributes, setAttributes } = props;

	const { 
		customCss
	} = attributes;

	// Block ID
	const blockId = `botiga-tb-block-${ props.clientId.substr( 0, 8 ) }`;
	setAttributes( { blockId: blockId } );

	// Block Props
	const blockProps = useBlockProps( {
		className: classnames( blockId, 'botiga-tb-block', 'botiga-tb-sp-rating' )
	} );

	// Tab Navigation
	const [ tab, setTab ] = useState( 'style' );

	// Defaults
	const defaults = {
		align: 'flex-start'
	}

	if( typeof customCss.align !== 'undefined' ) {
        defaults.align = customCss.align.value;
    }

	// Custom CSS
	generateBlockCustomCss( blockId, customCss );

	return (
		<div { ...blockProps }>
			<InspectorControls>
				<BotigaTabsNavigation
					exclude="settings"
					activeTab={ tab }
					setActiveTab={ setTab } />

				<BotigaTabsContent 
					tabid="style"
					activeTab={ tab }>
					
					<PanelBody title={ __( 'General', 'botiga-pro' ) } initialOpen={ true }>
						<BotigaTypography 
							label={ __( 'Typography', 'botiga-pro' ) }
							idPrefix={ '' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .woocommerce-product-rating, {{WRAPPER}} .woocommerce-product-rating a` }
						/>
						<SelectControl
							label={ __( 'Alignment', 'botiga-pro' ) }
							value={ defaults.align }
							options={ [
								{ label: 'Left', value: 'flex-start' },
								{ label: 'Center', value: 'center' },
								{ label: 'Right', value: 'flex-end' },
							] }
							onChange={ ( value ) => { 
								const css = {
									align: {
										selector: `{{WRAPPER}} .woocommerce-product-rating`,
										props: `justify-content: ${ value };`,
										value: value
									}
								};
								setAttributes( { customCss: { ...customCss, ...css } } ) 
							} }
						/>
						<BotigaColorPicker
							label={ __( 'Text Color', 'botiga-pro' ) }
							idPrefix={ 'text' }
							withHover={ false }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							customCssSelector={ `{{WRAPPER}} .woocommerce-product-rating` }
						/>
						<BotigaColorPicker
							label={ __( 'Links Color', 'botiga-pro' ) }
							idPrefix={ 'links' }
							withHover={ true }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							customCssSelector={ `{{WRAPPER}} .woocommerce-product-rating a` }
						/>
						<BotigaColorPicker
							label={ __( 'Rating Icon Background Color', 'botiga-pro' ) }
							idPrefix={ 'ratingIconBackground' }
							withHover={ false }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							customCssSelector={ `{{WRAPPER}} .woocommerce-product-rating .star-rating::before` }
						/>
						<BotigaColorPicker
							label={ __( 'Rating Icon Color', 'botiga-pro' ) }
							idPrefix={ 'ratingIconColor' }
							withHover={ false }
							attributes={ attributes } 
							setAttributes={ setAttributes } 
							customCssSelector={ `{{WRAPPER}} .woocommerce-product-rating .star-rating span::before` }
						/>
					</PanelBody>

				</BotigaTabsContent>
				<BotigaTabsContent 
					tabid="advanced"
					activeTab={ tab }>
					
					<PanelBody title={ __( 'Layout', 'botiga-pro' ) } initialOpen={ true }>
						<BotigaDimensions
							label={ __( 'Margin', 'botiga-pro' ) }
							idPrefix={ 'title' }
							type={ 'margin' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}}` } />

						<BotigaDimensions
							label={ __( 'Padding', 'botiga-pro' ) }
							idPrefix={ 'title' }
							type={ 'padding' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}}` } />
					</PanelBody>

				</BotigaTabsContent>
			</InspectorControls>

			<Disabled>
				<ServerSideRender
					block={ name }
					httpMethod="POST"
					attributes={ attributes }
				/>
			</Disabled>
		</div>
	);
}
