import classnames from 'classnames';
import { __ } from '@wordpress/i18n';
import { useState } from '@wordpress/element';
import ServerSideRender from '@wordpress/server-side-render';
import { PanelBody, SelectControl, Disabled } from '@wordpress/components';
import { BotigaDimensions, BotigaTypography, BotigaColorPicker, BotigaTabsNavigation, BotigaTabsContent, BotigaBorder } from '../_inc/global-components';
import { InspectorControls, useBlockProps } from '@wordpress/block-editor';
import { generateBlockCustomCss } from '../_inc/helpers';

/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {WPElement} Element to render.
 */
export default function Edit( props ) {
	const { name, attributes, setAttributes } = props;

	const { 
		customCss
	} = attributes;

	// Block ID
	const blockId = `botiga-tb-block-${ props.clientId.substr( 0, 8 ) }`;
	setAttributes( { blockId: blockId } );

	// Block Props
	const blockProps = useBlockProps( {
		className: classnames( blockId, 'botiga-tb-block', 'botiga-tb-sp-price' )
	} );

	// Tab Navigation
	const [ tab, setTab ] = useState( 'style' );

	// Defaults
	const defaults = {
		align: 'left'
	}

	if( typeof customCss.align !== 'undefined' ) {
        defaults.align = customCss.align.value;
    }

	// Custom CSS
	generateBlockCustomCss( blockId, customCss );

	// Separator Style
	const separatorStyle = {
		fontSize: 14,
		margin: '15px 0 15px',
		borderBottom: '1px solid #e5e5e5',
		padding: '13px 0px'
	}

	return (
		<div { ...blockProps }>
		
			<InspectorControls>
				<BotigaTabsNavigation
					exclude="settings"
					activeTab={ tab }
					setActiveTab={ setTab } />

				<BotigaTabsContent 
					tabid="style"
					activeTab={ tab }>
					
					<PanelBody title={ __( 'General', 'botiga-pro' ) } initialOpen={ true }>
						<SelectControl
							label={ __( 'Alignment', 'botiga-pro' ) }
							value={ defaults.align }
							options={ [
								{ label: 'Left', value: 'flex-start' },
								{ label: 'Center', value: 'center' },
								{ label: 'Right', value: 'flex-end' },
							] }
							onChange={ ( value ) => { 								
								const css = {
									align: {
										selector: `{{WRAPPER}} .product-gallery-summary`,
										props: `justify-content: ${ value };`,
										value: value
									}
								};
								setAttributes( { customCss: { ...customCss, ...css } } ) 
							} }
						/>
						
						<h6 style={ separatorStyle }>{ __( 'Regular Price', 'botiga-pro' ) }</h6>

						<BotigaTypography 
							label={ __( 'Typography', 'botiga-pro' ) }
							idPrefix={ 'regular_price_typography' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .price` }
						/>
						<BotigaColorPicker 
							label={ __( 'Color', 'botiga-pro' ) }
							idPrefix={ 'regular_price_color' }
							attributes={ attributes }
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .price .woocommerce-Price-amount, {{WRAPPER}} .price` }
						/>

						<h6 style={ separatorStyle }>{ __( 'Sale Price', 'botiga-pro' ) }</h6>

						<BotigaTypography 
							label={ __( 'Typography', 'botiga-pro' ) }
							idPrefix={ 'sale_price_typography' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .price del` }
						/>
						<BotigaColorPicker 
							label={ __( 'Color', 'botiga-pro' ) }
							idPrefix={ 'sale_price_color' }
							attributes={ attributes }
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}} .price del .woocommerce-Price-amount, .price del` }
						/>
					</PanelBody>
				</BotigaTabsContent>
				<BotigaTabsContent 
					tabid="advanced"
					activeTab={ tab }>
					
					<PanelBody title={ __( 'Layout', 'botiga-pro' ) } initialOpen={ true }>
						<BotigaDimensions
							label={ __( 'Margin', 'botiga-pro' ) }
							idPrefix={ '' }
							type={ 'margin' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}}` } />

						<BotigaDimensions
							label={ __( 'Padding', 'botiga-pro' ) }
							idPrefix={ '' }
							type={ 'padding' }
							attributes={ attributes } 
							setAttributes={ setAttributes }
							customCssSelector={ `{{WRAPPER}}` } />
					</PanelBody>

				</BotigaTabsContent>
			</InspectorControls>

			<Disabled>
				<ServerSideRender
					block={ name }
					httpMethod="POST"
					attributes={ attributes }
				/>
			</Disabled>

		</div>
	);
}
