<?php
/**
 * Mega Menu
 * Custom Walker Class
 * 
 * @package Botiga_Pro
 */

class Botiga_Custom_Nav_Walker extends Walker_Nav_Menu {

	private $mm_cols_number = 0;
	private $mm_flag        = false;

    /**
	 * Starts the list before the elements are added.
	 *
	 * @param string   $output Used to append additional content (passed by reference).
	 * @param int      $depth  Depth of menu item. Used for padding.
	 * @param stdClass $args   An object of wp_nav_menu() arguments.
	 */
	public function start_lvl( &$output, $depth = 0, $args = null ) {
		if ( isset( $args->item_spacing ) && 'discard' === $args->item_spacing ) {
			$t = '';
			$n = '';
		} else {
			$t = "\t";
			$n = "\n";
		}
		$indent = str_repeat( $t, $depth );

		// Default class.
		$classes = array( 'sub-menu' );

		// Check and identify if the entire header is full-width or not
		$header_container_width = get_theme_mod( 'header_container', 'container-fluid' );
		if( 'container-fluid' === $header_container_width ) {
			$classes[] = 'is-header-full-width';
		}

		/**
		 * Filters the CSS class(es) applied to a menu list element.
		 *
		 * @param string[] $classes Array of the CSS classes that are applied to the menu `<ul>` element.
		 * @param stdClass $args    An object of `wp_nav_menu()` arguments.
		 * @param int      $depth   Depth of menu item. Used for padding.
		 */
		$class_names = implode( ' ', apply_filters( 'nav_menu_submenu_css_class', $classes, $args, $depth ) );
		$class_names = $class_names ? ' class="' . esc_attr( $class_names ) . '"' : '';

		$output .= "{$n}{$indent}<ul$class_names>{$n}";

		// Flag used to identify mega menu with colums number class name
		if( ! $this->mm_flag ) {
			$this->mm_cols_number = 0;
		}
	}

	/**
	 * Ends the list of after the elements are added.
	 *
	 * @param string   $output Used to append additional content (passed by reference).
	 * @param int      $depth  Depth of menu item. Used for padding.
	 * @param stdClass $args   An object of wp_nav_menu() arguments.
	 */
	public function end_lvl( &$output, $depth = 0, $args = null ) {
		if ( isset( $args->item_spacing ) && 'discard' === $args->item_spacing ) {
			$t = '';
			$n = '';
		} else {
			$t = "\t";
			$n = "\n";
		}
		$indent  = str_repeat( $t, $depth );
		$output .= "$indent</ul>{$n}";

		// Identify mega menu with colums number class name
		// Increase the cols number variable
		if( $depth === 1 && $this->mm_flag ) {
			$this->mm_cols_number++;
		}
		
		// Replace the '{{cols_number}}' with the real cols number
		if( ! $this->mm_flag ) {
			$output = str_replace( '{{cols_number}}', $this->mm_cols_number, $output );
		}

		// Back mm flag to false, so it will runs correctly through others mega menu's 
		if( $depth === 1 && $this->mm_flag ) {
			$this->mm_flag = false;
		}
	}

	/**
	 * Starts the element output.
	 *
	 * @param string   $output            Used to append additional content (passed by reference).
	 * @param WP_Post  $data_object       Menu item data object.
	 * @param int      $depth             Depth of menu item. Used for padding.
	 * @param stdClass $args              An object of wp_nav_menu() arguments.
	 * @param int      $current_object_id Optional. ID of the current menu item. Default 0.
	 */
	public function start_el( &$output, $data_object, $depth = 0, $args = null, $current_object_id = 0 ) {
		// Restores the more descriptive, specific name for use within this method.
		$menu_item = $data_object;
        
        if( $depth === 0 ) {
            $this->_is_mega_menu                        = get_post_meta( $menu_item->ID, '_is_mega_menu', true );
            $this->_is_mega_menu_width                  = get_post_meta( $menu_item->ID, '_is_mega_menu_width', true );
			$this->_is_mega_menu_style_background_type  = get_post_meta( $menu_item->ID, '_is_mega_menu_style_background_type', true );
			$this->_is_mega_menu_style_background_image = get_post_meta( $menu_item->ID, '_is_mega_menu_style_background_image', true );
        }

		$this->_item_has_badge 						   	       = get_post_meta( $menu_item->ID, '_item_has_badge', true );
		$this->_is_mega_menu_item_heading 			   	       = get_post_meta( $menu_item->ID, '_is_mega_menu_item_heading', true );
		$this->_is_mega_menu_item_content_type 		   	       = get_post_meta( $menu_item->ID, '_is_mega_menu_item_content_type', true );
		$this->_is_mega_menu_item_content_custom_html  	       = get_post_meta( $menu_item->ID, '_is_mega_menu_item_content_custom_html', true );
		$this->_is_mega_menu_item_content_custom_content_block = get_post_meta( $menu_item->ID, '_is_mega_menu_item_content_custom_content_block', true );
		$this->_is_mega_menu_column_width 			   	       = get_post_meta( $menu_item->ID, '_is_mega_menu_column_width', true );
		$this->_is_mega_menu_column_width_custom_value 	       = get_post_meta( $menu_item->ID, '_is_mega_menu_column_width_custom_value', true );
		$this->_is_mega_menu_column_padding_top 		       = get_post_meta( $menu_item->ID, '_is_mega_menu_column_padding_top', true );
		$this->_is_mega_menu_column_padding_right 		       = get_post_meta( $menu_item->ID, '_is_mega_menu_column_padding_right', true );
		$this->_is_mega_menu_column_padding_bottom 		       = get_post_meta( $menu_item->ID, '_is_mega_menu_column_padding_bottom', true );
		$this->_is_mega_menu_column_padding_left 		       = get_post_meta( $menu_item->ID, '_is_mega_menu_column_padding_left', true );
		$this->_is_mega_menu_item_heading_divider 	   	       = get_post_meta( $menu_item->ID, '_is_mega_menu_item_heading_divider', true );
		$this->_is_mega_menu_style_heading_divider_color       = get_post_meta( $menu_item->ID, '_is_mega_menu_style_heading_divider_color', true );
		$this->_is_mega_menu_style_heading_text_color 	       = get_post_meta( $menu_item->ID, '_is_mega_menu_style_heading_text_color', true );
		$this->_is_mega_menu_style_heading_text_color_hover    = get_post_meta( $menu_item->ID, '_is_mega_menu_style_heading_text_color_hover', true );
		$this->_is_mega_menu_item_disable_link 				   = get_post_meta( $menu_item->ID, '_is_mega_menu_item_disable_link', true );
 
		if ( isset( $args->item_spacing ) && 'discard' === $args->item_spacing ) {
			$t = '';
			$n = '';
		} else {
			$t = "\t";
			$n = "\n";
		}
		$indent = ( $depth ) ? str_repeat( $t, $depth ) : '';

		$classes   = empty( $menu_item->classes ) ? array() : (array) $menu_item->classes;
		$classes[] = 'menu-item-' . $menu_item->ID;

		// Mega Menu
        if( $depth === 0 && $this->_is_mega_menu ) {
            $classes[] = 'botiga-mega-menu';
            $classes[] = 'botiga-mega-menu-' . esc_attr( $this->_is_mega_menu_width );
			$classes[] = 'botiga-mega-menu-{{cols_number}}-columns';
        }

		if( $depth === 1 && $this->_is_mega_menu ) {
			$classes[] = 'botiga-mega-menu-column';

			if( $this->_is_mega_menu_item_content_type !== 'default' ) {
				$classes[] = 'botiga-mega-menu-ws-normal';
			}
		}

		/**
		 * Filters the arguments for a single nav menu item.
		 *
		 * @param stdClass $args      An object of wp_nav_menu() arguments.
		 * @param WP_Post  $menu_item Menu item data object.
		 * @param int      $depth     Depth of menu item. Used for padding.
		 */
		$args = apply_filters( 'nav_menu_item_args', $args, $menu_item, $depth );

		/**
		 * Filters the CSS classes applied to a menu item's list item element.
		 *
		 * @param string[] $classes   Array of the CSS classes that are applied to the menu item's `<li>` element.
		 * @param WP_Post  $menu_item The current menu item object.
		 * @param stdClass $args      An object of wp_nav_menu() arguments.
		 * @param int      $depth     Depth of menu item. Used for padding.
		 */
		$class_names = implode( ' ', apply_filters( 'nav_menu_css_class', array_filter( $classes ), $menu_item, $args, $depth ) );
		$class_names = $class_names ? ' class="' . esc_attr( $class_names ) . '"' : '';

		/**
		 * Filters the ID applied to a menu item's list item element.
		 *
		 * @param string   $menu_id   The ID that is applied to the menu item's `<li>` element.
		 * @param WP_Post  $menu_item The current menu item.
		 * @param stdClass $args      An object of wp_nav_menu() arguments.
		 * @param int      $depth     Depth of menu item. Used for padding.
		 */
		$id = apply_filters( 'nav_menu_item_id', 'menu-item-' . $menu_item->ID, $menu_item, $args, $depth );
		$id = $id ? ' id="' . esc_attr( $id ) . '"' : '';

		$styles = array();
		$style_att = ''; 
		if( $depth === 1 && $this->_is_mega_menu ) {

			// Column Width
			if( $this->_is_mega_menu_column_width === 'custom' ) {
				$styles[] = 'width: '. absint( $this->_is_mega_menu_column_width_custom_value ) .'px;';
			}

			// Column Padding
			if( $this->_is_mega_menu_column_padding_top ) {
				$styles[] = 'padding-top: '. absint( $this->_is_mega_menu_column_padding_top ) .'px;';
			}
			if( $this->_is_mega_menu_column_padding_right ) {
				$styles[] = 'padding-right: '. absint( $this->_is_mega_menu_column_padding_right ) .'px;';
			}
			if( $this->_is_mega_menu_column_padding_bottom ) {
				$styles[] = 'padding-bottom: '. absint( $this->_is_mega_menu_column_padding_bottom ) .'px;';
			}
			if( $this->_is_mega_menu_column_padding_left ) {
				$styles[] = 'padding-left: '. absint( $this->_is_mega_menu_column_padding_left ) .'px;';
			}

			// Mount style att
			if( ! empty( $styles[0] ) ) {
				$style_att = 'style="'. implode( ' ', $styles ) .'"';
			}
		}

		$output .= $indent . '<li' . $id . $class_names . $style_att . '>';
		
		$atts           = array();
		$atts['title']  = ! empty( $menu_item->attr_title ) ? $menu_item->attr_title : '';
		$atts['target'] = ! empty( $menu_item->target ) ? $menu_item->target : '';
		if ( '_blank' === $menu_item->target && empty( $menu_item->xfn ) ) {
			$atts['rel'] = 'noopener';
		} else {
			$atts['rel'] = $menu_item->xfn;
		}
		$atts['href']         = ! empty( $menu_item->url ) ? $menu_item->url : '';
		$atts['aria-current'] = $menu_item->current ? 'page' : '';

		/**
		 * Filters the HTML attributes applied to a menu item's anchor element.
		 *
		 * @param array $atts {
		 *     The HTML attributes applied to the menu item's `<a>` element, empty strings are ignored.
		 *
		 *     @type string $title        Title attribute.
		 *     @type string $target       Target attribute.
		 *     @type string $rel          The rel attribute.
		 *     @type string $href         The href attribute.
		 *     @type string $aria-current The aria-current attribute.
		 * }
		 * @param WP_Post  $menu_item The current menu item object.
		 * @param stdClass $args      An object of wp_nav_menu() arguments.
		 * @param int      $depth     Depth of menu item. Used for padding.
		 */
		$atts = apply_filters( 'nav_menu_link_attributes', $atts, $menu_item, $args, $depth );

		$classes = array();

		// Identify botiga mega menu links
		$classes[] = 'botiga-mega-menu-link';

		// Is mega menu heading
		if( $this->_is_mega_menu_item_heading ) {
			$classes[] = 'is-mega-menu-heading';
		}

		// Mega menu heading has divider
		if( $this->_is_mega_menu_item_heading_divider  ) {
			$classes[] = 'has-mega-menu-heading-divider';
		}

		// Check if $atts has already some class to ensure
		// compatibility with the filter 'nav_menu_link_attributes'
		if( isset( $atts[ 'class' ] ) ) {
			$atts[ 'class' ] = $atts[ 'class' ] . ' ' . implode( ' ', $classes );
		} else {
			$atts[ 'class' ] = implode( ' ', $classes );
		}

		// Disable link
		if( $this->_is_mega_menu_item_disable_link ) {
			unset( $atts[ 'href' ] );
		}

		$attributes = '';
		foreach ( $atts as $attr => $value ) {
			if ( is_scalar( $value ) && '' !== $value && false !== $value ) {
				$value       = ( 'href' === $attr ) ? esc_url( $value ) : esc_attr( $value );
				$attributes .= ' ' . $attr . '="' . $value . '"';
			}
		}

		/** This filter is documented in wp-includes/post-template.php */
		$title = apply_filters( 'the_title', $menu_item->title, $menu_item->ID );

		/**
		 * Filters a menu item's title.
		 *
		 * @param string   $title     The menu item's title.
		 * @param WP_Post  $menu_item The current menu item object.
		 * @param stdClass $args      An object of wp_nav_menu() arguments.
		 * @param int      $depth     Depth of menu item. Used for padding.
		 */
		$title = apply_filters( 'nav_menu_item_title', $title, $menu_item, $args, $depth );

		$item_output  = isset( $args->before ) ? $args->before : '';
		$item_output .= '<a' . $attributes . '>';
		$item_output .= isset( $args->link_before ) ? $args->link_before . $title : $title; 
		$item_output .= isset( $args->link_after ) ? $args->link_after : '';

		// Display Badge/Label
		if( $this->_item_has_badge ) {
			$this->_item_badge_title 			= get_post_meta( $menu_item->ID, '_item_badge_title', true );
			$this->_item_badge_text_color 		= get_post_meta( $menu_item->ID, '_item_badge_text_color', true );
			$this->_item_badge_background_color = get_post_meta( $menu_item->ID, '_item_badge_background_color', true );

			$badge_atts   = array();
			$badge_styles = array();

			if( $this->_item_badge_text_color ) {
				$badge_styles[] = 'color: ' . $this->_item_badge_text_color;
			}

			if( $this->_item_badge_background_color ) {
				$badge_styles[] = 'background-color: ' . $this->_item_badge_background_color;
			}
			
			// Mount badge class attribute
			$badge_atts[] = 'class="botiga-badge"';

			// Mount badge style attribute
			if( ! empty(  $badge_styles[0] ) ) {
				$badge_atts[] = 'style="'. esc_attr( implode( '; ', $badge_styles ) ) .'"';
			}

			$item_output .= '<span '. implode( ' ', $badge_atts ) .'>'. esc_html( $this->_item_badge_title ) .'</span>';
		}

		$item_output .= '</a>';
		$item_output .= isset( $args->after ) ? $args->after : '';

		// Depth 1 items are used only to be columns, so the content
		// inside it should be removed
		if( $depth === 1 && $this->_is_mega_menu ) {
			$item_output = '';
		}

		/**
		 * Filters a menu item's starting output.
		 *
		 * The menu item's starting output only includes `$args->before`, the opening `<a>`,
		 * the menu item's title, the closing `</a>`, and `$args->after`. Currently, there is
		 * no filter for modifying the opening and closing `<li>` for a menu item.
		 *
		 * @param string   $item_output The menu item's starting HTML output.
		 * @param WP_Post  $menu_item   Menu item data object.
		 * @param int      $depth       Depth of menu item. Used for padding.
		 * @param stdClass $args        An object of wp_nav_menu() arguments.
		 */
		if( $this->_is_mega_menu && $this->_is_mega_menu_item_content_type === 'html' ) {
			$output .= do_shortcode( $this->_is_mega_menu_item_content_custom_html );
		} else if( $this->_is_mega_menu && $this->_is_mega_menu_item_content_type === 'content-block' ) {
			$template_id = $this->_is_mega_menu_item_content_custom_content_block;

			if( class_exists( 'Elementor\Plugin' ) && Elementor\Plugin::$instance->documents->get( $template_id )->is_built_with_elementor() ) {
				$block_content = Elementor\Plugin::instance()->frontend->get_builder_content( $template_id );
			} else {
				$block_content = get_post_field( 'post_content', $template_id );
			}

			$output .= do_blocks( $block_content );
		} else {
			$output .= apply_filters( 'walker_nav_menu_start_el', $item_output, $menu_item, $depth, $args );
		}

		// Flag used to identify mega menu with colums number class name
		if( $depth === 1 && $this->_is_mega_menu ) {
			$this->mm_flag = true;
		}
	}

	/**
	 * Ends the element output, if needed.
	 *
	 * @param string   $output      Used to append additional content (passed by reference).
	 * @param WP_Post  $data_object Menu item data object. Not used.
	 * @param int      $depth       Depth of page. Not Used.
	 * @param stdClass $args        An object of wp_nav_menu() arguments.
	 */
	public function end_el( &$output, $data_object, $depth = 0, $args = null ) {
		if ( isset( $args->item_spacing ) && 'discard' === $args->item_spacing ) {
			$t = '';
			$n = '';
		} else {
			$t = "\t";
			$n = "\n";
		}
		$output .= "</li>{$n}";
	}

}