<?php
/**
* Options Page
*
* @package   Elementor Custom icons
* @author    Michael Bourne
* @license   GPL3
* @link      https://ursa6.com
* @since     0.1.0
*/

if( ! defined( 'ABSPATH' ) ) {
    return;
}

$options = get_option( 'ec_icons_fonts' );

?>
<div class="eci-reset eci-wrap eci-wrap-about">
	<div class="eci-content">
		<div class="eci-main">
			<div class="eci-row">
				<div class="eci-column">
					<div class="eci-box">

						<header class="eci-box-header">
							<h2 class="eci-box-title"><?php esc_html_e( 'Elementor Custom Icons', 'custom-icons-for-elementor' ); ?></h2>
						</header>

						<div class="eci-box-content">
							<ul class="eci-box-features">
								<li>
									<div class="eci-box-feature-icon">
										<img src="data:image/svg+xml;utf8;base64,PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0iaXNvLTg4NTktMSI/Pgo8IS0tIEdlbmVyYXRvcjogQWRvYmUgSWxsdXN0cmF0b3IgMTguMS4xLCBTVkcgRXhwb3J0IFBsdWctSW4gLiBTVkcgVmVyc2lvbjogNi4wMCBCdWlsZCAwKSAgLS0+CjxzdmcgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIiB4bWxuczp4bGluaz0iaHR0cDovL3d3dy53My5vcmcvMTk5OS94bGluayIgdmVyc2lvbj0iMS4xIiBpZD0iQ2FwYV8xIiB4PSIwcHgiIHk9IjBweCIgdmlld0JveD0iMCAwIDIwLjU0NiAyMC41NDYiIHN0eWxlPSJlbmFibGUtYmFja2dyb3VuZDpuZXcgMCAwIDIwLjU0NiAyMC41NDY7IiB4bWw6c3BhY2U9InByZXNlcnZlIiB3aWR0aD0iMzJweCIgaGVpZ2h0PSIzMnB4Ij4KPGc+Cgk8Zz4KCQk8cGF0aCBkPSJNMTguNzc4LDBIMS43NjdDMC43OTMsMCwwLDAuNzkzLDAsMS43Njh2MTcuMDFjMCwwLjk3NSwwLjc5MiwxLjc2OCwxLjc2NywxLjc2OGgxNy4wMTEgICAgYzAuOTczLDAsMS43NjctMC43OTMsMS43NjctMS43NjhWMS43NjdDMjAuNTQ0LDAuNzkzLDE5Ljc1MSwwLDE4Ljc3OCwweiBNMTkuMDMsMTguNzc3YzAsMC4xMzktMC4xMTMsMC4yNTItMC4yNTIsMC4yNTJIMS43NjcgICAgYy0wLjEzOSwwLTAuMjUyLTAuMTEzLTAuMjUyLTAuMjUyVjEuNzY3YzAtMC4xNCwwLjExMy0wLjI1MywwLjI1Mi0wLjI1M2gxNy4wMTFjMC4xMzksMCwwLjI1MiwwLjExMywwLjI1MiwwLjI1MyAgICBDMTkuMDMxLDEuNzY3LDE5LjAzMSwxOC43NzcsMTkuMDMsMTguNzc3eiIgZmlsbD0iIzAwMDAwMCIvPgoJCTxwb2x5Z29uIHBvaW50cz0iNi42Miw1Ljg5IDcuMTA5LDguMTIxIDkuNTM1LDYuOTY1IDkuNTc1LDYuOTY1IDkuNTc1LDE3LjA0MyAxMi40NTMsMTcuMDQzIDEyLjQ1Myw0LjMyNCAgICAgOS45ODQsNC4zMjQgICAiIGZpbGw9IiMwMDAwMDAiLz4KCTwvZz4KPC9nPgo8Zz4KPC9nPgo8Zz4KPC9nPgo8Zz4KPC9nPgo8Zz4KPC9nPgo8Zz4KPC9nPgo8Zz4KPC9nPgo8Zz4KPC9nPgo8Zz4KPC9nPgo8Zz4KPC9nPgo8Zz4KPC9nPgo8Zz4KPC9nPgo8Zz4KPC9nPgo8Zz4KPC9nPgo8Zz4KPC9nPgo8Zz4KPC9nPgo8L3N2Zz4K" />
									</div>
									<div class="eci-box-feature-info">
										<h4 class="eci-box-content-title"><?php esc_html_e( 'Create a new Icon Font', 'custom-icons-for-elementor' ); ?></h4>
										<span class="eci-box-content-text">
										<?php
											/* translators: Options page step 1. KSES set to a, br, and i.  */
											echo sprintf( 
												wp_kses( 
													__( 'Visit <a href="%s" target="_blank">Fontello</a> and select the icons you would like to use. You can even upload custom SVG icons of your own on Fontello, whether they be custom made or from a source like <a href="%s" target="_blank">Flaticon!</a>', 'custom-icons-for-elementor' ), 
													array(  
														'a' => array( 
															'href' => array(),
															'target' => array(),
															'title' => array()
															),
														'br' => array(),
														'i' => array()
													) 
												), 
												esc_url( 'http://fontello.com/' ),
												esc_url( 'https://www.flaticon.com/' ) 
											);
										?>
										</span>
									</div>
								</li>
								<li>
									<div class="eci-box-feature-icon">
										<img src="data:image/svg+xml;utf8;base64,PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0iaXNvLTg4NTktMSI/Pgo8IS0tIEdlbmVyYXRvcjogQWRvYmUgSWxsdXN0cmF0b3IgMTguMS4xLCBTVkcgRXhwb3J0IFBsdWctSW4gLiBTVkcgVmVyc2lvbjogNi4wMCBCdWlsZCAwKSAgLS0+CjxzdmcgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIiB4bWxuczp4bGluaz0iaHR0cDovL3d3dy53My5vcmcvMTk5OS94bGluayIgdmVyc2lvbj0iMS4xIiBpZD0iQ2FwYV8xIiB4PSIwcHgiIHk9IjBweCIgdmlld0JveD0iMCAwIDIwLjM2OCAyMC4zNjgiIHN0eWxlPSJlbmFibGUtYmFja2dyb3VuZDpuZXcgMCAwIDIwLjM2OCAyMC4zNjg7IiB4bWw6c3BhY2U9InByZXNlcnZlIiB3aWR0aD0iMzJweCIgaGVpZ2h0PSIzMnB4Ij4KPGc+Cgk8Zz4KCQk8cGF0aCBkPSJNMTguNjE4LDBIMS43NTJDMC43ODcsMCwwLjAwMSwwLjc4NiwwLjAwMSwxLjc1MXYxNi44NjVjMCwwLjk2NiwwLjc4NiwxLjc1MiwxLjc1MSwxLjc1MmgxNi44NjUgICAgYzAuOTY1LDAsMS43NS0wLjc4NiwxLjc1LTEuNzUyVjEuNzUxQzIwLjM2OCwwLjc4NiwxOS41ODIsMCwxOC42MTgsMHogTTE4Ljg2OCwxOC42MTdjMCwwLjEzOS0wLjExMywwLjI1MS0wLjI1LDAuMjUxSDEuNzUyICAgIGMtMC4xMzgsMC0wLjI1LTAuMTEyLTAuMjUtMC4yNTFWMS43NTFjMC0wLjEzNywwLjExMi0wLjI1LDAuMjUtMC4yNWgxNi44NjVjMC4xMzcsMCwwLjI1LDAuMTEyLDAuMjUsMC4yNXYxNi44NjZIMTguODY4eiIgZmlsbD0iIzAwMDAwMCIvPgoJCTxwYXRoIGQ9Ik0xMC4yNzYsMTMuODlsMS4wOTItMC45MWMxLjcxMS0xLjUzLDMuMTUtMy4xMTQsMy4xNS01LjFjMC0yLjE0OS0xLjQ3Ny0zLjcxNS00LjE1Mi0zLjcxNSAgICBjLTEuNjA0LDAtMi45ODcsMC41NDYtMy44OCwxLjIxOUw3LjI2OCw3LjM3QzcuODg5LDYuODk3LDguNzgyLDYuMzg3LDkuOCw2LjM4N2MxLjM2NiwwLDEuOTUsMC43NjUsMS45NSwxLjcyOSAgICBjLTAuMDM3LDEuMzg2LTEuMjk0LDIuNzE0LTMuODgsNS4wMjlsLTEuNTMsMS4zODR2MS42NzdoOC4zOTZ2LTIuMjc4aC00LjQ2MVYxMy44OUgxMC4yNzZ6IiBmaWxsPSIjMDAwMDAwIi8+Cgk8L2c+CjwvZz4KPGc+CjwvZz4KPGc+CjwvZz4KPGc+CjwvZz4KPGc+CjwvZz4KPGc+CjwvZz4KPGc+CjwvZz4KPGc+CjwvZz4KPGc+CjwvZz4KPGc+CjwvZz4KPGc+CjwvZz4KPGc+CjwvZz4KPGc+CjwvZz4KPGc+CjwvZz4KPGc+CjwvZz4KPGc+CjwvZz4KPC9zdmc+Cg==" />
									</div>
									<div class="eci-box-feature-info">
										<h4 class="eci-box-content-title"><?php esc_html_e( 'Upload your ZIP file here', 'custom-icons-for-elementor' ); ?></h4>
										<span class="eci-box-content-text">
											<?php 
												/* translators: Options page step 2.  */
												esc_html_e( 'After selecting your icons on Fontello, give your font a unique name and hit download. Then, upload the Fontello ZIP file here.', 'custom-icons-for-elementor' ); 
											?>
										</span>
									</div>
								</li>
								<li>
									<div class="eci-box-feature-icon">
										<img src="data:image/svg+xml;utf8;base64,PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0iaXNvLTg4NTktMSI/Pgo8IS0tIEdlbmVyYXRvcjogQWRvYmUgSWxsdXN0cmF0b3IgMTguMS4xLCBTVkcgRXhwb3J0IFBsdWctSW4gLiBTVkcgVmVyc2lvbjogNi4wMCBCdWlsZCAwKSAgLS0+CjxzdmcgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIiB4bWxuczp4bGluaz0iaHR0cDovL3d3dy53My5vcmcvMTk5OS94bGluayIgdmVyc2lvbj0iMS4xIiBpZD0iQ2FwYV8xIiB4PSIwcHgiIHk9IjBweCIgdmlld0JveD0iMCAwIDE0LjM0MSAxNC4zNDEiIHN0eWxlPSJlbmFibGUtYmFja2dyb3VuZDpuZXcgMCAwIDE0LjM0MSAxNC4zNDE7IiB4bWw6c3BhY2U9InByZXNlcnZlIiB3aWR0aD0iMzJweCIgaGVpZ2h0PSIzMnB4Ij4KPGc+Cgk8Zz4KCQk8cGF0aCBkPSJNMTMuMTA4LDBIMS4yMzNDMC41NTMsMCwwLDAuNTUzLDAsMS4yMzR2MTEuODc0YzAsMC42ODEsMC41NTMsMS4yMzMsMS4yMzMsMS4yMzNoMTEuODc1ICAgIGMwLjY4MSwwLDEuMjMzLTAuNTUzLDEuMjMzLTEuMjMzVjEuMjM1QzE0LjM0MSwwLjU1MywxMy43ODgsMCwxMy4xMDgsMHogTTEzLjI4NCwxMy4xMDljMCwwLjA5OC0wLjA3OSwwLjE3Ny0wLjE3NywwLjE3N0gxLjIzMyAgICBjLTAuMDk3LDAtMC4xNzYtMC4wNzktMC4xNzYtMC4xNzdWMS4yMzVjMC0wLjA5NywwLjA3OS0wLjE3NiwwLjE3Ni0wLjE3NmgxMS44NzVjMC4wOTgsMCwwLjE3NywwLjA4LDAuMTc3LDAuMTc2TDEzLjI4NCwxMy4xMDkgICAgTDEzLjI4NCwxMy4xMDl6IiBmaWxsPSIjMDAwMDAwIi8+CgkJPHBhdGggZD0iTTguMjczLDYuODRWNi44MTZDOS4yNTksNi40NzEsOS43NCw1LjgwNiw5Ljc0LDQuOTU1YzAtMS4wOTgtMC45NDktMS45OTctMi42NjItMS45OTcgICAgYy0xLjAzNiwwLTEuOTk3LDAuMjk2LTIuNDc4LDAuNjA0bDAuMzgyLDEuMzU2YzAuMzMzLTAuMTk3LDEuMDIzLTAuNDgxLDEuNjc2LTAuNDgxYzAuNzg5LDAsMS4xODMsMC4zNTcsMS4xODMsMC44MzggICAgYzAsMC42NzgtMC44MDEsMC45MTItMS40MjksMC45MjRINS42ODR2MS4zNDNoMC43NjRjMC44MjYsMCwxLjYxNCwwLjM1OCwxLjYxNCwxLjE0NmMwLDAuNjA0LTAuNDkzLDEuMDczLTEuNDY3LDEuMDczICAgIGMtMC43NjQsMC0xLjUyOC0wLjMyMS0xLjg2MS0wLjQ5M2wtMC4zODIsMS40MDRjMC40NjgsMC4yOTYsMS4zNDQsMC41NjcsMi4zNzksMC41NjdjMi4wMzMsMCwzLjI3Ny0xLjAzNSwzLjI3Ny0yLjQ1MiAgICBDMTAuMDEsNy43NCw5LjIzNCw3LjAxNCw4LjI3Myw2Ljg0eiIgZmlsbD0iIzAwMDAwMCIvPgoJPC9nPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+CjxnPgo8L2c+Cjwvc3ZnPgo=" />
									</div>
									<div class="eci-box-feature-info">
										<h4 class="eci-box-content-title"><?php esc_html_e( 'Use Your Icons', 'custom-icons-for-elementor' ); ?></h4>
										<span class="eci-box-content-text">
										<?php
											/* translators: Options page step 3. KSES set to a, br, and i.  */
											echo sprintf( 
												wp_kses( 
													__( 'You will now see your custom icons in all Elementor elements that have icon selectors in them.<br><br>Looking for more help? You can watch an instructional video by WPBuilders <a href="%s" target="_blank">here.</a>', 'custom-icons-for-elementor' ), 
													array(  
														'a' => array( 
															'href' => array(),
															'target' => array(),
															'title' => array()
															),
														'br' => array(),
														'i' => array()
													) 
												), 
												esc_url( 'https://youtu.be/Rnu9XVD8AdI' )
											);
										?>
										</span>
									</div>
								</li>
							</ul>
						</div>

					</div>
				</div>          
			</div>			
			<div class="eci-row">
				<div class="eci-column">
					<div class="eci-box eci-box-min-height">

						<header class="eci-box-header">
							<h2 class="eci-box-title"><?php esc_html_e( 'Upload Fontello Zip', 'custom-icons-for-elementor' ); ?></h2>
						</header>

						<div class="eci-box-content uploader">
							<div class="ec-icons-wrapper">
								<div class="ec-icons-drop">
									<div class="ec-icons-cont">
										<img src="data:image/svg+xml;utf8;base64,PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0iaXNvLTg4NTktMSI/Pgo8IS0tIEdlbmVyYXRvcjogQWRvYmUgSWxsdXN0cmF0b3IgMTkuMC4wLCBTVkcgRXhwb3J0IFBsdWctSW4gLiBTVkcgVmVyc2lvbjogNi4wMCBCdWlsZCAwKSAgLS0+CjxzdmcgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIiB4bWxuczp4bGluaz0iaHR0cDovL3d3dy53My5vcmcvMTk5OS94bGluayIgdmVyc2lvbj0iMS4xIiBpZD0iQ2FwYV8xIiB4PSIwcHgiIHk9IjBweCIgdmlld0JveD0iMCAwIDMwNy4yIDMwNy4yIiBzdHlsZT0iZW5hYmxlLWJhY2tncm91bmQ6bmV3IDAgMCAzMDcuMiAzMDcuMjsiIHhtbDpzcGFjZT0icHJlc2VydmUiIHdpZHRoPSI2NHB4IiBoZWlnaHQ9IjY0cHgiPgo8Zz4KCTxnPgoJCTxwYXRoIGQ9Ik0xNjUuNjYzLDUuMDAxYy02LjY2Ni02LjY2Ni0xNy40OC02LjY2Ni0yNC4xMzEsMEw2OS4xMTUsNzcuNDAzbDI0LjE1MSwyNC4xMzFsNDMuMjY5LTQzLjI2NHYxODAuNjY0aDM0LjEzVjU4LjI2OSAgICBsNDMuMjg0LDQzLjI2NGwyNC4xMzYtMjQuMTMxTDE2NS42NjMsNS4wMDF6IiBmaWxsPSIjOGU5OWE1Ii8+Cgk8L2c+CjwvZz4KPGc+Cgk8Zz4KCQk8cGF0aCBkPSJNMjczLjA2NSwyMDQuNzk5djY4LjI3SDM0LjEzNXYtNjguMjdIMHY2OC4yN2MwLDE4Ljc2NSwxNS4zNjUsMzQuMTMsMzQuMTM1LDM0LjEzaDIzOC45MyAgICBjMTguNzcsMCwzNC4xMzUtMTUuMzY1LDM0LjEzNS0zNC4xM3YtNjguMjdIMjczLjA2NXoiIGZpbGw9IiM4ZTk5YTUiLz4KCTwvZz4KPC9nPgo8Zz4KPC9nPgo8Zz4KPC9nPgo8Zz4KPC9nPgo8Zz4KPC9nPgo8Zz4KPC9nPgo8Zz4KPC9nPgo8Zz4KPC9nPgo8Zz4KPC9nPgo8Zz4KPC9nPgo8Zz4KPC9nPgo8Zz4KPC9nPgo8Zz4KPC9nPgo8Zz4KPC9nPgo8Zz4KPC9nPgo8Zz4KPC9nPgo8L3N2Zz4K" />
										<div class="ec-icons-tit">
											<?php esc_html_e( 'Drag & Drop', 'custom-icons-for-elementor' ); ?>
										</div>
										<div class="ec-icons-desc">
											<?php esc_html_e( 'your Fontello .zip download here, or', 'custom-icons-for-elementor' ); ?>
										</div>
										<div class="ec-icons-browse">
											<?php esc_html_e( 'click here to browse', 'custom-icons-for-elementor' ); ?>
										</div>
									</div>
									<output id="ec-icons-list">
										<div class="ec-icons-progress">
											<div class="bar"></div>
										</div>
									</output>
									<input id="ec-icons-files" name="files[]" type="file" accept=".zip" />
									<?php wp_nonce_field( 'ec_icons_nonce' ); ?>
								</div>
							</div>
						</div>

					</div>
				</div>
			</div>

			<?php
			$class_wrap = '';
			if ( empty( $options ) ) {
				$class_wrap = 'hidden';
			} ?>

			<div class="eci-row">
				<div class="eci-column">
				<div class="eci-box-content eci-pan eci-ta-center wrapper-list-fonts <?php echo esc_attr($class_wrap); ?>">
					<p class="eci-extensions-info"><b><?php esc_html_e( 'Uploaded Fonts', 'custom-icons-for-elementor' ); ?></b></p>
					<div class="eci-extensions">

						<style>
						.eci-extension-content .eci:before {
							font-style: normal;
							font-weight: normal;
							speak: none;
							display: inline-block;
							text-decoration: inherit;
							width: 1em;
							margin-right: .2em;
							text-align: center;
							font-variant: normal;
							text-transform: none;
							line-height: 1em;
							margin-left: .2em;
							-webkit-font-smoothing: antialiased;
							-moz-osx-font-smoothing: grayscale;
						}
					</style>
					<?php if ( ! empty( $options ) && is_array( $options ) ) : ?>
						<?php foreach ( $options as $key => $font ) :

						if ( empty( $font['data'] ) ) {
							continue;
						}

						$font_data  = json_decode( $font['data'], true );
						$icons      = ec_icons_manager()->parse_css( $font_data['css_root'], $key, $font_data['css_url'] );
						$first_icon = ! empty( $icons ) ? key( $icons ) : '';

						?>
						<div class="eci-extension eci-extension-installed font-item " id="<?php echo esc_attr( $key ); ?>" >
							<div class="eci-extension-content">
								<i class="eci <?php echo esc_attr( $first_icon ); ?>" style="font-size: 80px;"></i>
								<h4 class="eci-extension-title"><?php echo esc_html( $key ); ?></h4>
								<div class="eci-extension-info">
									<span class="eci-extension-info-details">
										<span class="value"><?php echo esc_html( count( $font_data['icons'] ) ) ?></span> icons</span>

										<div class="iconlist hide">
											<?php
											foreach($icons as $iconkey => $iconcode){
												echo '<div><i class="eci ' . $iconkey . '" style="font-size: 16px; margin-right: 10px;"></i><span>' . $iconkey . '</span></div>';
											}
											?>
										</div>
								</div>
								<a class="eci-btn delete-font" data-font="<?php echo esc_attr( $font['data'] ); ?>" href="#">Delete</a>
								</div>

								<footer class="eci-extension-footer">

									<?php $status = ($font['status'] == '1') ? 'yes' : 'no'; ?>
									<a class="eci-extension-status-icon " href="#">
										<svg>
											<use xmlns:xlink="http://www.w3.org/1999/xlink" xlink:href="<?php echo esc_url(ECIcons_URI . 'assets'); ?>/icons.svg#<?php echo esc_attr($status); ?>"></use>
										</svg>
									</a>

								</footer>
							</div>
						<?php endforeach; ?>
					<?php endif; ?>
				</div>
			</div>
		</div>
		</div>
		<div class="eci-row">
			<div class="eci-column">
				<div class="eci-box eci-box-min-height">

						<header class="eci-box-header">
							<h2 class="eci-box-title"><?php esc_html_e( 'Regenerate CSS', 'custom-icons-for-elementor' ); ?></h2>
						</header>

						<div class="eci-box-content">
							<p><?php esc_html_e( 'Sometimes you may need to regenerate the CSS file for your custom icons, such as when you change your WordPress domain, add SSL, or update this plugin and notice font issues.', 'custom-icons-for-elementor' ); ?></p>
										<div class="ec-icons-regen">
											<?php esc_html_e( 'Regenerate', 'custom-icons-for-elementor' ); ?>
										</div>

						</div>

				</div>
			</div>
		</div>

	</div>


	<div class="eci-sidebar">
		<div class="eci-cta">
			<a href="https://ursa6.com/" target="_blank">
				<svg id="b33f915c-dbb2-49a6-938e-3b1cd7a9f797" data-name="Layer 1" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 338.06 478.86"><defs><style>.a838b416-9a9e-4fdb-8b0b-1ef8f173e35b{fill:#fbde12;}</style></defs><path class="a838b416-9a9e-4fdb-8b0b-1ef8f173e35b" d="M830.68,406C920.5,350.29,903,240.9,880.93,198c-12.68-24.63-26.56-55-20.77-84.93,2-10,4.39-20.44,2.19-30-2.62-11.37-11.12-18.81-16.72-28.31-4.6-7.78-10.56-20.58-5.43-30.29,0,0,3.42-6.48,2.55-8.32-2.45-5.19-16-9.34-21.56-7-4.65,1.93-9.3,13-13.75,15.47-8.74,4.84-28.73-3-38.11-.11-6.52,2-15.17,15-21.06,18.71-11.63,7.41-43.89,2.71-51.81,14.22-5.43,7.88,1,29.45,7.38,35.36,6.62,6.15,27.85,6.15,34.07,12.73,37.65,39.82-14,98-53,104-17.19,2.64-47.49-23.19-63.4-14.92-9.19,4.78-21.92,33.34-16.42,35.16,44,14.56,81,53.43,90.92,45.4,53.57-43.26,110.81,5.28,28.83,92.6-11.51,12.26,15.44,58.91,3.78,71-4.33,4.5-18.1-3.14-23.43-.06-7.59,4.38-16,25.67-12.43,33.1,10.64,22.19,80.93,19.07,99.71.29C806,458.58,793.7,429,830.68,406Z" transform="translate(-603.8 -8.43)"/><path class="a838b416-9a9e-4fdb-8b0b-1ef8f173e35b" d="M666.72,174.23c20.92,7.81,20.82,2.46,33.25,14.18,15.84-8.24,36.09-33.27,35-55.85-75.29-7.47-81.8-65.5-100.89-63.55-14.19,1.44-20.2,50.53-15.48,51.68C653.15,129.09,649.07,167.63,666.72,174.23Z" transform="translate(-603.8 -8.43)"/><path class="a838b416-9a9e-4fdb-8b0b-1ef8f173e35b" d="M901.05,337.27c-11.29,40.1-31.91,59.28-55.68,74.89,0,0,27,6.63,29.59,12.92,1.44,3.52-4.83,12.28-8.42,14.05-8.18,4-27-4.68-34.9-.1-7.59,4.38-16,25.69-12.43,33.1,10.13,20.93,75.8,17.28,94.21.28,3.84-3.55,2.07-8.06,5.87-19.89,7.64-23.82,20.29-26,22.34-32l0-.06C945,398.54,901.05,337.27,901.05,337.27Z" transform="translate(-603.8 -8.43)"/></svg>
			</a>
			<hr class="eci-cta-spacing">
			<a href="https://ursa6.com/" target="_blank">
				<svg id="a681b75e-a836-4b27-a8fe-eaaf45b5525e" data-name="Layer 1" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 748.9 154.54"><defs><style>.\34 d796d4f-b091-45bc-941b-d60ede9debbe{fill:#333;}.fd894280-7ab4-4a26-84e7-c559953ee0b7{fill:#fbde12;}</style></defs><path class="4d796d4f-b091-45bc-941b-d60ede9debbe" d="M517,264.3c0,40.31-26.73,61.95-59.4,61.95-32.45,0-59.18-21.64-59.18-61.95v-88H425V266c0,24.18,14.64,35.85,32.67,35.85,18.25,0,32.89-11.67,32.89-35.85V176.26H517Z" transform="translate(-398.46 -172.77)"/><path class="4d796d4f-b091-45bc-941b-d60ede9debbe" d="M631.81,324.76l-42.42-60.67H575v60.67H548.44V176.26H597c28.64,0,50.49,15.06,50.49,44.34,0,22.06-12.3,35.43-30.33,40.73l47.3,63.43ZM596,200.44H575V239.9h21c16.76,0,25.25-7,25.25-19.51C621.21,208.08,612.72,200.44,596,200.44Z" transform="translate(-398.46 -172.77)"/><path class="4d796d4f-b091-45bc-941b-d60ede9debbe" d="M720.07,327.31c-26.73,0-46.46-17.61-51.55-23.34l19.72-16.76c6.79,7.64,18.46,16.12,31.83,16.12,14.63,0,23.33-8.48,23.33-19.3,0-27.58-69.79-22.49-69.79-68.95,0-24.82,19.73-42,48.79-42,25.46,0,43.49,17.39,43.49,17.39l-19.31,17s-10.18-10-24.18-10c-13.58,0-20.79,6.58-20.79,16.76,0,26.1,68.95,21.43,68.95,69.16C770.56,313.09,745.1,327.31,720.07,327.31Z" transform="translate(-398.46 -172.77)"/><path class="4d796d4f-b091-45bc-941b-d60ede9debbe" d="M887,324.76l-11.24-32.67h-54.1l-11,32.67H782.86l52.4-148.5H862l51.34,148.5ZM851.38,216.57c-1.27-3.82-2.54-10.19-2.54-10.19s-1.49,6.37-2.76,10.19l-16.55,52h38.4Z" transform="translate(-398.46 -172.77)"/><path class="4d796d4f-b091-45bc-941b-d60ede9debbe" d="M1097.09,327.31c-37.53,0-56.51-24.16-56.51-58.89,0-33.87,18.55-71.62,45.29-93.4H1129c-25.46,10.57-47.46,33.86-56.52,59.1,5-3.67,16-5.82,23.3-5.82,30.63,0,51.55,20.06,51.55,50C1147.35,304.66,1125.56,327.31,1097.09,327.31Zm-2.59-75.5c-10.35,0-18.33,2.16-26.74,7.76a63.18,63.18,0,0,0-1.08,11c0,17,8,32.57,27.82,32.57,15.32,0,26.75-9.71,26.75-25C1121.25,263.24,1110,251.81,1094.5,251.81Z" transform="translate(-398.46 -172.77)"/><path class="fd894280-7ab4-4a26-84e7-c559953ee0b7" d="M988.11,299.69c28.67-17.8,23.09-52.71,16-66.41-4-7.87-8.48-17.56-6.63-27.11.62-3.21,1.4-6.53.7-9.59-.83-3.63-3.55-6-5.34-9-1.46-2.49-3.37-6.57-1.73-9.67,0,0,1.09-2.07.81-2.66-.78-1.65-5.1-3-6.88-2.24-1.48.61-3,4.15-4.39,4.94-2.79,1.54-9.17-1-12.16,0-2.08.64-4.84,4.78-6.72,6-3.72,2.37-14,.87-16.54,4.54-1.73,2.52.33,9.4,2.35,11.29s8.89,2,10.88,4.06c12,12.71-4.47,31.29-16.91,33.2-5.48.84-15.16-7.4-20.23-4.76-2.94,1.52-7,10.64-5.24,11.22,14,4.65,25.84,17.06,29,14.49,17.1-13.81,35.37,1.69,9.2,29.56-3.67,3.91,4.93,18.81,1.21,22.67-1.39,1.44-5.78-1-7.48,0-2.43,1.4-5.11,8.2-4,10.57,3.4,7.08,25.83,6.09,31.83.09C980.23,316.46,976.31,307,988.11,299.69Z" transform="translate(-398.46 -172.77)"/><path class="fd894280-7ab4-4a26-84e7-c559953ee0b7" d="M935.78,225.69c6.67,2.5,6.64.79,10.61,4.53,5.05-2.63,11.52-10.62,11.16-17.83-24-2.38-26.11-20.9-32.2-20.28-4.53.46-6.45,16.13-4.94,16.49C931.44,211.29,930.14,223.59,935.78,225.69Z" transform="translate(-398.46 -172.77)"/><path class="fd894280-7ab4-4a26-84e7-c559953ee0b7" d="M1010.57,277.74c-3.6,12.8-10.18,18.92-17.77,23.9,0,0,8.62,2.12,9.44,4.13.47,1.12-1.53,3.92-2.68,4.48-2.61,1.28-8.61-1.49-11.14,0s-5.11,8.2-4,10.57c3.23,6.68,24.2,5.51,30.07.08,1.23-1.13.66-2.57,1.88-6.34,2.44-7.61,6.47-8.3,7.13-10.23a0,0,0,0,0,0,0C1024.61,297.3,1010.57,277.74,1010.57,277.74Z" transform="translate(-398.46 -172.77)"/></svg>
			</a>
			<hr class="eci-cta-spacing">
			<p class="eci-cta-note">Plugin created by <a href="https://michaelbourne.ca" target="_blank">Michael Bourne</a> of URSA6 and is provided free to the Elementor community.</p>
			<hr class="eci-cta-spacing">
			<p class="eci-cta-note">We recommend <a href="http://bit.ly/cloudwayseci" tagret="_blank">CloudWays</a> hosting to our WordPress clients. Use the coupon code <strong>XPROS</strong> during sign-up for a $30 credit!</p><br />
			<a href="http://bit.ly/cloudwayseci" title="Cloudways Hosting">
				<img src="data:image/jpeg;base64,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" alt="Cloudways Hosting" /></a>
		</div>
	</div>



</div>
</div>

<div class="eci-extension eci-extension-installed font-item ec-icons-clone" id="">
	<div class="eci-extension-content">
		<i class="eci" style="font-size: 80px;"></i>
		<h4 class="eci-extension-title"></h4>
		<div class="eci-extension-info">
			<span class="eci-extension-info-details">
				<span class="value">0</span> icons</span>
				<div class="iconlist hide">
				</div>
			</div>
			<a class="eci-btn delete-font" data-font="" href="#"><?php esc_html_e( 'Delete', 'custom-icons-for-elementor' ); ?></a>
		</div>

		<footer class="eci-extension-footer">

			<a class="eci-extension-status-icon" href="#">
				<svg>
					<use xmlns:xlink="http://www.w3.org/1999/xlink" xlink:href=""></use>
				</svg>
			</a>

		</footer>
	</div>